//     Underscore.js 1.6.0
//     http://underscorejs.org
//     (c) 2009-2014 Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
//     Underscore may be freely distributed under the MIT license.

(function() {

  // Baseline setup
  // --------------

  // Establish the root object, `window` in the browser, or `exports` on the server.
  var root = this;

  // Save the previous value of the `_` variable.
  var previousUnderscore = root._;

  // Establish the object that gets returned to break out of a loop iteration.
  var breaker = {};

  // Save bytes in the minified (but not gzipped) version:
  var ArrayProto = Array.prototype, ObjProto = Object.prototype, FuncProto = Function.prototype;

  // Create quick reference variables for speed access to core prototypes.
  var
    push             = ArrayProto.push,
    slice            = ArrayProto.slice,
    concat           = ArrayProto.concat,
    toString         = ObjProto.toString,
    hasOwnProperty   = ObjProto.hasOwnProperty;

  // All **ECMAScript 5** native function implementations that we hope to use
  // are declared here.
  var
    nativeForEach      = ArrayProto.forEach,
    nativeMap          = ArrayProto.map,
    nativeReduce       = ArrayProto.reduce,
    nativeReduceRight  = ArrayProto.reduceRight,
    nativeFilter       = ArrayProto.filter,
    nativeEvery        = ArrayProto.every,
    nativeSome         = ArrayProto.some,
    nativeIndexOf      = ArrayProto.indexOf,
    nativeLastIndexOf  = ArrayProto.lastIndexOf,
    nativeIsArray      = Array.isArray,
    nativeKeys         = Object.keys,
    nativeBind         = FuncProto.bind;

  // Create a safe reference to the Underscore object for use below.
  var _ = function(obj) {
    if (obj instanceof _) return obj;
    if (!(this instanceof _)) return new _(obj);
    this._wrapped = obj;
  };

  // Export the Underscore object for **Node.js**, with
  // backwards-compatibility for the old `require()` API. If we're in
  // the browser, add `_` as a global object via a string identifier,
  // for Closure Compiler "advanced" mode.
  if (typeof exports !== 'undefined') {
    if (typeof module !== 'undefined' && module.exports) {
      exports = module.exports = _;
    }
    exports._ = _;
  } else {
    root._ = _;
  }

  // Current version.
  _.VERSION = '1.6.0';

  // Collection Functions
  // --------------------

  // The cornerstone, an `each` implementation, aka `forEach`.
  // Handles objects with the built-in `forEach`, arrays, and raw objects.
  // Delegates to **ECMAScript 5**'s native `forEach` if available.
  var each = _.each = _.forEach = function(obj, iterator, context) {
    if (obj == null) return obj;
    if (nativeForEach && obj.forEach === nativeForEach) {
      obj.forEach(iterator, context);
    } else if (obj.length === +obj.length) {
      for (var i = 0, length = obj.length; i < length; i++) {
        if (iterator.call(context, obj[i], i, obj) === breaker) return;
      }
    } else {
      var keys = _.keys(obj);
      for (var i = 0, length = keys.length; i < length; i++) {
        if (iterator.call(context, obj[keys[i]], keys[i], obj) === breaker) return;
      }
    }
    return obj;
  };

  // Return the results of applying the iterator to each element.
  // Delegates to **ECMAScript 5**'s native `map` if available.
  _.map = _.collect = function(obj, iterator, context) {
    var results = [];
    if (obj == null) return results;
    if (nativeMap && obj.map === nativeMap) return obj.map(iterator, context);
    each(obj, function(value, index, list) {
      results.push(iterator.call(context, value, index, list));
    });
    return results;
  };

  var reduceError = 'Reduce of empty array with no initial value';

  // **Reduce** builds up a single result from a list of values, aka `inject`,
  // or `foldl`. Delegates to **ECMAScript 5**'s native `reduce` if available.
  _.reduce = _.foldl = _.inject = function(obj, iterator, memo, context) {
    var initial = arguments.length > 2;
    if (obj == null) obj = [];
    if (nativeReduce && obj.reduce === nativeReduce) {
      if (context) iterator = _.bind(iterator, context);
      return initial ? obj.reduce(iterator, memo) : obj.reduce(iterator);
    }
    each(obj, function(value, index, list) {
      if (!initial) {
        memo = value;
        initial = true;
      } else {
        memo = iterator.call(context, memo, value, index, list);
      }
    });
    if (!initial) throw new TypeError(reduceError);
    return memo;
  };

  // The right-associative version of reduce, also known as `foldr`.
  // Delegates to **ECMAScript 5**'s native `reduceRight` if available.
  _.reduceRight = _.foldr = function(obj, iterator, memo, context) {
    var initial = arguments.length > 2;
    if (obj == null) obj = [];
    if (nativeReduceRight && obj.reduceRight === nativeReduceRight) {
      if (context) iterator = _.bind(iterator, context);
      return initial ? obj.reduceRight(iterator, memo) : obj.reduceRight(iterator);
    }
    var length = obj.length;
    if (length !== +length) {
      var keys = _.keys(obj);
      length = keys.length;
    }
    each(obj, function(value, index, list) {
      index = keys ? keys[--length] : --length;
      if (!initial) {
        memo = obj[index];
        initial = true;
      } else {
        memo = iterator.call(context, memo, obj[index], index, list);
      }
    });
    if (!initial) throw new TypeError(reduceError);
    return memo;
  };

  // Return the first value which passes a truth test. Aliased as `detect`.
  _.find = _.detect = function(obj, predicate, context) {
    var result;
    any(obj, function(value, index, list) {
      if (predicate.call(context, value, index, list)) {
        result = value;
        return true;
      }
    });
    return result;
  };

  // Return all the elements that pass a truth test.
  // Delegates to **ECMAScript 5**'s native `filter` if available.
  // Aliased as `select`.
  _.filter = _.select = function(obj, predicate, context) {
    var results = [];
    if (obj == null) return results;
    if (nativeFilter && obj.filter === nativeFilter) return obj.filter(predicate, context);
    each(obj, function(value, index, list) {
      if (predicate.call(context, value, index, list)) results.push(value);
    });
    return results;
  };

  // Return all the elements for which a truth test fails.
  _.reject = function(obj, predicate, context) {
    return _.filter(obj, function(value, index, list) {
      return !predicate.call(context, value, index, list);
    }, context);
  };

  // Determine whether all of the elements match a truth test.
  // Delegates to **ECMAScript 5**'s native `every` if available.
  // Aliased as `all`.
  _.every = _.all = function(obj, predicate, context) {
    predicate || (predicate = _.identity);
    var result = true;
    if (obj == null) return result;
    if (nativeEvery && obj.every === nativeEvery) return obj.every(predicate, context);
    each(obj, function(value, index, list) {
      if (!(result = result && predicate.call(context, value, index, list))) return breaker;
    });
    return !!result;
  };

  // Determine if at least one element in the object matches a truth test.
  // Delegates to **ECMAScript 5**'s native `some` if available.
  // Aliased as `any`.
  var any = _.some = _.any = function(obj, predicate, context) {
    predicate || (predicate = _.identity);
    var result = false;
    if (obj == null) return result;
    if (nativeSome && obj.some === nativeSome) return obj.some(predicate, context);
    each(obj, function(value, index, list) {
      if (result || (result = predicate.call(context, value, index, list))) return breaker;
    });
    return !!result;
  };

  // Determine if the array or object contains a given value (using `===`).
  // Aliased as `include`.
  _.contains = _.include = function(obj, target) {
    if (obj == null) return false;
    if (nativeIndexOf && obj.indexOf === nativeIndexOf) return obj.indexOf(target) != -1;
    return any(obj, function(value) {
      return value === target;
    });
  };

  // Invoke a method (with arguments) on every item in a collection.
  _.invoke = function(obj, method) {
    var args = slice.call(arguments, 2);
    var isFunc = _.isFunction(method);
    return _.map(obj, function(value) {
      return (isFunc ? method : value[method]).apply(value, args);
    });
  };

  // Convenience version of a common use case of `map`: fetching a property.
  _.pluck = function(obj, key) {
    return _.map(obj, _.property(key));
  };

  // Convenience version of a common use case of `filter`: selecting only objects
  // containing specific `key:value` pairs.
  _.where = function(obj, attrs) {
    return _.filter(obj, _.matches(attrs));
  };

  // Convenience version of a common use case of `find`: getting the first object
  // containing specific `key:value` pairs.
  _.findWhere = function(obj, attrs) {
    return _.find(obj, _.matches(attrs));
  };

  // Return the maximum element or (element-based computation).
  // Can't optimize arrays of integers longer than 65,535 elements.
  // See [WebKit Bug 80797](https://bugs.webkit.org/show_bug.cgi?id=80797)
  _.max = function(obj, iterator, context) {
    if (!iterator && _.isArray(obj) && obj[0] === +obj[0] && obj.length < 65535) {
      return Math.max.apply(Math, obj);
    }
    var result = -Infinity, lastComputed = -Infinity;
    each(obj, function(value, index, list) {
      var computed = iterator ? iterator.call(context, value, index, list) : value;
      if (computed > lastComputed) {
        result = value;
        lastComputed = computed;
      }
    });
    return result;
  };

  // Return the minimum element (or element-based computation).
  _.min = function(obj, iterator, context) {
    if (!iterator && _.isArray(obj) && obj[0] === +obj[0] && obj.length < 65535) {
      return Math.min.apply(Math, obj);
    }
    var result = Infinity, lastComputed = Infinity;
    each(obj, function(value, index, list) {
      var computed = iterator ? iterator.call(context, value, index, list) : value;
      if (computed < lastComputed) {
        result = value;
        lastComputed = computed;
      }
    });
    return result;
  };

  // Shuffle an array, using the modern version of the
  // [Fisher-Yates shuffle](http://en.wikipedia.org/wiki/Fisher–Yates_shuffle).
  _.shuffle = function(obj) {
    var rand;
    var index = 0;
    var shuffled = [];
    each(obj, function(value) {
      rand = _.random(index++);
      shuffled[index - 1] = shuffled[rand];
      shuffled[rand] = value;
    });
    return shuffled;
  };

  // Sample **n** random values from a collection.
  // If **n** is not specified, returns a single random element.
  // The internal `guard` argument allows it to work with `map`.
  _.sample = function(obj, n, guard) {
    if (n == null || guard) {
      if (obj.length !== +obj.length) obj = _.values(obj);
      return obj[_.random(obj.length - 1)];
    }
    return _.shuffle(obj).slice(0, Math.max(0, n));
  };

  // An internal function to generate lookup iterators.
  var lookupIterator = function(value) {
    if (value == null) return _.identity;
    if (_.isFunction(value)) return value;
    return _.property(value);
  };

  // Sort the object's values by a criterion produced by an iterator.
  _.sortBy = function(obj, iterator, context) {
    iterator = lookupIterator(iterator);
    return _.pluck(_.map(obj, function(value, index, list) {
      return {
        value: value,
        index: index,
        criteria: iterator.call(context, value, index, list)
      };
    }).sort(function(left, right) {
      var a = left.criteria;
      var b = right.criteria;
      if (a !== b) {
        if (a > b || a === void 0) return 1;
        if (a < b || b === void 0) return -1;
      }
      return left.index - right.index;
    }), 'value');
  };

  // An internal function used for aggregate "group by" operations.
  var group = function(behavior) {
    return function(obj, iterator, context) {
      var result = {};
      iterator = lookupIterator(iterator);
      each(obj, function(value, index) {
        var key = iterator.call(context, value, index, obj);
        behavior(result, key, value);
      });
      return result;
    };
  };

  // Groups the object's values by a criterion. Pass either a string attribute
  // to group by, or a function that returns the criterion.
  _.groupBy = group(function(result, key, value) {
    _.has(result, key) ? result[key].push(value) : result[key] = [value];
  });

  // Indexes the object's values by a criterion, similar to `groupBy`, but for
  // when you know that your index values will be unique.
  _.indexBy = group(function(result, key, value) {
    result[key] = value;
  });

  // Counts instances of an object that group by a certain criterion. Pass
  // either a string attribute to count by, or a function that returns the
  // criterion.
  _.countBy = group(function(result, key) {
    _.has(result, key) ? result[key]++ : result[key] = 1;
  });

  // Use a comparator function to figure out the smallest index at which
  // an object should be inserted so as to maintain order. Uses binary search.
  _.sortedIndex = function(array, obj, iterator, context) {
    iterator = lookupIterator(iterator);
    var value = iterator.call(context, obj);
    var low = 0, high = array.length;
    while (low < high) {
      var mid = (low + high) >>> 1;
      iterator.call(context, array[mid]) < value ? low = mid + 1 : high = mid;
    }
    return low;
  };

  // Safely create a real, live array from anything iterable.
  _.toArray = function(obj) {
    if (!obj) return [];
    if (_.isArray(obj)) return slice.call(obj);
    if (obj.length === +obj.length) return _.map(obj, _.identity);
    return _.values(obj);
  };

  // Return the number of elements in an object.
  _.size = function(obj) {
    if (obj == null) return 0;
    return (obj.length === +obj.length) ? obj.length : _.keys(obj).length;
  };

  // Array Functions
  // ---------------

  // Get the first element of an array. Passing **n** will return the first N
  // values in the array. Aliased as `head` and `take`. The **guard** check
  // allows it to work with `_.map`.
  _.first = _.head = _.take = function(array, n, guard) {
    if (array == null) return void 0;
    if ((n == null) || guard) return array[0];
    if (n < 0) return [];
    return slice.call(array, 0, n);
  };

  // Returns everything but the last entry of the array. Especially useful on
  // the arguments object. Passing **n** will return all the values in
  // the array, excluding the last N. The **guard** check allows it to work with
  // `_.map`.
  _.initial = function(array, n, guard) {
    return slice.call(array, 0, array.length - ((n == null) || guard ? 1 : n));
  };

  // Get the last element of an array. Passing **n** will return the last N
  // values in the array. The **guard** check allows it to work with `_.map`.
  _.last = function(array, n, guard) {
    if (array == null) return void 0;
    if ((n == null) || guard) return array[array.length - 1];
    return slice.call(array, Math.max(array.length - n, 0));
  };

  // Returns everything but the first entry of the array. Aliased as `tail` and `drop`.
  // Especially useful on the arguments object. Passing an **n** will return
  // the rest N values in the array. The **guard**
  // check allows it to work with `_.map`.
  _.rest = _.tail = _.drop = function(array, n, guard) {
    return slice.call(array, (n == null) || guard ? 1 : n);
  };

  // Trim out all falsy values from an array.
  _.compact = function(array) {
    return _.filter(array, _.identity);
  };

  // Internal implementation of a recursive `flatten` function.
  var flatten = function(input, shallow, output) {
    if (shallow && _.every(input, _.isArray)) {
      return concat.apply(output, input);
    }
    each(input, function(value) {
      if (_.isArray(value) || _.isArguments(value)) {
        shallow ? push.apply(output, value) : flatten(value, shallow, output);
      } else {
        output.push(value);
      }
    });
    return output;
  };

  // Flatten out an array, either recursively (by default), or just one level.
  _.flatten = function(array, shallow) {
    return flatten(array, shallow, []);
  };

  // Return a version of the array that does not contain the specified value(s).
  _.without = function(array) {
    return _.difference(array, slice.call(arguments, 1));
  };

  // Split an array into two arrays: one whose elements all satisfy the given
  // predicate, and one whose elements all do not satisfy the predicate.
  _.partition = function(array, predicate) {
    var pass = [], fail = [];
    each(array, function(elem) {
      (predicate(elem) ? pass : fail).push(elem);
    });
    return [pass, fail];
  };

  // Produce a duplicate-free version of the array. If the array has already
  // been sorted, you have the option of using a faster algorithm.
  // Aliased as `unique`.
  _.uniq = _.unique = function(array, isSorted, iterator, context) {
    if (_.isFunction(isSorted)) {
      context = iterator;
      iterator = isSorted;
      isSorted = false;
    }
    var initial = iterator ? _.map(array, iterator, context) : array;
    var results = [];
    var seen = [];
    each(initial, function(value, index) {
      if (isSorted ? (!index || seen[seen.length - 1] !== value) : !_.contains(seen, value)) {
        seen.push(value);
        results.push(array[index]);
      }
    });
    return results;
  };

  // Produce an array that contains the union: each distinct element from all of
  // the passed-in arrays.
  _.union = function() {
    return _.uniq(_.flatten(arguments, true));
  };

  // Produce an array that contains every item shared between all the
  // passed-in arrays.
  _.intersection = function(array) {
    var rest = slice.call(arguments, 1);
    return _.filter(_.uniq(array), function(item) {
      return _.every(rest, function(other) {
        return _.contains(other, item);
      });
    });
  };

  // Take the difference between one array and a number of other arrays.
  // Only the elements present in just the first array will remain.
  _.difference = function(array) {
    var rest = concat.apply(ArrayProto, slice.call(arguments, 1));
    return _.filter(array, function(value){ return !_.contains(rest, value); });
  };

  // Zip together multiple lists into a single array -- elements that share
  // an index go together.
  _.zip = function() {
    var length = _.max(_.pluck(arguments, 'length').concat(0));
    var results = new Array(length);
    for (var i = 0; i < length; i++) {
      results[i] = _.pluck(arguments, '' + i);
    }
    return results;
  };

  // Converts lists into objects. Pass either a single array of `[key, value]`
  // pairs, or two parallel arrays of the same length -- one of keys, and one of
  // the corresponding values.
  _.object = function(list, values) {
    if (list == null) return {};
    var result = {};
    for (var i = 0, length = list.length; i < length; i++) {
      if (values) {
        result[list[i]] = values[i];
      } else {
        result[list[i][0]] = list[i][1];
      }
    }
    return result;
  };

  // If the browser doesn't supply us with indexOf (I'm looking at you, **MSIE**),
  // we need this function. Return the position of the first occurrence of an
  // item in an array, or -1 if the item is not included in the array.
  // Delegates to **ECMAScript 5**'s native `indexOf` if available.
  // If the array is large and already in sort order, pass `true`
  // for **isSorted** to use binary search.
  _.indexOf = function(array, item, isSorted) {
    if (array == null) return -1;
    var i = 0, length = array.length;
    if (isSorted) {
      if (typeof isSorted == 'number') {
        i = (isSorted < 0 ? Math.max(0, length + isSorted) : isSorted);
      } else {
        i = _.sortedIndex(array, item);
        return array[i] === item ? i : -1;
      }
    }
    if (nativeIndexOf && array.indexOf === nativeIndexOf) return array.indexOf(item, isSorted);
    for (; i < length; i++) if (array[i] === item) return i;
    return -1;
  };

  // Delegates to **ECMAScript 5**'s native `lastIndexOf` if available.
  _.lastIndexOf = function(array, item, from) {
    if (array == null) return -1;
    var hasIndex = from != null;
    if (nativeLastIndexOf && array.lastIndexOf === nativeLastIndexOf) {
      return hasIndex ? array.lastIndexOf(item, from) : array.lastIndexOf(item);
    }
    var i = (hasIndex ? from : array.length);
    while (i--) if (array[i] === item) return i;
    return -1;
  };

  // Generate an integer Array containing an arithmetic progression. A port of
  // the native Python `range()` function. See
  // [the Python documentation](http://docs.python.org/library/functions.html#range).
  _.range = function(start, stop, step) {
    if (arguments.length <= 1) {
      stop = start || 0;
      start = 0;
    }
    step = arguments[2] || 1;

    var length = Math.max(Math.ceil((stop - start) / step), 0);
    var idx = 0;
    var range = new Array(length);

    while(idx < length) {
      range[idx++] = start;
      start += step;
    }

    return range;
  };

  // Function (ahem) Functions
  // ------------------

  // Reusable constructor function for prototype setting.
  var ctor = function(){};

  // Create a function bound to a given object (assigning `this`, and arguments,
  // optionally). Delegates to **ECMAScript 5**'s native `Function.bind` if
  // available.
  _.bind = function(func, context) {
    var args, bound;
    if (nativeBind && func.bind === nativeBind) return nativeBind.apply(func, slice.call(arguments, 1));
    if (!_.isFunction(func)) throw new TypeError;
    args = slice.call(arguments, 2);
    return bound = function() {
      if (!(this instanceof bound)) return func.apply(context, args.concat(slice.call(arguments)));
      ctor.prototype = func.prototype;
      var self = new ctor;
      ctor.prototype = null;
      var result = func.apply(self, args.concat(slice.call(arguments)));
      if (Object(result) === result) return result;
      return self;
    };
  };

  // Partially apply a function by creating a version that has had some of its
  // arguments pre-filled, without changing its dynamic `this` context. _ acts
  // as a placeholder, allowing any combination of arguments to be pre-filled.
  _.partial = function(func) {
    var boundArgs = slice.call(arguments, 1);
    return function() {
      var position = 0;
      var args = boundArgs.slice();
      for (var i = 0, length = args.length; i < length; i++) {
        if (args[i] === _) args[i] = arguments[position++];
      }
      while (position < arguments.length) args.push(arguments[position++]);
      return func.apply(this, args);
    };
  };

  // Bind a number of an object's methods to that object. Remaining arguments
  // are the method names to be bound. Useful for ensuring that all callbacks
  // defined on an object belong to it.
  _.bindAll = function(obj) {
    var funcs = slice.call(arguments, 1);
    if (funcs.length === 0) throw new Error('bindAll must be passed function names');
    each(funcs, function(f) { obj[f] = _.bind(obj[f], obj); });
    return obj;
  };

  // Memoize an expensive function by storing its results.
  _.memoize = function(func, hasher) {
    var memo = {};
    hasher || (hasher = _.identity);
    return function() {
      var key = hasher.apply(this, arguments);
      return _.has(memo, key) ? memo[key] : (memo[key] = func.apply(this, arguments));
    };
  };

  // Delays a function for the given number of milliseconds, and then calls
  // it with the arguments supplied.
  _.delay = function(func, wait) {
    var args = slice.call(arguments, 2);
    return setTimeout(function(){ return func.apply(null, args); }, wait);
  };

  // Defers a function, scheduling it to run after the current call stack has
  // cleared.
  _.defer = function(func) {
    return _.delay.apply(_, [func, 1].concat(slice.call(arguments, 1)));
  };

  // Returns a function, that, when invoked, will only be triggered at most once
  // during a given window of time. Normally, the throttled function will run
  // as much as it can, without ever going more than once per `wait` duration;
  // but if you'd like to disable the execution on the leading edge, pass
  // `{leading: false}`. To disable execution on the trailing edge, ditto.
  _.throttle = function(func, wait, options) {
    var context, args, result;
    var timeout = null;
    var previous = 0;
    options || (options = {});
    var later = function() {
      previous = options.leading === false ? 0 : _.now();
      timeout = null;
      result = func.apply(context, args);
      context = args = null;
    };
    return function() {
      var now = _.now();
      if (!previous && options.leading === false) previous = now;
      var remaining = wait - (now - previous);
      context = this;
      args = arguments;
      if (remaining <= 0) {
        clearTimeout(timeout);
        timeout = null;
        previous = now;
        result = func.apply(context, args);
        context = args = null;
      } else if (!timeout && options.trailing !== false) {
        timeout = setTimeout(later, remaining);
      }
      return result;
    };
  };

  // Returns a function, that, as long as it continues to be invoked, will not
  // be triggered. The function will be called after it stops being called for
  // N milliseconds. If `immediate` is passed, trigger the function on the
  // leading edge, instead of the trailing.
  _.debounce = function(func, wait, immediate) {
    var timeout, args, context, timestamp, result;

    var later = function() {
      var last = _.now() - timestamp;
      if (last < wait) {
        timeout = setTimeout(later, wait - last);
      } else {
        timeout = null;
        if (!immediate) {
          result = func.apply(context, args);
          context = args = null;
        }
      }
    };

    return function() {
      context = this;
      args = arguments;
      timestamp = _.now();
      var callNow = immediate && !timeout;
      if (!timeout) {
        timeout = setTimeout(later, wait);
      }
      if (callNow) {
        result = func.apply(context, args);
        context = args = null;
      }

      return result;
    };
  };

  // Returns a function that will be executed at most one time, no matter how
  // often you call it. Useful for lazy initialization.
  _.once = function(func) {
    var ran = false, memo;
    return function() {
      if (ran) return memo;
      ran = true;
      memo = func.apply(this, arguments);
      func = null;
      return memo;
    };
  };

  // Returns the first function passed as an argument to the second,
  // allowing you to adjust arguments, run code before and after, and
  // conditionally execute the original function.
  _.wrap = function(func, wrapper) {
    return _.partial(wrapper, func);
  };

  // Returns a function that is the composition of a list of functions, each
  // consuming the return value of the function that follows.
  _.compose = function() {
    var funcs = arguments;
    return function() {
      var args = arguments;
      for (var i = funcs.length - 1; i >= 0; i--) {
        args = [funcs[i].apply(this, args)];
      }
      return args[0];
    };
  };

  // Returns a function that will only be executed after being called N times.
  _.after = function(times, func) {
    return function() {
      if (--times < 1) {
        return func.apply(this, arguments);
      }
    };
  };

  // Object Functions
  // ----------------

  // Retrieve the names of an object's properties.
  // Delegates to **ECMAScript 5**'s native `Object.keys`
  _.keys = function(obj) {
    if (!_.isObject(obj)) return [];
    if (nativeKeys) return nativeKeys(obj);
    var keys = [];
    for (var key in obj) if (_.has(obj, key)) keys.push(key);
    return keys;
  };

  // Retrieve the values of an object's properties.
  _.values = function(obj) {
    var keys = _.keys(obj);
    var length = keys.length;
    var values = new Array(length);
    for (var i = 0; i < length; i++) {
      values[i] = obj[keys[i]];
    }
    return values;
  };

  // Convert an object into a list of `[key, value]` pairs.
  _.pairs = function(obj) {
    var keys = _.keys(obj);
    var length = keys.length;
    var pairs = new Array(length);
    for (var i = 0; i < length; i++) {
      pairs[i] = [keys[i], obj[keys[i]]];
    }
    return pairs;
  };

  // Invert the keys and values of an object. The values must be serializable.
  _.invert = function(obj) {
    var result = {};
    var keys = _.keys(obj);
    for (var i = 0, length = keys.length; i < length; i++) {
      result[obj[keys[i]]] = keys[i];
    }
    return result;
  };

  // Return a sorted list of the function names available on the object.
  // Aliased as `methods`
  _.functions = _.methods = function(obj) {
    var names = [];
    for (var key in obj) {
      if (_.isFunction(obj[key])) names.push(key);
    }
    return names.sort();
  };

  // Extend a given object with all the properties in passed-in object(s).
  _.extend = function(obj) {
    each(slice.call(arguments, 1), function(source) {
      if (source) {
        for (var prop in source) {
          obj[prop] = source[prop];
        }
      }
    });
    return obj;
  };

  // Return a copy of the object only containing the whitelisted properties.
  _.pick = function(obj) {
    var copy = {};
    var keys = concat.apply(ArrayProto, slice.call(arguments, 1));
    each(keys, function(key) {
      if (key in obj) copy[key] = obj[key];
    });
    return copy;
  };

   // Return a copy of the object without the blacklisted properties.
  _.omit = function(obj) {
    var copy = {};
    var keys = concat.apply(ArrayProto, slice.call(arguments, 1));
    for (var key in obj) {
      if (!_.contains(keys, key)) copy[key] = obj[key];
    }
    return copy;
  };

  // Fill in a given object with default properties.
  _.defaults = function(obj) {
    each(slice.call(arguments, 1), function(source) {
      if (source) {
        for (var prop in source) {
          if (obj[prop] === void 0) obj[prop] = source[prop];
        }
      }
    });
    return obj;
  };

  // Create a (shallow-cloned) duplicate of an object.
  _.clone = function(obj) {
    if (!_.isObject(obj)) return obj;
    return _.isArray(obj) ? obj.slice() : _.extend({}, obj);
  };

  // Invokes interceptor with the obj, and then returns obj.
  // The primary purpose of this method is to "tap into" a method chain, in
  // order to perform operations on intermediate results within the chain.
  _.tap = function(obj, interceptor) {
    interceptor(obj);
    return obj;
  };

  // Internal recursive comparison function for `isEqual`.
  var eq = function(a, b, aStack, bStack) {
    // Identical objects are equal. `0 === -0`, but they aren't identical.
    // See the [Harmony `egal` proposal](http://wiki.ecmascript.org/doku.php?id=harmony:egal).
    if (a === b) return a !== 0 || 1 / a == 1 / b;
    // A strict comparison is necessary because `null == undefined`.
    if (a == null || b == null) return a === b;
    // Unwrap any wrapped objects.
    if (a instanceof _) a = a._wrapped;
    if (b instanceof _) b = b._wrapped;
    // Compare `[[Class]]` names.
    var className = toString.call(a);
    if (className != toString.call(b)) return false;
    switch (className) {
      // Strings, numbers, dates, and booleans are compared by value.
      case '[object String]':
        // Primitives and their corresponding object wrappers are equivalent; thus, `"5"` is
        // equivalent to `new String("5")`.
        return a == String(b);
      case '[object Number]':
        // `NaN`s are equivalent, but non-reflexive. An `egal` comparison is performed for
        // other numeric values.
        return a != +a ? b != +b : (a == 0 ? 1 / a == 1 / b : a == +b);
      case '[object Date]':
      case '[object Boolean]':
        // Coerce dates and booleans to numeric primitive values. Dates are compared by their
        // millisecond representations. Note that invalid dates with millisecond representations
        // of `NaN` are not equivalent.
        return +a == +b;
      // RegExps are compared by their source patterns and flags.
      case '[object RegExp]':
        return a.source == b.source &&
               a.global == b.global &&
               a.multiline == b.multiline &&
               a.ignoreCase == b.ignoreCase;
    }
    if (typeof a != 'object' || typeof b != 'object') return false;
    // Assume equality for cyclic structures. The algorithm for detecting cyclic
    // structures is adapted from ES 5.1 section 15.12.3, abstract operation `JO`.
    var length = aStack.length;
    while (length--) {
      // Linear search. Performance is inversely proportional to the number of
      // unique nested structures.
      if (aStack[length] == a) return bStack[length] == b;
    }
    // Objects with different constructors are not equivalent, but `Object`s
    // from different frames are.
    var aCtor = a.constructor, bCtor = b.constructor;
    if (aCtor !== bCtor && !(_.isFunction(aCtor) && (aCtor instanceof aCtor) &&
                             _.isFunction(bCtor) && (bCtor instanceof bCtor))
                        && ('constructor' in a && 'constructor' in b)) {
      return false;
    }
    // Add the first object to the stack of traversed objects.
    aStack.push(a);
    bStack.push(b);
    var size = 0, result = true;
    // Recursively compare objects and arrays.
    if (className == '[object Array]') {
      // Compare array lengths to determine if a deep comparison is necessary.
      size = a.length;
      result = size == b.length;
      if (result) {
        // Deep compare the contents, ignoring non-numeric properties.
        while (size--) {
          if (!(result = eq(a[size], b[size], aStack, bStack))) break;
        }
      }
    } else {
      // Deep compare objects.
      for (var key in a) {
        if (_.has(a, key)) {
          // Count the expected number of properties.
          size++;
          // Deep compare each member.
          if (!(result = _.has(b, key) && eq(a[key], b[key], aStack, bStack))) break;
        }
      }
      // Ensure that both objects contain the same number of properties.
      if (result) {
        for (key in b) {
          if (_.has(b, key) && !(size--)) break;
        }
        result = !size;
      }
    }
    // Remove the first object from the stack of traversed objects.
    aStack.pop();
    bStack.pop();
    return result;
  };

  // Perform a deep comparison to check if two objects are equal.
  _.isEqual = function(a, b) {
    return eq(a, b, [], []);
  };

  // Is a given array, string, or object empty?
  // An "empty" object has no enumerable own-properties.
  _.isEmpty = function(obj) {
    if (obj == null) return true;
    if (_.isArray(obj) || _.isString(obj)) return obj.length === 0;
    for (var key in obj) if (_.has(obj, key)) return false;
    return true;
  };

  // Is a given value a DOM element?
  _.isElement = function(obj) {
    return !!(obj && obj.nodeType === 1);
  };

  // Is a given value an array?
  // Delegates to ECMA5's native Array.isArray
  _.isArray = nativeIsArray || function(obj) {
    return toString.call(obj) == '[object Array]';
  };

  // Is a given variable an object?
  _.isObject = function(obj) {
    return obj === Object(obj);
  };

  // Add some isType methods: isArguments, isFunction, isString, isNumber, isDate, isRegExp.
  each(['Arguments', 'Function', 'String', 'Number', 'Date', 'RegExp'], function(name) {
    _['is' + name] = function(obj) {
      return toString.call(obj) == '[object ' + name + ']';
    };
  });

  // Define a fallback version of the method in browsers (ahem, IE), where
  // there isn't any inspectable "Arguments" type.
  if (!_.isArguments(arguments)) {
    _.isArguments = function(obj) {
      return !!(obj && _.has(obj, 'callee'));
    };
  }

  // Optimize `isFunction` if appropriate.
  if (typeof (/./) !== 'function') {
    _.isFunction = function(obj) {
      return typeof obj === 'function';
    };
  }

  // Is a given object a finite number?
  _.isFinite = function(obj) {
    return isFinite(obj) && !isNaN(parseFloat(obj));
  };

  // Is the given value `NaN`? (NaN is the only number which does not equal itself).
  _.isNaN = function(obj) {
    return _.isNumber(obj) && obj != +obj;
  };

  // Is a given value a boolean?
  _.isBoolean = function(obj) {
    return obj === true || obj === false || toString.call(obj) == '[object Boolean]';
  };

  // Is a given value equal to null?
  _.isNull = function(obj) {
    return obj === null;
  };

  // Is a given variable undefined?
  _.isUndefined = function(obj) {
    return obj === void 0;
  };

  // Shortcut function for checking if an object has a given property directly
  // on itself (in other words, not on a prototype).
  _.has = function(obj, key) {
    return hasOwnProperty.call(obj, key);
  };

  // Utility Functions
  // -----------------

  // Run Underscore.js in *noConflict* mode, returning the `_` variable to its
  // previous owner. Returns a reference to the Underscore object.
  _.noConflict = function() {
    root._ = previousUnderscore;
    return this;
  };

  // Keep the identity function around for default iterators.
  _.identity = function(value) {
    return value;
  };

  _.constant = function(value) {
    return function () {
      return value;
    };
  };

  _.property = function(key) {
    return function(obj) {
      return obj[key];
    };
  };

  // Returns a predicate for checking whether an object has a given set of `key:value` pairs.
  _.matches = function(attrs) {
    return function(obj) {
      if (obj === attrs) return true; //avoid comparing an object to itself.
      for (var key in attrs) {
        if (attrs[key] !== obj[key])
          return false;
      }
      return true;
    }
  };

  // Run a function **n** times.
  _.times = function(n, iterator, context) {
    var accum = Array(Math.max(0, n));
    for (var i = 0; i < n; i++) accum[i] = iterator.call(context, i);
    return accum;
  };

  // Return a random integer between min and max (inclusive).
  _.random = function(min, max) {
    if (max == null) {
      max = min;
      min = 0;
    }
    return min + Math.floor(Math.random() * (max - min + 1));
  };

  // A (possibly faster) way to get the current timestamp as an integer.
  _.now = Date.now || function() { return new Date().getTime(); };

  // List of HTML entities for escaping.
  var entityMap = {
    escape: {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#x27;'
    }
  };
  entityMap.unescape = _.invert(entityMap.escape);

  // Regexes containing the keys and values listed immediately above.
  var entityRegexes = {
    escape:   new RegExp('[' + _.keys(entityMap.escape).join('') + ']', 'g'),
    unescape: new RegExp('(' + _.keys(entityMap.unescape).join('|') + ')', 'g')
  };

  // Functions for escaping and unescaping strings to/from HTML interpolation.
  _.each(['escape', 'unescape'], function(method) {
    _[method] = function(string) {
      if (string == null) return '';
      return ('' + string).replace(entityRegexes[method], function(match) {
        return entityMap[method][match];
      });
    };
  });

  // If the value of the named `property` is a function then invoke it with the
  // `object` as context; otherwise, return it.
  _.result = function(object, property) {
    if (object == null) return void 0;
    var value = object[property];
    return _.isFunction(value) ? value.call(object) : value;
  };

  // Add your own custom functions to the Underscore object.
  _.mixin = function(obj) {
    each(_.functions(obj), function(name) {
      var func = _[name] = obj[name];
      _.prototype[name] = function() {
        var args = [this._wrapped];
        push.apply(args, arguments);
        return result.call(this, func.apply(_, args));
      };
    });
  };

  // Generate a unique integer id (unique within the entire client session).
  // Useful for temporary DOM ids.
  var idCounter = 0;
  _.uniqueId = function(prefix) {
    var id = ++idCounter + '';
    return prefix ? prefix + id : id;
  };

  // By default, Underscore uses ERB-style template delimiters, change the
  // following template settings to use alternative delimiters.
  _.templateSettings = {
    evaluate    : /<%([\s\S]+?)%>/g,
    interpolate : /<%=([\s\S]+?)%>/g,
    escape      : /<%-([\s\S]+?)%>/g
  };

  // When customizing `templateSettings`, if you don't want to define an
  // interpolation, evaluation or escaping regex, we need one that is
  // guaranteed not to match.
  var noMatch = /(.)^/;

  // Certain characters need to be escaped so that they can be put into a
  // string literal.
  var escapes = {
    "'":      "'",
    '\\':     '\\',
    '\r':     'r',
    '\n':     'n',
    '\t':     't',
    '\u2028': 'u2028',
    '\u2029': 'u2029'
  };

  var escaper = /\\|'|\r|\n|\t|\u2028|\u2029/g;

  // JavaScript micro-templating, similar to John Resig's implementation.
  // Underscore templating handles arbitrary delimiters, preserves whitespace,
  // and correctly escapes quotes within interpolated code.
  _.template = function(text, data, settings) {
    var render;
    settings = _.defaults({}, settings, _.templateSettings);

    // Combine delimiters into one regular expression via alternation.
    var matcher = new RegExp([
      (settings.escape || noMatch).source,
      (settings.interpolate || noMatch).source,
      (settings.evaluate || noMatch).source
    ].join('|') + '|$', 'g');

    // Compile the template source, escaping string literals appropriately.
    var index = 0;
    var source = "__p+='";
    text.replace(matcher, function(match, escape, interpolate, evaluate, offset) {
      source += text.slice(index, offset)
        .replace(escaper, function(match) { return '\\' + escapes[match]; });

      if (escape) {
        source += "'+\n((__t=(" + escape + "))==null?'':_.escape(__t))+\n'";
      }
      if (interpolate) {
        source += "'+\n((__t=(" + interpolate + "))==null?'':__t)+\n'";
      }
      if (evaluate) {
        source += "';\n" + evaluate + "\n__p+='";
      }
      index = offset + match.length;
      return match;
    });
    source += "';\n";

    // If a variable is not specified, place data values in local scope.
    if (!settings.variable) source = 'with(obj||{}){\n' + source + '}\n';

    source = "var __t,__p='',__j=Array.prototype.join," +
      "print=function(){__p+=__j.call(arguments,'');};\n" +
      source + "return __p;\n";

    try {
      render = new Function(settings.variable || 'obj', '_', source);
    } catch (e) {
      e.source = source;
      throw e;
    }

    if (data) return render(data, _);
    var template = function(data) {
      return render.call(this, data, _);
    };

    // Provide the compiled function source as a convenience for precompilation.
    template.source = 'function(' + (settings.variable || 'obj') + '){\n' + source + '}';

    return template;
  };

  // Add a "chain" function, which will delegate to the wrapper.
  _.chain = function(obj) {
    return _(obj).chain();
  };

  // OOP
  // ---------------
  // If Underscore is called as a function, it returns a wrapped object that
  // can be used OO-style. This wrapper holds altered versions of all the
  // underscore functions. Wrapped objects may be chained.

  // Helper function to continue chaining intermediate results.
  var result = function(obj) {
    return this._chain ? _(obj).chain() : obj;
  };

  // Add all of the Underscore functions to the wrapper object.
  _.mixin(_);

  // Add all mutator Array functions to the wrapper.
  each(['pop', 'push', 'reverse', 'shift', 'sort', 'splice', 'unshift'], function(name) {
    var method = ArrayProto[name];
    _.prototype[name] = function() {
      var obj = this._wrapped;
      method.apply(obj, arguments);
      if ((name == 'shift' || name == 'splice') && obj.length === 0) delete obj[0];
      return result.call(this, obj);
    };
  });

  // Add all accessor Array functions to the wrapper.
  each(['concat', 'join', 'slice'], function(name) {
    var method = ArrayProto[name];
    _.prototype[name] = function() {
      return result.call(this, method.apply(this._wrapped, arguments));
    };
  });

  _.extend(_.prototype, {

    // Start chaining a wrapped Underscore object.
    chain: function() {
      this._chain = true;
      return this;
    },

    // Extracts the result from a wrapped and chained object.
    value: function() {
      return this._wrapped;
    }

  });

  // AMD registration happens at the end for compatibility with AMD loaders
  // that may not enforce next-turn semantics on modules. Even though general
  // practice for AMD registration is to be anonymous, underscore registers
  // as a named module because, like jQuery, it is a base library that is
  // popular enough to be bundled in a third party lib, but not be part of
  // an AMD load request. Those cases could generate an error when an
  // anonymous define() is called outside of a loader request.
  if (typeof define === 'function' && define.amd) {
    define('underscore', [], function() {
      return _;
    });
  }
}).call(this);

/*!
 * VERSION: 1.16.1
 * DATE: 2015-03-13
 * UPDATES AND DOCS AT: http://www.greensock.com
 *
 * @license Copyright (c) 2008-2015, GreenSock. All rights reserved.
 * This work is subject to the terms at http://greensock.com/standard-license or for
 * Club GreenSock members, the software agreement that was issued with your membership.
 * 
 * @author: Jack Doyle, jack@greensock.com
 */
var _gsScope = (typeof(module) !== "undefined" && module.exports && typeof(global) !== "undefined") ? global : this || window; //helps ensure compatibility with AMD/RequireJS and CommonJS/Node
(_gsScope._gsQueue || (_gsScope._gsQueue = [])).push( function() {

	"use strict";

	_gsScope._gsDefine("plugins.CSSPlugin", ["plugins.TweenPlugin","TweenLite"], function(TweenPlugin, TweenLite) {

		/** @constructor **/
		var CSSPlugin = function() {
				TweenPlugin.call(this, "css");
				this._overwriteProps.length = 0;
				this.setRatio = CSSPlugin.prototype.setRatio; //speed optimization (avoid prototype lookup on this "hot" method)
			},
			_globals = _gsScope._gsDefine.globals,
			_hasPriority, //turns true whenever a CSSPropTween instance is created that has a priority other than 0. This helps us discern whether or not we should spend the time organizing the linked list or not after a CSSPlugin's _onInitTween() method is called.
			_suffixMap, //we set this in _onInitTween() each time as a way to have a persistent variable we can use in other methods like _parse() without having to pass it around as a parameter and we keep _parse() decoupled from a particular CSSPlugin instance
			_cs, //computed style (we store this in a shared variable to conserve memory and make minification tighter
			_overwriteProps, //alias to the currently instantiating CSSPlugin's _overwriteProps array. We use this closure in order to avoid having to pass a reference around from method to method and aid in minification.
			_specialProps = {},
			p = CSSPlugin.prototype = new TweenPlugin("css");

		p.constructor = CSSPlugin;
		CSSPlugin.version = "1.16.1";
		CSSPlugin.API = 2;
		CSSPlugin.defaultTransformPerspective = 0;
		CSSPlugin.defaultSkewType = "compensated";
		p = "px"; //we'll reuse the "p" variable to keep file size down
		CSSPlugin.suffixMap = {top:p, right:p, bottom:p, left:p, width:p, height:p, fontSize:p, padding:p, margin:p, perspective:p, lineHeight:""};


		var _numExp = /(?:\d|\-\d|\.\d|\-\.\d)+/g,
			_relNumExp = /(?:\d|\-\d|\.\d|\-\.\d|\+=\d|\-=\d|\+=.\d|\-=\.\d)+/g,
			_valuesExp = /(?:\+=|\-=|\-|\b)[\d\-\.]+[a-zA-Z0-9]*(?:%|\b)/gi, //finds all the values that begin with numbers or += or -= and then a number. Includes suffixes. We use this to split complex values apart like "1px 5px 20px rgb(255,102,51)"
			_NaNExp = /(?![+-]?\d*\.?\d+|[+-]|e[+-]\d+)[^0-9]/g, //also allows scientific notation and doesn't kill the leading -/+ in -= and +=
			_suffixExp = /(?:\d|\-|\+|=|#|\.)*/g,
			_opacityExp = /opacity *= *([^)]*)/i,
			_opacityValExp = /opacity:([^;]*)/i,
			_alphaFilterExp = /alpha\(opacity *=.+?\)/i,
			_rgbhslExp = /^(rgb|hsl)/,
			_capsExp = /([A-Z])/g,
			_camelExp = /-([a-z])/gi,
			_urlExp = /(^(?:url\(\"|url\())|(?:(\"\))$|\)$)/gi, //for pulling out urls from url(...) or url("...") strings (some browsers wrap urls in quotes, some don't when reporting things like backgroundImage)
			_camelFunc = function(s, g) { return g.toUpperCase(); },
			_horizExp = /(?:Left|Right|Width)/i,
			_ieGetMatrixExp = /(M11|M12|M21|M22)=[\d\-\.e]+/gi,
			_ieSetMatrixExp = /progid\:DXImageTransform\.Microsoft\.Matrix\(.+?\)/i,
			_commasOutsideParenExp = /,(?=[^\)]*(?:\(|$))/gi, //finds any commas that are not within parenthesis
			_DEG2RAD = Math.PI / 180,
			_RAD2DEG = 180 / Math.PI,
			_forcePT = {},
			_doc = document,
			_createElement = function(type) {
				return _doc.createElementNS ? _doc.createElementNS("http://www.w3.org/1999/xhtml", type) : _doc.createElement(type);
			},
			_tempDiv = _createElement("div"),
			_tempImg = _createElement("img"),
			_internals = CSSPlugin._internals = {_specialProps:_specialProps}, //provides a hook to a few internal methods that we need to access from inside other plugins
			_agent = navigator.userAgent,
			_autoRound,
			_reqSafariFix, //we won't apply the Safari transform fix until we actually come across a tween that affects a transform property (to maintain best performance).

			_isSafari,
			_isFirefox, //Firefox has a bug that causes 3D transformed elements to randomly disappear unless a repaint is forced after each update on each element.
			_isSafariLT6, //Safari (and Android 4 which uses a flavor of Safari) has a bug that prevents changes to "top" and "left" properties from rendering properly if changed on the same frame as a transform UNLESS we set the element's WebkitBackfaceVisibility to hidden (weird, I know). Doing this for Android 3 and earlier seems to actually cause other problems, though (fun!)
			_ieVers,
			_supportsOpacity = (function() { //we set _isSafari, _ieVers, _isFirefox, and _supportsOpacity all in one function here to reduce file size slightly, especially in the minified version.
				var i = _agent.indexOf("Android"),
					a = _createElement("a");
				_isSafari = (_agent.indexOf("Safari") !== -1 && _agent.indexOf("Chrome") === -1 && (i === -1 || Number(_agent.substr(i+8, 1)) > 3));
				_isSafariLT6 = (_isSafari && (Number(_agent.substr(_agent.indexOf("Version/")+8, 1)) < 6));
				_isFirefox = (_agent.indexOf("Firefox") !== -1);
				if ((/MSIE ([0-9]{1,}[\.0-9]{0,})/).exec(_agent) || (/Trident\/.*rv:([0-9]{1,}[\.0-9]{0,})/).exec(_agent)) {
					_ieVers = parseFloat( RegExp.$1 );
				}
				if (!a) {
					return false;
				}
				a.style.cssText = "top:1px;opacity:.55;";
				return /^0.55/.test(a.style.opacity);
			}()),
			_getIEOpacity = function(v) {
				return (_opacityExp.test( ((typeof(v) === "string") ? v : (v.currentStyle ? v.currentStyle.filter : v.style.filter) || "") ) ? ( parseFloat( RegExp.$1 ) / 100 ) : 1);
			},
			_log = function(s) {//for logging messages, but in a way that won't throw errors in old versions of IE.
				if (window.console) {
					console.log(s);
				}
			},

			_prefixCSS = "", //the non-camelCase vendor prefix like "-o-", "-moz-", "-ms-", or "-webkit-"
			_prefix = "", //camelCase vendor prefix like "O", "ms", "Webkit", or "Moz".

			// @private feed in a camelCase property name like "transform" and it will check to see if it is valid as-is or if it needs a vendor prefix. It returns the corrected camelCase property name (i.e. "WebkitTransform" or "MozTransform" or "transform" or null if no such property is found, like if the browser is IE8 or before, "transform" won't be found at all)
			_checkPropPrefix = function(p, e) {
				e = e || _tempDiv;
				var s = e.style,
					a, i;
				if (s[p] !== undefined) {
					return p;
				}
				p = p.charAt(0).toUpperCase() + p.substr(1);
				a = ["O","Moz","ms","Ms","Webkit"];
				i = 5;
				while (--i > -1 && s[a[i]+p] === undefined) { }
				if (i >= 0) {
					_prefix = (i === 3) ? "ms" : a[i];
					_prefixCSS = "-" + _prefix.toLowerCase() + "-";
					return _prefix + p;
				}
				return null;
			},

			_getComputedStyle = _doc.defaultView ? _doc.defaultView.getComputedStyle : function() {},

			/**
			 * @private Returns the css style for a particular property of an element. For example, to get whatever the current "left" css value for an element with an ID of "myElement", you could do:
			 * var currentLeft = CSSPlugin.getStyle( document.getElementById("myElement"), "left");
			 *
			 * @param {!Object} t Target element whose style property you want to query
			 * @param {!string} p Property name (like "left" or "top" or "marginTop", etc.)
			 * @param {Object=} cs Computed style object. This just provides a way to speed processing if you're going to get several properties on the same element in quick succession - you can reuse the result of the getComputedStyle() call.
			 * @param {boolean=} calc If true, the value will not be read directly from the element's "style" property (if it exists there), but instead the getComputedStyle() result will be used. This can be useful when you want to ensure that the browser itself is interpreting the value.
			 * @param {string=} dflt Default value that should be returned in the place of null, "none", "auto" or "auto auto".
			 * @return {?string} The current property value
			 */
			_getStyle = CSSPlugin.getStyle = function(t, p, cs, calc, dflt) {
				var rv;
				if (!_supportsOpacity) if (p === "opacity") { //several versions of IE don't use the standard "opacity" property - they use things like filter:alpha(opacity=50), so we parse that here.
					return _getIEOpacity(t);
				}
				if (!calc && t.style[p]) {
					rv = t.style[p];
				} else if ((cs = cs || _getComputedStyle(t))) {
					rv = cs[p] || cs.getPropertyValue(p) || cs.getPropertyValue(p.replace(_capsExp, "-$1").toLowerCase());
				} else if (t.currentStyle) {
					rv = t.currentStyle[p];
				}
				return (dflt != null && (!rv || rv === "none" || rv === "auto" || rv === "auto auto")) ? dflt : rv;
			},

			/**
			 * @private Pass the target element, the property name, the numeric value, and the suffix (like "%", "em", "px", etc.) and it will spit back the equivalent pixel number.
			 * @param {!Object} t Target element
			 * @param {!string} p Property name (like "left", "top", "marginLeft", etc.)
			 * @param {!number} v Value
			 * @param {string=} sfx Suffix (like "px" or "%" or "em")
			 * @param {boolean=} recurse If true, the call is a recursive one. In some browsers (like IE7/8), occasionally the value isn't accurately reported initially, but if we run the function again it will take effect.
			 * @return {number} value in pixels
			 */
			_convertToPixels = _internals.convertToPixels = function(t, p, v, sfx, recurse) {
				if (sfx === "px" || !sfx) { return v; }
				if (sfx === "auto" || !v) { return 0; }
				var horiz = _horizExp.test(p),
					node = t,
					style = _tempDiv.style,
					neg = (v < 0),
					pix, cache, time;
				if (neg) {
					v = -v;
				}
				if (sfx === "%" && p.indexOf("border") !== -1) {
					pix = (v / 100) * (horiz ? t.clientWidth : t.clientHeight);
				} else {
					style.cssText = "border:0 solid red;position:" + _getStyle(t, "position") + ";line-height:0;";
					if (sfx === "%" || !node.appendChild) {
						node = t.parentNode || _doc.body;
						cache = node._gsCache;
						time = TweenLite.ticker.frame;
						if (cache && horiz && cache.time === time) { //performance optimization: we record the width of elements along with the ticker frame so that we can quickly get it again on the same tick (seems relatively safe to assume it wouldn't change on the same tick)
							return cache.width * v / 100;
						}
						style[(horiz ? "width" : "height")] = v + sfx;
					} else {
						style[(horiz ? "borderLeftWidth" : "borderTopWidth")] = v + sfx;
					}
					node.appendChild(_tempDiv);
					pix = parseFloat(_tempDiv[(horiz ? "offsetWidth" : "offsetHeight")]);
					node.removeChild(_tempDiv);
					if (horiz && sfx === "%" && CSSPlugin.cacheWidths !== false) {
						cache = node._gsCache = node._gsCache || {};
						cache.time = time;
						cache.width = pix / v * 100;
					}
					if (pix === 0 && !recurse) {
						pix = _convertToPixels(t, p, v, sfx, true);
					}
				}
				return neg ? -pix : pix;
			},
			_calculateOffset = _internals.calculateOffset = function(t, p, cs) { //for figuring out "top" or "left" in px when it's "auto". We need to factor in margin with the offsetLeft/offsetTop
				if (_getStyle(t, "position", cs) !== "absolute") { return 0; }
				var dim = ((p === "left") ? "Left" : "Top"),
					v = _getStyle(t, "margin" + dim, cs);
				return t["offset" + dim] - (_convertToPixels(t, p, parseFloat(v), v.replace(_suffixExp, "")) || 0);
			},

			// @private returns at object containing ALL of the style properties in camelCase and their associated values.
			_getAllStyles = function(t, cs) {
				var s = {},
					i, tr, p;
				if ((cs = cs || _getComputedStyle(t, null))) {
					if ((i = cs.length)) {
						while (--i > -1) {
							p = cs[i];
							if (p.indexOf("-transform") === -1 || _transformPropCSS === p) { //Some webkit browsers duplicate transform values, one non-prefixed and one prefixed ("transform" and "WebkitTransform"), so we must weed out the extra one here.
								s[p.replace(_camelExp, _camelFunc)] = cs.getPropertyValue(p);
							}
						}
					} else { //some browsers behave differently - cs.length is always 0, so we must do a for...in loop.
						for (i in cs) {
							if (i.indexOf("Transform") === -1 || _transformProp === i) { //Some webkit browsers duplicate transform values, one non-prefixed and one prefixed ("transform" and "WebkitTransform"), so we must weed out the extra one here.
								s[i] = cs[i];
							}
						}
					}
				} else if ((cs = t.currentStyle || t.style)) {
					for (i in cs) {
						if (typeof(i) === "string" && s[i] === undefined) {
							s[i.replace(_camelExp, _camelFunc)] = cs[i];
						}
					}
				}
				if (!_supportsOpacity) {
					s.opacity = _getIEOpacity(t);
				}
				tr = _getTransform(t, cs, false);
				s.rotation = tr.rotation;
				s.skewX = tr.skewX;
				s.scaleX = tr.scaleX;
				s.scaleY = tr.scaleY;
				s.x = tr.x;
				s.y = tr.y;
				if (_supports3D) {
					s.z = tr.z;
					s.rotationX = tr.rotationX;
					s.rotationY = tr.rotationY;
					s.scaleZ = tr.scaleZ;
				}
				if (s.filters) {
					delete s.filters;
				}
				return s;
			},

			// @private analyzes two style objects (as returned by _getAllStyles()) and only looks for differences between them that contain tweenable values (like a number or color). It returns an object with a "difs" property which refers to an object containing only those isolated properties and values for tweening, and a "firstMPT" property which refers to the first MiniPropTween instance in a linked list that recorded all the starting values of the different properties so that we can revert to them at the end or beginning of the tween - we don't want the cascading to get messed up. The forceLookup parameter is an optional generic object with properties that should be forced into the results - this is necessary for className tweens that are overwriting others because imagine a scenario where a rollover/rollout adds/removes a class and the user swipes the mouse over the target SUPER fast, thus nothing actually changed yet and the subsequent comparison of the properties would indicate they match (especially when px rounding is taken into consideration), thus no tweening is necessary even though it SHOULD tween and remove those properties after the tween (otherwise the inline styles will contaminate things). See the className SpecialProp code for details.
			_cssDif = function(t, s1, s2, vars, forceLookup) {
				var difs = {},
					style = t.style,
					val, p, mpt;
				for (p in s2) {
					if (p !== "cssText") if (p !== "length") if (isNaN(p)) if (s1[p] !== (val = s2[p]) || (forceLookup && forceLookup[p])) if (p.indexOf("Origin") === -1) if (typeof(val) === "number" || typeof(val) === "string") {
						difs[p] = (val === "auto" && (p === "left" || p === "top")) ? _calculateOffset(t, p) : ((val === "" || val === "auto" || val === "none") && typeof(s1[p]) === "string" && s1[p].replace(_NaNExp, "") !== "") ? 0 : val; //if the ending value is defaulting ("" or "auto"), we check the starting value and if it can be parsed into a number (a string which could have a suffix too, like 700px), then we swap in 0 for "" or "auto" so that things actually tween.
						if (style[p] !== undefined) { //for className tweens, we must remember which properties already existed inline - the ones that didn't should be removed when the tween isn't in progress because they were only introduced to facilitate the transition between classes.
							mpt = new MiniPropTween(style, p, style[p], mpt);
						}
					}
				}
				if (vars) {
					for (p in vars) { //copy properties (except className)
						if (p !== "className") {
							difs[p] = vars[p];
						}
					}
				}
				return {difs:difs, firstMPT:mpt};
			},
			_dimensions = {width:["Left","Right"], height:["Top","Bottom"]},
			_margins = ["marginLeft","marginRight","marginTop","marginBottom"],

			/**
			 * @private Gets the width or height of an element
			 * @param {!Object} t Target element
			 * @param {!string} p Property name ("width" or "height")
			 * @param {Object=} cs Computed style object (if one exists). Just a speed optimization.
			 * @return {number} Dimension (in pixels)
			 */
			_getDimension = function(t, p, cs) {
				var v = parseFloat((p === "width") ? t.offsetWidth : t.offsetHeight),
					a = _dimensions[p],
					i = a.length;
				cs = cs || _getComputedStyle(t, null);
				while (--i > -1) {
					v -= parseFloat( _getStyle(t, "padding" + a[i], cs, true) ) || 0;
					v -= parseFloat( _getStyle(t, "border" + a[i] + "Width", cs, true) ) || 0;
				}
				return v;
			},

			// @private Parses position-related complex strings like "top left" or "50px 10px" or "70% 20%", etc. which are used for things like transformOrigin or backgroundPosition. Optionally decorates a supplied object (recObj) with the following properties: "ox" (offsetX), "oy" (offsetY), "oxp" (if true, "ox" is a percentage not a pixel value), and "oxy" (if true, "oy" is a percentage not a pixel value)
			_parsePosition = function(v, recObj) {
				if (v == null || v === "" || v === "auto" || v === "auto auto") { //note: Firefox uses "auto auto" as default whereas Chrome uses "auto".
					v = "0 0";
				}
				var a = v.split(" "),
					x = (v.indexOf("left") !== -1) ? "0%" : (v.indexOf("right") !== -1) ? "100%" : a[0],
					y = (v.indexOf("top") !== -1) ? "0%" : (v.indexOf("bottom") !== -1) ? "100%" : a[1];
				if (y == null) {
					y = (x === "center") ? "50%" : "0";
				} else if (y === "center") {
					y = "50%";
				}
				if (x === "center" || (isNaN(parseFloat(x)) && (x + "").indexOf("=") === -1)) { //remember, the user could flip-flop the values and say "bottom center" or "center bottom", etc. "center" is ambiguous because it could be used to describe horizontal or vertical, hence the isNaN(). If there's an "=" sign in the value, it's relative.
					x = "50%";
				}
				v = x + " " + y + ((a.length > 2) ? " " + a[2] : "");
				if (recObj) {
					recObj.oxp = (x.indexOf("%") !== -1);
					recObj.oyp = (y.indexOf("%") !== -1);
					recObj.oxr = (x.charAt(1) === "=");
					recObj.oyr = (y.charAt(1) === "=");
					recObj.ox = parseFloat(x.replace(_NaNExp, ""));
					recObj.oy = parseFloat(y.replace(_NaNExp, ""));
					recObj.v = v;
				}
				return recObj || v;
			},

			/**
			 * @private Takes an ending value (typically a string, but can be a number) and a starting value and returns the change between the two, looking for relative value indicators like += and -= and it also ignores suffixes (but make sure the ending value starts with a number or +=/-= and that the starting value is a NUMBER!)
			 * @param {(number|string)} e End value which is typically a string, but could be a number
			 * @param {(number|string)} b Beginning value which is typically a string but could be a number
			 * @return {number} Amount of change between the beginning and ending values (relative values that have a "+=" or "-=" are recognized)
			 */
			_parseChange = function(e, b) {
				return (typeof(e) === "string" && e.charAt(1) === "=") ? parseInt(e.charAt(0) + "1", 10) * parseFloat(e.substr(2)) : parseFloat(e) - parseFloat(b);
			},

			/**
			 * @private Takes a value and a default number, checks if the value is relative, null, or numeric and spits back a normalized number accordingly. Primarily used in the _parseTransform() function.
			 * @param {Object} v Value to be parsed
			 * @param {!number} d Default value (which is also used for relative calculations if "+=" or "-=" is found in the first parameter)
			 * @return {number} Parsed value
			 */
			_parseVal = function(v, d) {
				return (v == null) ? d : (typeof(v) === "string" && v.charAt(1) === "=") ? parseInt(v.charAt(0) + "1", 10) * parseFloat(v.substr(2)) + d : parseFloat(v);
			},

			/**
			 * @private Translates strings like "40deg" or "40" or 40rad" or "+=40deg" or "270_short" or "-90_cw" or "+=45_ccw" to a numeric radian angle. Of course a starting/default value must be fed in too so that relative values can be calculated properly.
			 * @param {Object} v Value to be parsed
			 * @param {!number} d Default value (which is also used for relative calculations if "+=" or "-=" is found in the first parameter)
			 * @param {string=} p property name for directionalEnd (optional - only used when the parsed value is directional ("_short", "_cw", or "_ccw" suffix). We need a way to store the uncompensated value so that at the end of the tween, we set it to exactly what was requested with no directional compensation). Property name would be "rotation", "rotationX", or "rotationY"
			 * @param {Object=} directionalEnd An object that will store the raw end values for directional angles ("_short", "_cw", or "_ccw" suffix). We need a way to store the uncompensated value so that at the end of the tween, we set it to exactly what was requested with no directional compensation.
			 * @return {number} parsed angle in radians
			 */
			_parseAngle = function(v, d, p, directionalEnd) {
				var min = 0.000001,
					cap, split, dif, result, isRelative;
				if (v == null) {
					result = d;
				} else if (typeof(v) === "number") {
					result = v;
				} else {
					cap = 360;
					split = v.split("_");
					isRelative = (v.charAt(1) === "=");
					dif = (isRelative ? parseInt(v.charAt(0) + "1", 10) * parseFloat(split[0].substr(2)) : parseFloat(split[0])) * ((v.indexOf("rad") === -1) ? 1 : _RAD2DEG) - (isRelative ? 0 : d);
					if (split.length) {
						if (directionalEnd) {
							directionalEnd[p] = d + dif;
						}
						if (v.indexOf("short") !== -1) {
							dif = dif % cap;
							if (dif !== dif % (cap / 2)) {
								dif = (dif < 0) ? dif + cap : dif - cap;
							}
						}
						if (v.indexOf("_cw") !== -1 && dif < 0) {
							dif = ((dif + cap * 9999999999) % cap) - ((dif / cap) | 0) * cap;
						} else if (v.indexOf("ccw") !== -1 && dif > 0) {
							dif = ((dif - cap * 9999999999) % cap) - ((dif / cap) | 0) * cap;
						}
					}
					result = d + dif;
				}
				if (result < min && result > -min) {
					result = 0;
				}
				return result;
			},

			_colorLookup = {aqua:[0,255,255],
				lime:[0,255,0],
				silver:[192,192,192],
				black:[0,0,0],
				maroon:[128,0,0],
				teal:[0,128,128],
				blue:[0,0,255],
				navy:[0,0,128],
				white:[255,255,255],
				fuchsia:[255,0,255],
				olive:[128,128,0],
				yellow:[255,255,0],
				orange:[255,165,0],
				gray:[128,128,128],
				purple:[128,0,128],
				green:[0,128,0],
				red:[255,0,0],
				pink:[255,192,203],
				cyan:[0,255,255],
				transparent:[255,255,255,0]},

			_hue = function(h, m1, m2) {
				h = (h < 0) ? h + 1 : (h > 1) ? h - 1 : h;
				return ((((h * 6 < 1) ? m1 + (m2 - m1) * h * 6 : (h < 0.5) ? m2 : (h * 3 < 2) ? m1 + (m2 - m1) * (2 / 3 - h) * 6 : m1) * 255) + 0.5) | 0;
			},

			/**
			 * @private Parses a color (like #9F0, #FF9900, or rgb(255,51,153)) into an array with 3 elements for red, green, and blue. Also handles rgba() values (splits into array of 4 elements of course)
			 * @param {(string|number)} v The value the should be parsed which could be a string like #9F0 or rgb(255,102,51) or rgba(255,0,0,0.5) or it could be a number like 0xFF00CC or even a named color like red, blue, purple, etc.
			 * @return {Array.<number>} An array containing red, green, and blue (and optionally alpha) in that order.
			 */
			_parseColor = CSSPlugin.parseColor = function(v) {
				var c1, c2, c3, h, s, l;
				if (!v || v === "") {
					return _colorLookup.black;
				}
				if (typeof(v) === "number") {
					return [v >> 16, (v >> 8) & 255, v & 255];
				}
				if (v.charAt(v.length - 1) === ",") { //sometimes a trailing commma is included and we should chop it off (typically from a comma-delimited list of values like a textShadow:"2px 2px 2px blue, 5px 5px 5px rgb(255,0,0)" - in this example "blue," has a trailing comma. We could strip it out inside parseComplex() but we'd need to do it to the beginning and ending values plus it wouldn't provide protection from other potential scenarios like if the user passes in a similar value.
					v = v.substr(0, v.length - 1);
				}
				if (_colorLookup[v]) {
					return _colorLookup[v];
				}
				if (v.charAt(0) === "#") {
					if (v.length === 4) { //for shorthand like #9F0
						c1 = v.charAt(1),
						c2 = v.charAt(2),
						c3 = v.charAt(3);
						v = "#" + c1 + c1 + c2 + c2 + c3 + c3;
					}
					v = parseInt(v.substr(1), 16);
					return [v >> 16, (v >> 8) & 255, v & 255];
				}
				if (v.substr(0, 3) === "hsl") {
					v = v.match(_numExp);
					h = (Number(v[0]) % 360) / 360;
					s = Number(v[1]) / 100;
					l = Number(v[2]) / 100;
					c2 = (l <= 0.5) ? l * (s + 1) : l + s - l * s;
					c1 = l * 2 - c2;
					if (v.length > 3) {
						v[3] = Number(v[3]);
					}
					v[0] = _hue(h + 1 / 3, c1, c2);
					v[1] = _hue(h, c1, c2);
					v[2] = _hue(h - 1 / 3, c1, c2);
					return v;
				}
				v = v.match(_numExp) || _colorLookup.transparent;
				v[0] = Number(v[0]);
				v[1] = Number(v[1]);
				v[2] = Number(v[2]);
				if (v.length > 3) {
					v[3] = Number(v[3]);
				}
				return v;
			},
			_colorExp = "(?:\\b(?:(?:rgb|rgba|hsl|hsla)\\(.+?\\))|\\B#.+?\\b"; //we'll dynamically build this Regular Expression to conserve file size. After building it, it will be able to find rgb(), rgba(), # (hexadecimal), and named color values like red, blue, purple, etc.

		for (p in _colorLookup) {
			_colorExp += "|" + p + "\\b";
		}
		_colorExp = new RegExp(_colorExp+")", "gi");

		/**
		 * @private Returns a formatter function that handles taking a string (or number in some cases) and returning a consistently formatted one in terms of delimiters, quantity of values, etc. For example, we may get boxShadow values defined as "0px red" or "0px 0px 10px rgb(255,0,0)" or "0px 0px 20px 20px #F00" and we need to ensure that what we get back is described with 4 numbers and a color. This allows us to feed it into the _parseComplex() method and split the values up appropriately. The neat thing about this _getFormatter() function is that the dflt defines a pattern as well as a default, so for example, _getFormatter("0px 0px 0px 0px #777", true) not only sets the default as 0px for all distances and #777 for the color, but also sets the pattern such that 4 numbers and a color will always get returned.
		 * @param {!string} dflt The default value and pattern to follow. So "0px 0px 0px 0px #777" will ensure that 4 numbers and a color will always get returned.
		 * @param {boolean=} clr If true, the values should be searched for color-related data. For example, boxShadow values typically contain a color whereas borderRadius don't.
		 * @param {boolean=} collapsible If true, the value is a top/left/right/bottom style one that acts like margin or padding, where if only one value is received, it's used for all 4; if 2 are received, the first is duplicated for 3rd (bottom) and the 2nd is duplicated for the 4th spot (left), etc.
		 * @return {Function} formatter function
		 */
		var _getFormatter = function(dflt, clr, collapsible, multi) {
				if (dflt == null) {
					return function(v) {return v;};
				}
				var dColor = clr ? (dflt.match(_colorExp) || [""])[0] : "",
					dVals = dflt.split(dColor).join("").match(_valuesExp) || [],
					pfx = dflt.substr(0, dflt.indexOf(dVals[0])),
					sfx = (dflt.charAt(dflt.length - 1) === ")") ? ")" : "",
					delim = (dflt.indexOf(" ") !== -1) ? " " : ",",
					numVals = dVals.length,
					dSfx = (numVals > 0) ? dVals[0].replace(_numExp, "") : "",
					formatter;
				if (!numVals) {
					return function(v) {return v;};
				}
				if (clr) {
					formatter = function(v) {
						var color, vals, i, a;
						if (typeof(v) === "number") {
							v += dSfx;
						} else if (multi && _commasOutsideParenExp.test(v)) {
							a = v.replace(_commasOutsideParenExp, "|").split("|");
							for (i = 0; i < a.length; i++) {
								a[i] = formatter(a[i]);
							}
							return a.join(",");
						}
						color = (v.match(_colorExp) || [dColor])[0];
						vals = v.split(color).join("").match(_valuesExp) || [];
						i = vals.length;
						if (numVals > i--) {
							while (++i < numVals) {
								vals[i] = collapsible ? vals[(((i - 1) / 2) | 0)] : dVals[i];
							}
						}
						return pfx + vals.join(delim) + delim + color + sfx + (v.indexOf("inset") !== -1 ? " inset" : "");
					};
					return formatter;

				}
				formatter = function(v) {
					var vals, a, i;
					if (typeof(v) === "number") {
						v += dSfx;
					} else if (multi && _commasOutsideParenExp.test(v)) {
						a = v.replace(_commasOutsideParenExp, "|").split("|");
						for (i = 0; i < a.length; i++) {
							a[i] = formatter(a[i]);
						}
						return a.join(",");
					}
					vals = v.match(_valuesExp) || [];
					i = vals.length;
					if (numVals > i--) {
						while (++i < numVals) {
							vals[i] = collapsible ? vals[(((i - 1) / 2) | 0)] : dVals[i];
						}
					}
					return pfx + vals.join(delim) + sfx;
				};
				return formatter;
			},

			/**
			 * @private returns a formatter function that's used for edge-related values like marginTop, marginLeft, paddingBottom, paddingRight, etc. Just pass a comma-delimited list of property names related to the edges.
			 * @param {!string} props a comma-delimited list of property names in order from top to left, like "marginTop,marginRight,marginBottom,marginLeft"
			 * @return {Function} a formatter function
			 */
			_getEdgeParser = function(props) {
				props = props.split(",");
				return function(t, e, p, cssp, pt, plugin, vars) {
					var a = (e + "").split(" "),
						i;
					vars = {};
					for (i = 0; i < 4; i++) {
						vars[props[i]] = a[i] = a[i] || a[(((i - 1) / 2) >> 0)];
					}
					return cssp.parse(t, vars, pt, plugin);
				};
			},

			// @private used when other plugins must tween values first, like BezierPlugin or ThrowPropsPlugin, etc. That plugin's setRatio() gets called first so that the values are updated, and then we loop through the MiniPropTweens  which handle copying the values into their appropriate slots so that they can then be applied correctly in the main CSSPlugin setRatio() method. Remember, we typically create a proxy object that has a bunch of uniquely-named properties that we feed to the sub-plugin and it does its magic normally, and then we must interpret those values and apply them to the css because often numbers must get combined/concatenated, suffixes added, etc. to work with css, like boxShadow could have 4 values plus a color.
			_setPluginRatio = _internals._setPluginRatio = function(v) {
				this.plugin.setRatio(v);
				var d = this.data,
					proxy = d.proxy,
					mpt = d.firstMPT,
					min = 0.000001,
					val, pt, i, str;
				while (mpt) {
					val = proxy[mpt.v];
					if (mpt.r) {
						val = Math.round(val);
					} else if (val < min && val > -min) {
						val = 0;
					}
					mpt.t[mpt.p] = val;
					mpt = mpt._next;
				}
				if (d.autoRotate) {
					d.autoRotate.rotation = proxy.rotation;
				}
				//at the end, we must set the CSSPropTween's "e" (end) value dynamically here because that's what is used in the final setRatio() method.
				if (v === 1) {
					mpt = d.firstMPT;
					while (mpt) {
						pt = mpt.t;
						if (!pt.type) {
							pt.e = pt.s + pt.xs0;
						} else if (pt.type === 1) {
							str = pt.xs0 + pt.s + pt.xs1;
							for (i = 1; i < pt.l; i++) {
								str += pt["xn"+i] + pt["xs"+(i+1)];
							}
							pt.e = str;
						}
						mpt = mpt._next;
					}
				}
			},

			/**
			 * @private @constructor Used by a few SpecialProps to hold important values for proxies. For example, _parseToProxy() creates a MiniPropTween instance for each property that must get tweened on the proxy, and we record the original property name as well as the unique one we create for the proxy, plus whether or not the value needs to be rounded plus the original value.
			 * @param {!Object} t target object whose property we're tweening (often a CSSPropTween)
			 * @param {!string} p property name
			 * @param {(number|string|object)} v value
			 * @param {MiniPropTween=} next next MiniPropTween in the linked list
			 * @param {boolean=} r if true, the tweened value should be rounded to the nearest integer
			 */
			MiniPropTween = function(t, p, v, next, r) {
				this.t = t;
				this.p = p;
				this.v = v;
				this.r = r;
				if (next) {
					next._prev = this;
					this._next = next;
				}
			},

			/**
			 * @private Most other plugins (like BezierPlugin and ThrowPropsPlugin and others) can only tween numeric values, but CSSPlugin must accommodate special values that have a bunch of extra data (like a suffix or strings between numeric values, etc.). For example, boxShadow has values like "10px 10px 20px 30px rgb(255,0,0)" which would utterly confuse other plugins. This method allows us to split that data apart and grab only the numeric data and attach it to uniquely-named properties of a generic proxy object ({}) so that we can feed that to virtually any plugin to have the numbers tweened. However, we must also keep track of which properties from the proxy go with which CSSPropTween values and instances. So we create a linked list of MiniPropTweens. Each one records a target (the original CSSPropTween), property (like "s" or "xn1" or "xn2") that we're tweening and the unique property name that was used for the proxy (like "boxShadow_xn1" and "boxShadow_xn2") and whether or not they need to be rounded. That way, in the _setPluginRatio() method we can simply copy the values over from the proxy to the CSSPropTween instance(s). Then, when the main CSSPlugin setRatio() method runs and applies the CSSPropTween values accordingly, they're updated nicely. So the external plugin tweens the numbers, _setPluginRatio() copies them over, and setRatio() acts normally, applying css-specific values to the element.
			 * This method returns an object that has the following properties:
			 *  - proxy: a generic object containing the starting values for all the properties that will be tweened by the external plugin.  This is what we feed to the external _onInitTween() as the target
			 *  - end: a generic object containing the ending values for all the properties that will be tweened by the external plugin. This is what we feed to the external plugin's _onInitTween() as the destination values
			 *  - firstMPT: the first MiniPropTween in the linked list
			 *  - pt: the first CSSPropTween in the linked list that was created when parsing. If shallow is true, this linked list will NOT attach to the one passed into the _parseToProxy() as the "pt" (4th) parameter.
			 * @param {!Object} t target object to be tweened
			 * @param {!(Object|string)} vars the object containing the information about the tweening values (typically the end/destination values) that should be parsed
			 * @param {!CSSPlugin} cssp The CSSPlugin instance
			 * @param {CSSPropTween=} pt the next CSSPropTween in the linked list
			 * @param {TweenPlugin=} plugin the external TweenPlugin instance that will be handling tweening the numeric values
			 * @param {boolean=} shallow if true, the resulting linked list from the parse will NOT be attached to the CSSPropTween that was passed in as the "pt" (4th) parameter.
			 * @return An object containing the following properties: proxy, end, firstMPT, and pt (see above for descriptions)
			 */
			_parseToProxy = _internals._parseToProxy = function(t, vars, cssp, pt, plugin, shallow) {
				var bpt = pt,
					start = {},
					end = {},
					transform = cssp._transform,
					oldForce = _forcePT,
					i, p, xp, mpt, firstPT;
				cssp._transform = null;
				_forcePT = vars;
				pt = firstPT = cssp.parse(t, vars, pt, plugin);
				_forcePT = oldForce;
				//break off from the linked list so the new ones are isolated.
				if (shallow) {
					cssp._transform = transform;
					if (bpt) {
						bpt._prev = null;
						if (bpt._prev) {
							bpt._prev._next = null;
						}
					}
				}
				while (pt && pt !== bpt) {
					if (pt.type <= 1) {
						p = pt.p;
						end[p] = pt.s + pt.c;
						start[p] = pt.s;
						if (!shallow) {
							mpt = new MiniPropTween(pt, "s", p, mpt, pt.r);
							pt.c = 0;
						}
						if (pt.type === 1) {
							i = pt.l;
							while (--i > 0) {
								xp = "xn" + i;
								p = pt.p + "_" + xp;
								end[p] = pt.data[xp];
								start[p] = pt[xp];
								if (!shallow) {
									mpt = new MiniPropTween(pt, xp, p, mpt, pt.rxp[xp]);
								}
							}
						}
					}
					pt = pt._next;
				}
				return {proxy:start, end:end, firstMPT:mpt, pt:firstPT};
			},



			/**
			 * @constructor Each property that is tweened has at least one CSSPropTween associated with it. These instances store important information like the target, property, starting value, amount of change, etc. They can also optionally have a number of "extra" strings and numeric values named xs1, xn1, xs2, xn2, xs3, xn3, etc. where "s" indicates string and "n" indicates number. These can be pieced together in a complex-value tween (type:1) that has alternating types of data like a string, number, string, number, etc. For example, boxShadow could be "5px 5px 8px rgb(102, 102, 51)". In that value, there are 6 numbers that may need to tween and then pieced back together into a string again with spaces, suffixes, etc. xs0 is special in that it stores the suffix for standard (type:0) tweens, -OR- the first string (prefix) in a complex-value (type:1) CSSPropTween -OR- it can be the non-tweening value in a type:-1 CSSPropTween. We do this to conserve memory.
			 * CSSPropTweens have the following optional properties as well (not defined through the constructor):
			 *  - l: Length in terms of the number of extra properties that the CSSPropTween has (default: 0). For example, for a boxShadow we may need to tween 5 numbers in which case l would be 5; Keep in mind that the start/end values for the first number that's tweened are always stored in the s and c properties to conserve memory. All additional values thereafter are stored in xn1, xn2, etc.
			 *  - xfirst: The first instance of any sub-CSSPropTweens that are tweening properties of this instance. For example, we may split up a boxShadow tween so that there's a main CSSPropTween of type:1 that has various xs* and xn* values associated with the h-shadow, v-shadow, blur, color, etc. Then we spawn a CSSPropTween for each of those that has a higher priority and runs BEFORE the main CSSPropTween so that the values are all set by the time it needs to re-assemble them. The xfirst gives us an easy way to identify the first one in that chain which typically ends at the main one (because they're all prepende to the linked list)
			 *  - plugin: The TweenPlugin instance that will handle the tweening of any complex values. For example, sometimes we don't want to use normal subtweens (like xfirst refers to) to tween the values - we might want ThrowPropsPlugin or BezierPlugin some other plugin to do the actual tweening, so we create a plugin instance and store a reference here. We need this reference so that if we get a request to round values or disable a tween, we can pass along that request.
			 *  - data: Arbitrary data that needs to be stored with the CSSPropTween. Typically if we're going to have a plugin handle the tweening of a complex-value tween, we create a generic object that stores the END values that we're tweening to and the CSSPropTween's xs1, xs2, etc. have the starting values. We store that object as data. That way, we can simply pass that object to the plugin and use the CSSPropTween as the target.
			 *  - setRatio: Only used for type:2 tweens that require custom functionality. In this case, we call the CSSPropTween's setRatio() method and pass the ratio each time the tween updates. This isn't quite as efficient as doing things directly in the CSSPlugin's setRatio() method, but it's very convenient and flexible.
			 * @param {!Object} t Target object whose property will be tweened. Often a DOM element, but not always. It could be anything.
			 * @param {string} p Property to tween (name). For example, to tween element.width, p would be "width".
			 * @param {number} s Starting numeric value
			 * @param {number} c Change in numeric value over the course of the entire tween. For example, if element.width starts at 5 and should end at 100, c would be 95.
			 * @param {CSSPropTween=} next The next CSSPropTween in the linked list. If one is defined, we will define its _prev as the new instance, and the new instance's _next will be pointed at it.
			 * @param {number=} type The type of CSSPropTween where -1 = a non-tweening value, 0 = a standard simple tween, 1 = a complex value (like one that has multiple numbers in a comma- or space-delimited string like border:"1px solid red"), and 2 = one that uses a custom setRatio function that does all of the work of applying the values on each update.
			 * @param {string=} n Name of the property that should be used for overwriting purposes which is typically the same as p but not always. For example, we may need to create a subtween for the 2nd part of a "clip:rect(...)" tween in which case "p" might be xs1 but "n" is still "clip"
			 * @param {boolean=} r If true, the value(s) should be rounded
			 * @param {number=} pr Priority in the linked list order. Higher priority CSSPropTweens will be updated before lower priority ones. The default priority is 0.
			 * @param {string=} b Beginning value. We store this to ensure that it is EXACTLY what it was when the tween began without any risk of interpretation issues.
			 * @param {string=} e Ending value. We store this to ensure that it is EXACTLY what the user defined at the end of the tween without any risk of interpretation issues.
			 */
			CSSPropTween = _internals.CSSPropTween = function(t, p, s, c, next, type, n, r, pr, b, e) {
				this.t = t; //target
				this.p = p; //property
				this.s = s; //starting value
				this.c = c; //change value
				this.n = n || p; //name that this CSSPropTween should be associated to (usually the same as p, but not always - n is what overwriting looks at)
				if (!(t instanceof CSSPropTween)) {
					_overwriteProps.push(this.n);
				}
				this.r = r; //round (boolean)
				this.type = type || 0; //0 = normal tween, -1 = non-tweening (in which case xs0 will be applied to the target's property, like tp.t[tp.p] = tp.xs0), 1 = complex-value SpecialProp, 2 = custom setRatio() that does all the work
				if (pr) {
					this.pr = pr;
					_hasPriority = true;
				}
				this.b = (b === undefined) ? s : b;
				this.e = (e === undefined) ? s + c : e;
				if (next) {
					this._next = next;
					next._prev = this;
				}
			},

			/**
			 * Takes a target, the beginning value and ending value (as strings) and parses them into a CSSPropTween (possibly with child CSSPropTweens) that accommodates multiple numbers, colors, comma-delimited values, etc. For example:
			 * sp.parseComplex(element, "boxShadow", "5px 10px 20px rgb(255,102,51)", "0px 0px 0px red", true, "0px 0px 0px rgb(0,0,0,0)", pt);
			 * It will walk through the beginning and ending values (which should be in the same format with the same number and type of values) and figure out which parts are numbers, what strings separate the numeric/tweenable values, and then create the CSSPropTweens accordingly. If a plugin is defined, no child CSSPropTweens will be created. Instead, the ending values will be stored in the "data" property of the returned CSSPropTween like: {s:-5, xn1:-10, xn2:-20, xn3:255, xn4:0, xn5:0} so that it can be fed to any other plugin and it'll be plain numeric tweens but the recomposition of the complex value will be handled inside CSSPlugin's setRatio().
			 * If a setRatio is defined, the type of the CSSPropTween will be set to 2 and recomposition of the values will be the responsibility of that method.
			 *
			 * @param {!Object} t Target whose property will be tweened
			 * @param {!string} p Property that will be tweened (its name, like "left" or "backgroundColor" or "boxShadow")
			 * @param {string} b Beginning value
			 * @param {string} e Ending value
			 * @param {boolean} clrs If true, the value could contain a color value like "rgb(255,0,0)" or "#F00" or "red". The default is false, so no colors will be recognized (a performance optimization)
			 * @param {(string|number|Object)} dflt The default beginning value that should be used if no valid beginning value is defined or if the number of values inside the complex beginning and ending values don't match
			 * @param {?CSSPropTween} pt CSSPropTween instance that is the current head of the linked list (we'll prepend to this).
			 * @param {number=} pr Priority in the linked list order. Higher priority properties will be updated before lower priority ones. The default priority is 0.
			 * @param {TweenPlugin=} plugin If a plugin should handle the tweening of extra properties, pass the plugin instance here. If one is defined, then NO subtweens will be created for any extra properties (the properties will be created - just not additional CSSPropTween instances to tween them) because the plugin is expected to do so. However, the end values WILL be populated in the "data" property, like {s:100, xn1:50, xn2:300}
			 * @param {function(number)=} setRatio If values should be set in a custom function instead of being pieced together in a type:1 (complex-value) CSSPropTween, define that custom function here.
			 * @return {CSSPropTween} The first CSSPropTween in the linked list which includes the new one(s) added by the parseComplex() call.
			 */
			_parseComplex = CSSPlugin.parseComplex = function(t, p, b, e, clrs, dflt, pt, pr, plugin, setRatio) {
				//DEBUG: _log("parseComplex: "+p+", b: "+b+", e: "+e);
				b = b || dflt || "";
				pt = new CSSPropTween(t, p, 0, 0, pt, (setRatio ? 2 : 1), null, false, pr, b, e);
				e += ""; //ensures it's a string
				var ba = b.split(", ").join(",").split(" "), //beginning array
					ea = e.split(", ").join(",").split(" "), //ending array
					l = ba.length,
					autoRound = (_autoRound !== false),
					i, xi, ni, bv, ev, bnums, enums, bn, rgba, temp, cv, str;
				if (e.indexOf(",") !== -1 || b.indexOf(",") !== -1) {
					ba = ba.join(" ").replace(_commasOutsideParenExp, ", ").split(" ");
					ea = ea.join(" ").replace(_commasOutsideParenExp, ", ").split(" ");
					l = ba.length;
				}
				if (l !== ea.length) {
					//DEBUG: _log("mismatched formatting detected on " + p + " (" + b + " vs " + e + ")");
					ba = (dflt || "").split(" ");
					l = ba.length;
				}
				pt.plugin = plugin;
				pt.setRatio = setRatio;
				for (i = 0; i < l; i++) {
					bv = ba[i];
					ev = ea[i];
					bn = parseFloat(bv);
					//if the value begins with a number (most common). It's fine if it has a suffix like px
					if (bn || bn === 0) {
						pt.appendXtra("", bn, _parseChange(ev, bn), ev.replace(_relNumExp, ""), (autoRound && ev.indexOf("px") !== -1), true);

					//if the value is a color
					} else if (clrs && (bv.charAt(0) === "#" || _colorLookup[bv] || _rgbhslExp.test(bv))) {
						str = ev.charAt(ev.length - 1) === "," ? ")," : ")"; //if there's a comma at the end, retain it.
						bv = _parseColor(bv);
						ev = _parseColor(ev);
						rgba = (bv.length + ev.length > 6);
						if (rgba && !_supportsOpacity && ev[3] === 0) { //older versions of IE don't support rgba(), so if the destination alpha is 0, just use "transparent" for the end color
							pt["xs" + pt.l] += pt.l ? " transparent" : "transparent";
							pt.e = pt.e.split(ea[i]).join("transparent");
						} else {
							if (!_supportsOpacity) { //old versions of IE don't support rgba().
								rgba = false;
							}
							pt.appendXtra((rgba ? "rgba(" : "rgb("), bv[0], ev[0] - bv[0], ",", true, true)
								.appendXtra("", bv[1], ev[1] - bv[1], ",", true)
								.appendXtra("", bv[2], ev[2] - bv[2], (rgba ? "," : str), true);
							if (rgba) {
								bv = (bv.length < 4) ? 1 : bv[3];
								pt.appendXtra("", bv, ((ev.length < 4) ? 1 : ev[3]) - bv, str, false);
							}
						}

					} else {
						bnums = bv.match(_numExp); //gets each group of numbers in the beginning value string and drops them into an array

						//if no number is found, treat it as a non-tweening value and just append the string to the current xs.
						if (!bnums) {
							pt["xs" + pt.l] += pt.l ? " " + bv : bv;

						//loop through all the numbers that are found and construct the extra values on the pt.
						} else {
							enums = ev.match(_relNumExp); //get each group of numbers in the end value string and drop them into an array. We allow relative values too, like +=50 or -=.5
							if (!enums || enums.length !== bnums.length) {
								//DEBUG: _log("mismatched formatting detected on " + p + " (" + b + " vs " + e + ")");
								return pt;
							}
							ni = 0;
							for (xi = 0; xi < bnums.length; xi++) {
								cv = bnums[xi];
								temp = bv.indexOf(cv, ni);
								pt.appendXtra(bv.substr(ni, temp - ni), Number(cv), _parseChange(enums[xi], cv), "", (autoRound && bv.substr(temp + cv.length, 2) === "px"), (xi === 0));
								ni = temp + cv.length;
							}
							pt["xs" + pt.l] += bv.substr(ni);
						}
					}
				}
				//if there are relative values ("+=" or "-=" prefix), we need to adjust the ending value to eliminate the prefixes and combine the values properly.
				if (e.indexOf("=") !== -1) if (pt.data) {
					str = pt.xs0 + pt.data.s;
					for (i = 1; i < pt.l; i++) {
						str += pt["xs" + i] + pt.data["xn" + i];
					}
					pt.e = str + pt["xs" + i];
				}
				if (!pt.l) {
					pt.type = -1;
					pt.xs0 = pt.e;
				}
				return pt.xfirst || pt;
			},
			i = 9;


		p = CSSPropTween.prototype;
		p.l = p.pr = 0; //length (number of extra properties like xn1, xn2, xn3, etc.
		while (--i > 0) {
			p["xn" + i] = 0;
			p["xs" + i] = "";
		}
		p.xs0 = "";
		p._next = p._prev = p.xfirst = p.data = p.plugin = p.setRatio = p.rxp = null;


		/**
		 * Appends and extra tweening value to a CSSPropTween and automatically manages any prefix and suffix strings. The first extra value is stored in the s and c of the main CSSPropTween instance, but thereafter any extras are stored in the xn1, xn2, xn3, etc. The prefixes and suffixes are stored in the xs0, xs1, xs2, etc. properties. For example, if I walk through a clip value like "rect(10px, 5px, 0px, 20px)", the values would be stored like this:
		 * xs0:"rect(", s:10, xs1:"px, ", xn1:5, xs2:"px, ", xn2:0, xs3:"px, ", xn3:20, xn4:"px)"
		 * And they'd all get joined together when the CSSPlugin renders (in the setRatio() method).
		 * @param {string=} pfx Prefix (if any)
		 * @param {!number} s Starting value
		 * @param {!number} c Change in numeric value over the course of the entire tween. For example, if the start is 5 and the end is 100, the change would be 95.
		 * @param {string=} sfx Suffix (if any)
		 * @param {boolean=} r Round (if true).
		 * @param {boolean=} pad If true, this extra value should be separated by the previous one by a space. If there is no previous extra and pad is true, it will automatically drop the space.
		 * @return {CSSPropTween} returns itself so that multiple methods can be chained together.
		 */
		p.appendXtra = function(pfx, s, c, sfx, r, pad) {
			var pt = this,
				l = pt.l;
			pt["xs" + l] += (pad && l) ? " " + pfx : pfx || "";
			if (!c) if (l !== 0 && !pt.plugin) { //typically we'll combine non-changing values right into the xs to optimize performance, but we don't combine them when there's a plugin that will be tweening the values because it may depend on the values being split apart, like for a bezier, if a value doesn't change between the first and second iteration but then it does on the 3rd, we'll run into trouble because there's no xn slot for that value!
				pt["xs" + l] += s + (sfx || "");
				return pt;
			}
			pt.l++;
			pt.type = pt.setRatio ? 2 : 1;
			pt["xs" + pt.l] = sfx || "";
			if (l > 0) {
				pt.data["xn" + l] = s + c;
				pt.rxp["xn" + l] = r; //round extra property (we need to tap into this in the _parseToProxy() method)
				pt["xn" + l] = s;
				if (!pt.plugin) {
					pt.xfirst = new CSSPropTween(pt, "xn" + l, s, c, pt.xfirst || pt, 0, pt.n, r, pt.pr);
					pt.xfirst.xs0 = 0; //just to ensure that the property stays numeric which helps modern browsers speed up processing. Remember, in the setRatio() method, we do pt.t[pt.p] = val + pt.xs0 so if pt.xs0 is "" (the default), it'll cast the end value as a string. When a property is a number sometimes and a string sometimes, it prevents the compiler from locking in the data type, slowing things down slightly.
				}
				return pt;
			}
			pt.data = {s:s + c};
			pt.rxp = {};
			pt.s = s;
			pt.c = c;
			pt.r = r;
			return pt;
		};

		/**
		 * @constructor A SpecialProp is basically a css property that needs to be treated in a non-standard way, like if it may contain a complex value like boxShadow:"5px 10px 15px rgb(255, 102, 51)" or if it is associated with another plugin like ThrowPropsPlugin or BezierPlugin. Every SpecialProp is associated with a particular property name like "boxShadow" or "throwProps" or "bezier" and it will intercept those values in the vars object that's passed to the CSSPlugin and handle them accordingly.
		 * @param {!string} p Property name (like "boxShadow" or "throwProps")
		 * @param {Object=} options An object containing any of the following configuration options:
		 *                      - defaultValue: the default value
		 *                      - parser: A function that should be called when the associated property name is found in the vars. This function should return a CSSPropTween instance and it should ensure that it is properly inserted into the linked list. It will receive 4 paramters: 1) The target, 2) The value defined in the vars, 3) The CSSPlugin instance (whose _firstPT should be used for the linked list), and 4) A computed style object if one was calculated (this is a speed optimization that allows retrieval of starting values quicker)
		 *                      - formatter: a function that formats any value received for this special property (for example, boxShadow could take "5px 5px red" and format it to "5px 5px 0px 0px red" so that both the beginning and ending values have a common order and quantity of values.)
		 *                      - prefix: if true, we'll determine whether or not this property requires a vendor prefix (like Webkit or Moz or ms or O)
		 *                      - color: set this to true if the value for this SpecialProp may contain color-related values like rgb(), rgba(), etc.
		 *                      - priority: priority in the linked list order. Higher priority SpecialProps will be updated before lower priority ones. The default priority is 0.
		 *                      - multi: if true, the formatter should accommodate a comma-delimited list of values, like boxShadow could have multiple boxShadows listed out.
		 *                      - collapsible: if true, the formatter should treat the value like it's a top/right/bottom/left value that could be collapsed, like "5px" would apply to all, "5px, 10px" would use 5px for top/bottom and 10px for right/left, etc.
		 *                      - keyword: a special keyword that can [optionally] be found inside the value (like "inset" for boxShadow). This allows us to validate beginning/ending values to make sure they match (if the keyword is found in one, it'll be added to the other for consistency by default).
		 */
		var SpecialProp = function(p, options) {
				options = options || {};
				this.p = options.prefix ? _checkPropPrefix(p) || p : p;
				_specialProps[p] = _specialProps[this.p] = this;
				this.format = options.formatter || _getFormatter(options.defaultValue, options.color, options.collapsible, options.multi);
				if (options.parser) {
					this.parse = options.parser;
				}
				this.clrs = options.color;
				this.multi = options.multi;
				this.keyword = options.keyword;
				this.dflt = options.defaultValue;
				this.pr = options.priority || 0;
			},

			//shortcut for creating a new SpecialProp that can accept multiple properties as a comma-delimited list (helps minification). dflt can be an array for multiple values (we don't do a comma-delimited list because the default value may contain commas, like rect(0px,0px,0px,0px)). We attach this method to the SpecialProp class/object instead of using a private _createSpecialProp() method so that we can tap into it externally if necessary, like from another plugin.
			_registerComplexSpecialProp = _internals._registerComplexSpecialProp = function(p, options, defaults) {
				if (typeof(options) !== "object") {
					options = {parser:defaults}; //to make backwards compatible with older versions of BezierPlugin and ThrowPropsPlugin
				}
				var a = p.split(","),
					d = options.defaultValue,
					i, temp;
				defaults = defaults || [d];
				for (i = 0; i < a.length; i++) {
					options.prefix = (i === 0 && options.prefix);
					options.defaultValue = defaults[i] || d;
					temp = new SpecialProp(a[i], options);
				}
			},

			//creates a placeholder special prop for a plugin so that the property gets caught the first time a tween of it is attempted, and at that time it makes the plugin register itself, thus taking over for all future tweens of that property. This allows us to not mandate that things load in a particular order and it also allows us to log() an error that informs the user when they attempt to tween an external plugin-related property without loading its .js file.
			_registerPluginProp = function(p) {
				if (!_specialProps[p]) {
					var pluginName = p.charAt(0).toUpperCase() + p.substr(1) + "Plugin";
					_registerComplexSpecialProp(p, {parser:function(t, e, p, cssp, pt, plugin, vars) {
						var pluginClass = _globals.com.greensock.plugins[pluginName];
						if (!pluginClass) {
							_log("Error: " + pluginName + " js file not loaded.");
							return pt;
						}
						pluginClass._cssRegister();
						return _specialProps[p].parse(t, e, p, cssp, pt, plugin, vars);
					}});
				}
			};


		p = SpecialProp.prototype;

		/**
		 * Alias for _parseComplex() that automatically plugs in certain values for this SpecialProp, like its property name, whether or not colors should be sensed, the default value, and priority. It also looks for any keyword that the SpecialProp defines (like "inset" for boxShadow) and ensures that the beginning and ending values have the same number of values for SpecialProps where multi is true (like boxShadow and textShadow can have a comma-delimited list)
		 * @param {!Object} t target element
		 * @param {(string|number|object)} b beginning value
		 * @param {(string|number|object)} e ending (destination) value
		 * @param {CSSPropTween=} pt next CSSPropTween in the linked list
		 * @param {TweenPlugin=} plugin If another plugin will be tweening the complex value, that TweenPlugin instance goes here.
		 * @param {function=} setRatio If a custom setRatio() method should be used to handle this complex value, that goes here.
		 * @return {CSSPropTween=} First CSSPropTween in the linked list
		 */
		p.parseComplex = function(t, b, e, pt, plugin, setRatio) {
			var kwd = this.keyword,
				i, ba, ea, l, bi, ei;
			//if this SpecialProp's value can contain a comma-delimited list of values (like boxShadow or textShadow), we must parse them in a special way, and look for a keyword (like "inset" for boxShadow) and ensure that the beginning and ending BOTH have it if the end defines it as such. We also must ensure that there are an equal number of values specified (we can't tween 1 boxShadow to 3 for example)
			if (this.multi) if (_commasOutsideParenExp.test(e) || _commasOutsideParenExp.test(b)) {
				ba = b.replace(_commasOutsideParenExp, "|").split("|");
				ea = e.replace(_commasOutsideParenExp, "|").split("|");
			} else if (kwd) {
				ba = [b];
				ea = [e];
			}
			if (ea) {
				l = (ea.length > ba.length) ? ea.length : ba.length;
				for (i = 0; i < l; i++) {
					b = ba[i] = ba[i] || this.dflt;
					e = ea[i] = ea[i] || this.dflt;
					if (kwd) {
						bi = b.indexOf(kwd);
						ei = e.indexOf(kwd);
						if (bi !== ei) {
							if (ei === -1) { //if the keyword isn't in the end value, remove it from the beginning one.
								ba[i] = ba[i].split(kwd).join("");
							} else if (bi === -1) { //if the keyword isn't in the beginning, add it.
								ba[i] += " " + kwd;
							}
						}
					}
				}
				b = ba.join(", ");
				e = ea.join(", ");
			}
			return _parseComplex(t, this.p, b, e, this.clrs, this.dflt, pt, this.pr, plugin, setRatio);
		};

		/**
		 * Accepts a target and end value and spits back a CSSPropTween that has been inserted into the CSSPlugin's linked list and conforms with all the conventions we use internally, like type:-1, 0, 1, or 2, setting up any extra property tweens, priority, etc. For example, if we have a boxShadow SpecialProp and call:
		 * this._firstPT = sp.parse(element, "5px 10px 20px rgb(2550,102,51)", "boxShadow", this);
		 * It should figure out the starting value of the element's boxShadow, compare it to the provided end value and create all the necessary CSSPropTweens of the appropriate types to tween the boxShadow. The CSSPropTween that gets spit back should already be inserted into the linked list (the 4th parameter is the current head, so prepend to that).
		 * @param {!Object} t Target object whose property is being tweened
		 * @param {Object} e End value as provided in the vars object (typically a string, but not always - like a throwProps would be an object).
		 * @param {!string} p Property name
		 * @param {!CSSPlugin} cssp The CSSPlugin instance that should be associated with this tween.
		 * @param {?CSSPropTween} pt The CSSPropTween that is the current head of the linked list (we'll prepend to it)
		 * @param {TweenPlugin=} plugin If a plugin will be used to tween the parsed value, this is the plugin instance.
		 * @param {Object=} vars Original vars object that contains the data for parsing.
		 * @return {CSSPropTween} The first CSSPropTween in the linked list which includes the new one(s) added by the parse() call.
		 */
		p.parse = function(t, e, p, cssp, pt, plugin, vars) {
			return this.parseComplex(t.style, this.format(_getStyle(t, this.p, _cs, false, this.dflt)), this.format(e), pt, plugin);
		};

		/**
		 * Registers a special property that should be intercepted from any "css" objects defined in tweens. This allows you to handle them however you want without CSSPlugin doing it for you. The 2nd parameter should be a function that accepts 3 parameters:
		 *  1) Target object whose property should be tweened (typically a DOM element)
		 *  2) The end/destination value (could be a string, number, object, or whatever you want)
		 *  3) The tween instance (you probably don't need to worry about this, but it can be useful for looking up information like the duration)
		 *
		 * Then, your function should return a function which will be called each time the tween gets rendered, passing a numeric "ratio" parameter to your function that indicates the change factor (usually between 0 and 1). For example:
		 *
		 * CSSPlugin.registerSpecialProp("myCustomProp", function(target, value, tween) {
		 *      var start = target.style.width;
		 *      return function(ratio) {
		 *              target.style.width = (start + value * ratio) + "px";
		 *              console.log("set width to " + target.style.width);
		 *          }
		 * }, 0);
		 *
		 * Then, when I do this tween, it will trigger my special property:
		 *
		 * TweenLite.to(element, 1, {css:{myCustomProp:100}});
		 *
		 * In the example, of course, we're just changing the width, but you can do anything you want.
		 *
		 * @param {!string} name Property name (or comma-delimited list of property names) that should be intercepted and handled by your function. For example, if I define "myCustomProp", then it would handle that portion of the following tween: TweenLite.to(element, 1, {css:{myCustomProp:100}})
		 * @param {!function(Object, Object, Object, string):function(number)} onInitTween The function that will be called when a tween of this special property is performed. The function will receive 4 parameters: 1) Target object that should be tweened, 2) Value that was passed to the tween, 3) The tween instance itself (rarely used), and 4) The property name that's being tweened. Your function should return a function that should be called on every update of the tween. That function will receive a single parameter that is a "change factor" value (typically between 0 and 1) indicating the amount of change as a ratio. You can use this to determine how to set the values appropriately in your function.
		 * @param {number=} priority Priority that helps the engine determine the order in which to set the properties (default: 0). Higher priority properties will be updated before lower priority ones.
		 */
		CSSPlugin.registerSpecialProp = function(name, onInitTween, priority) {
			_registerComplexSpecialProp(name, {parser:function(t, e, p, cssp, pt, plugin, vars) {
				var rv = new CSSPropTween(t, p, 0, 0, pt, 2, p, false, priority);
				rv.plugin = plugin;
				rv.setRatio = onInitTween(t, e, cssp._tween, p);
				return rv;
			}, priority:priority});
		};






		//transform-related methods and properties
		CSSPlugin.useSVGTransformAttr = _isSafari; //Safari has some rendering bugs when applying CSS transforms to SVG elements, so default to using the "transform" attribute instead.
		var _transformProps = ("scaleX,scaleY,scaleZ,x,y,z,skewX,skewY,rotation,rotationX,rotationY,perspective,xPercent,yPercent").split(","),
			_transformProp = _checkPropPrefix("transform"), //the Javascript (camelCase) transform property, like msTransform, WebkitTransform, MozTransform, or OTransform.
			_transformPropCSS = _prefixCSS + "transform",
			_transformOriginProp = _checkPropPrefix("transformOrigin"),
			_supports3D = (_checkPropPrefix("perspective") !== null),
			Transform = _internals.Transform = function() {
				this.perspective = parseFloat(CSSPlugin.defaultTransformPerspective) || 0;
				this.force3D = (CSSPlugin.defaultForce3D === false || !_supports3D) ? false : CSSPlugin.defaultForce3D || "auto";
			},
			_SVGElement = window.SVGElement,
			_useSVGTransformAttr,
			//Some browsers (like Firefox and IE) don't honor transform-origin properly in SVG elements, so we need to manually adjust the matrix accordingly. We feature detect here rather than always doing the conversion for certain browsers because they may fix the problem at some point in the future.

			_createSVG = function(type, container, attributes) {
				var element = _doc.createElementNS("http://www.w3.org/2000/svg", type),
					reg = /([a-z])([A-Z])/g,
					p;
				for (p in attributes) {
					element.setAttributeNS(null, p.replace(reg, "$1-$2").toLowerCase(), attributes[p]);
				}
				container.appendChild(element);
				return element;
			},
			_docElement = _doc.documentElement,
			_forceSVGTransformAttr = (function() {
				//IE and Android stock don't support CSS transforms on SVG elements, so we must write them to the "transform" attribute. We populate this variable in the _parseTransform() method, and only if/when we come across an SVG element
				var force = _ieVers || (/Android/i.test(_agent) && !window.chrome),
					svg, rect, width;
				if (_doc.createElementNS && !force) { //IE8 and earlier doesn't support SVG anyway
					svg = _createSVG("svg", _docElement);
					rect = _createSVG("rect", svg, {width:100, height:50, x:100});
					width = rect.getBoundingClientRect().width;
					rect.style[_transformOriginProp] = "50% 50%";
					rect.style[_transformProp] = "scaleX(0.5)";
					force = (width === rect.getBoundingClientRect().width && !(_isFirefox && _supports3D)); //note: Firefox fails the test even though it does support CSS transforms in 3D. Since we can't push 3D stuff into the transform attribute, we force Firefox to pass the test here (as long as it does truly support 3D).
					_docElement.removeChild(svg);
				}
				return force;
			})(),
			_parseSVGOrigin = function(e, local, decoratee, absolute) {
				var bbox, v;
				if (!absolute || !(v = absolute.split(" ")).length) {
					bbox = e.getBBox();
					local = _parsePosition(local).split(" ");
					v = [(local[0].indexOf("%") !== -1 ? parseFloat(local[0]) / 100 * bbox.width : parseFloat(local[0])) + bbox.x,
						 (local[1].indexOf("%") !== -1 ? parseFloat(local[1]) / 100 * bbox.height : parseFloat(local[1])) + bbox.y];
				}
				decoratee.xOrigin = parseFloat(v[0]);
				decoratee.yOrigin = parseFloat(v[1]);
				e.setAttribute("data-svg-origin", v.join(" "));
			},

			/**
			 * Parses the transform values for an element, returning an object with x, y, z, scaleX, scaleY, scaleZ, rotation, rotationX, rotationY, skewX, and skewY properties. Note: by default (for performance reasons), all skewing is combined into skewX and rotation but skewY still has a place in the transform object so that we can record how much of the skew is attributed to skewX vs skewY. Remember, a skewY of 10 looks the same as a rotation of 10 and skewX of -10.
			 * @param {!Object} t target element
			 * @param {Object=} cs computed style object (optional)
			 * @param {boolean=} rec if true, the transform values will be recorded to the target element's _gsTransform object, like target._gsTransform = {x:0, y:0, z:0, scaleX:1...}
			 * @param {boolean=} parse if true, we'll ignore any _gsTransform values that already exist on the element, and force a reparsing of the css (calculated style)
			 * @return {object} object containing all of the transform properties/values like {x:0, y:0, z:0, scaleX:1...}
			 */
			_getTransform = _internals.getTransform = function(t, cs, rec, parse) {
				if (t._gsTransform && rec && !parse) {
					return t._gsTransform; //if the element already has a _gsTransform, use that. Note: some browsers don't accurately return the calculated style for the transform (particularly for SVG), so it's almost always safest to just use the values we've already applied rather than re-parsing things.
				}
				var tm = rec ? t._gsTransform || new Transform() : new Transform(),
					invX = (tm.scaleX < 0), //in order to interpret things properly, we need to know if the user applied a negative scaleX previously so that we can adjust the rotation and skewX accordingly. Otherwise, if we always interpret a flipped matrix as affecting scaleY and the user only wants to tween the scaleX on multiple sequential tweens, it would keep the negative scaleY without that being the user's intent.
					min = 0.00002,
					rnd = 100000,
					zOrigin = _supports3D ? parseFloat(_getStyle(t, _transformOriginProp, cs, false, "0 0 0").split(" ")[2]) || tm.zOrigin  || 0 : 0,
					defaultTransformPerspective = parseFloat(CSSPlugin.defaultTransformPerspective) || 0,
					isDefault, s, m, i, n, dec, scaleX, scaleY, rotation, skewX;
				if (_transformProp) {
					s = _getStyle(t, _transformPropCSS, cs, true);
				} else if (t.currentStyle) {
					//for older versions of IE, we need to interpret the filter portion that is in the format: progid:DXImageTransform.Microsoft.Matrix(M11=6.123233995736766e-17, M12=-1, M21=1, M22=6.123233995736766e-17, sizingMethod='auto expand') Notice that we need to swap b and c compared to a normal matrix.
					s = t.currentStyle.filter.match(_ieGetMatrixExp);
					s = (s && s.length === 4) ? [s[0].substr(4), Number(s[2].substr(4)), Number(s[1].substr(4)), s[3].substr(4), (tm.x || 0), (tm.y || 0)].join(",") : "";
				}
				isDefault = (!s || s === "none" || s === "matrix(1, 0, 0, 1, 0, 0)");
				tm.svg = !!(_SVGElement && typeof(t.getBBox) === "function" && t.getCTM && (!t.parentNode || (t.parentNode.getBBox && t.parentNode.getCTM))); //don't just rely on "instanceof _SVGElement" because if the SVG is embedded via an object tag, it won't work (SVGElement is mapped to a different object)
				if (tm.svg) {
					if (isDefault && (t.style[_transformProp] + "").indexOf("matrix") !== -1) { //some browsers (like Chrome 40) don't correctly report transforms that are applied inline on an SVG element (they don't get included in the computed style), so we double-check here and accept matrix values
						s = t.style[_transformProp];
						isDefault = false;
					}
					_parseSVGOrigin(t, _getStyle(t, _transformOriginProp, _cs, false, "50% 50%") + "", tm, t.getAttribute("data-svg-origin"));
					_useSVGTransformAttr = CSSPlugin.useSVGTransformAttr || _forceSVGTransformAttr;
					m = t.getAttribute("transform");
					if (isDefault && m && m.indexOf("matrix") !== -1) { //just in case there's a "transform" value specified as an attribute instead of CSS style. Only accept a matrix, though.
						s = m;
						isDefault = 0;
					}
				}
				if (!isDefault) {
					//split the matrix values out into an array (m for matrix)
					m = (s || "").match(/(?:\-|\b)[\d\-\.e]+\b/gi) || [];
					i = m.length;
					while (--i > -1) {
						n = Number(m[i]);
						m[i] = (dec = n - (n |= 0)) ? ((dec * rnd + (dec < 0 ? -0.5 : 0.5)) | 0) / rnd + n : n; //convert strings to Numbers and round to 5 decimal places to avoid issues with tiny numbers. Roughly 20x faster than Number.toFixed(). We also must make sure to round before dividing so that values like 0.9999999999 become 1 to avoid glitches in browser rendering and interpretation of flipped/rotated 3D matrices. And don't just multiply the number by rnd, floor it, and then divide by rnd because the bitwise operations max out at a 32-bit signed integer, thus it could get clipped at a relatively low value (like 22,000.00000 for example).
					}
					if (m.length === 16) {
						//we'll only look at these position-related 6 variables first because if x/y/z all match, it's relatively safe to assume we don't need to re-parse everything which risks losing important rotational information (like rotationX:180 plus rotationY:180 would look the same as rotation:180 - there's no way to know for sure which direction was taken based solely on the matrix3d() values)
						var a11 = m[0], a21 = m[1], a31 = m[2], a41 = m[3],
							a12 = m[4], a22 = m[5], a32 = m[6], a42 = m[7],
							a13 = m[8], a23 = m[9], a33 = m[10],
							a14 = m[12], a24 = m[13], a34 = m[14],
							a43 = m[11],
							angle = Math.atan2(a32, a33),
							t1, t2, t3, t4, cos, sin;

						//we manually compensate for non-zero z component of transformOrigin to work around bugs in Safari
						if (tm.zOrigin) {
							a34 = -tm.zOrigin;
							a14 = a13*a34-m[12];
							a24 = a23*a34-m[13];
							a34 = a33*a34+tm.zOrigin-m[14];
						}
						tm.rotationX = angle * _RAD2DEG;
						//rotationX
						if (angle) {
							cos = Math.cos(-angle);
							sin = Math.sin(-angle);
							t1 = a12*cos+a13*sin;
							t2 = a22*cos+a23*sin;
							t3 = a32*cos+a33*sin;
							a13 = a12*-sin+a13*cos;
							a23 = a22*-sin+a23*cos;
							a33 = a32*-sin+a33*cos;
							a43 = a42*-sin+a43*cos;
							a12 = t1;
							a22 = t2;
							a32 = t3;
						}
						//rotationY
						angle = Math.atan2(a13, a33);
						tm.rotationY = angle * _RAD2DEG;
						if (angle) {
							cos = Math.cos(-angle);
							sin = Math.sin(-angle);
							t1 = a11*cos-a13*sin;
							t2 = a21*cos-a23*sin;
							t3 = a31*cos-a33*sin;
							a23 = a21*sin+a23*cos;
							a33 = a31*sin+a33*cos;
							a43 = a41*sin+a43*cos;
							a11 = t1;
							a21 = t2;
							a31 = t3;
						}
						//rotationZ
						angle = Math.atan2(a21, a11);
						tm.rotation = angle * _RAD2DEG;
						if (angle) {
							cos = Math.cos(-angle);
							sin = Math.sin(-angle);
							a11 = a11*cos+a12*sin;
							t2 = a21*cos+a22*sin;
							a22 = a21*-sin+a22*cos;
							a32 = a31*-sin+a32*cos;
							a21 = t2;
						}

						if (tm.rotationX && Math.abs(tm.rotationX) + Math.abs(tm.rotation) > 359.9) { //when rotationY is set, it will often be parsed as 180 degrees different than it should be, and rotationX and rotation both being 180 (it looks the same), so we adjust for that here.
							tm.rotationX = tm.rotation = 0;
							tm.rotationY += 180;
						}

						tm.scaleX = ((Math.sqrt(a11 * a11 + a21 * a21) * rnd + 0.5) | 0) / rnd;
						tm.scaleY = ((Math.sqrt(a22 * a22 + a23 * a23) * rnd + 0.5) | 0) / rnd;
						tm.scaleZ = ((Math.sqrt(a32 * a32 + a33 * a33) * rnd + 0.5) | 0) / rnd;
						tm.skewX = 0;
						tm.perspective = a43 ? 1 / ((a43 < 0) ? -a43 : a43) : 0;
						tm.x = a14;
						tm.y = a24;
						tm.z = a34;
						if (tm.svg) {
							tm.x -= tm.xOrigin - (tm.xOrigin * a11 - tm.yOrigin * a12);
							tm.y -= tm.yOrigin - (tm.yOrigin * a21 - tm.xOrigin * a22);
						}

					} else if ((!_supports3D || parse || !m.length || tm.x !== m[4] || tm.y !== m[5] || (!tm.rotationX && !tm.rotationY)) && !(tm.x !== undefined && _getStyle(t, "display", cs) === "none")) { //sometimes a 6-element matrix is returned even when we performed 3D transforms, like if rotationX and rotationY are 180. In cases like this, we still need to honor the 3D transforms. If we just rely on the 2D info, it could affect how the data is interpreted, like scaleY might get set to -1 or rotation could get offset by 180 degrees. For example, do a TweenLite.to(element, 1, {css:{rotationX:180, rotationY:180}}) and then later, TweenLite.to(element, 1, {css:{rotationX:0}}) and without this conditional logic in place, it'd jump to a state of being unrotated when the 2nd tween starts. Then again, we need to honor the fact that the user COULD alter the transforms outside of CSSPlugin, like by manually applying new css, so we try to sense that by looking at x and y because if those changed, we know the changes were made outside CSSPlugin and we force a reinterpretation of the matrix values. Also, in Webkit browsers, if the element's "display" is "none", its calculated style value will always return empty, so if we've already recorded the values in the _gsTransform object, we'll just rely on those.
						var k = (m.length >= 6),
							a = k ? m[0] : 1,
							b = m[1] || 0,
							c = m[2] || 0,
							d = k ? m[3] : 1;
						tm.x = m[4] || 0;
						tm.y = m[5] || 0;
						scaleX = Math.sqrt(a * a + b * b);
						scaleY = Math.sqrt(d * d + c * c);
						rotation = (a || b) ? Math.atan2(b, a) * _RAD2DEG : tm.rotation || 0; //note: if scaleX is 0, we cannot accurately measure rotation. Same for skewX with a scaleY of 0. Therefore, we default to the previously recorded value (or zero if that doesn't exist).
						skewX = (c || d) ? Math.atan2(c, d) * _RAD2DEG + rotation : tm.skewX || 0;
						if (Math.abs(skewX) > 90 && Math.abs(skewX) < 270) {
							if (invX) {
								scaleX *= -1;
								skewX += (rotation <= 0) ? 180 : -180;
								rotation += (rotation <= 0) ? 180 : -180;
							} else {
								scaleY *= -1;
								skewX += (skewX <= 0) ? 180 : -180;
							}
						}
						tm.scaleX = scaleX;
						tm.scaleY = scaleY;
						tm.rotation = rotation;
						tm.skewX = skewX;
						if (_supports3D) {
							tm.rotationX = tm.rotationY = tm.z = 0;
							tm.perspective = defaultTransformPerspective;
							tm.scaleZ = 1;
						}
						if (tm.svg) {
							tm.x -= tm.xOrigin - (tm.xOrigin * a - tm.yOrigin * b);
							tm.y -= tm.yOrigin - (tm.yOrigin * d - tm.xOrigin * c);
						}
					}
					tm.zOrigin = zOrigin;
					//some browsers have a hard time with very small values like 2.4492935982947064e-16 (notice the "e-" towards the end) and would render the object slightly off. So we round to 0 in these cases. The conditional logic here is faster than calling Math.abs(). Also, browsers tend to render a SLIGHTLY rotated object in a fuzzy way, so we need to snap to exactly 0 when appropriate.
					for (i in tm) {
						if (tm[i] < min) if (tm[i] > -min) {
							tm[i] = 0;
						}
					}
				}
				//DEBUG: _log("parsed rotation of " + t.getAttribute("id")+": "+(tm.rotationX)+", "+(tm.rotationY)+", "+(tm.rotation)+", scale: "+tm.scaleX+", "+tm.scaleY+", "+tm.scaleZ+", position: "+tm.x+", "+tm.y+", "+tm.z+", perspective: "+tm.perspective);
				if (rec) {
					t._gsTransform = tm; //record to the object's _gsTransform which we use so that tweens can control individual properties independently (we need all the properties to accurately recompose the matrix in the setRatio() method)
					if (tm.svg) { //if we're supposed to apply transforms to the SVG element's "transform" attribute, make sure there aren't any CSS transforms applied or they'll override the attribute ones. Also clear the transform attribute if we're using CSS, just to be clean.
						if (_useSVGTransformAttr && t.style[_transformProp]) {
							_removeProp(t.style, _transformProp);
						} else if (!_useSVGTransformAttr && t.getAttribute("transform")) {
							t.removeAttribute("transform");
						}
					}
				}
				return tm;
			},

			//for setting 2D transforms in IE6, IE7, and IE8 (must use a "filter" to emulate the behavior of modern day browser transforms)
			_setIETransformRatio = function(v) {
				var t = this.data, //refers to the element's _gsTransform object
					ang = -t.rotation * _DEG2RAD,
					skew = ang + t.skewX * _DEG2RAD,
					rnd = 100000,
					a = ((Math.cos(ang) * t.scaleX * rnd) | 0) / rnd,
					b = ((Math.sin(ang) * t.scaleX * rnd) | 0) / rnd,
					c = ((Math.sin(skew) * -t.scaleY * rnd) | 0) / rnd,
					d = ((Math.cos(skew) * t.scaleY * rnd) | 0) / rnd,
					style = this.t.style,
					cs = this.t.currentStyle,
					filters, val;
				if (!cs) {
					return;
				}
				val = b; //just for swapping the variables an inverting them (reused "val" to avoid creating another variable in memory). IE's filter matrix uses a non-standard matrix configuration (angle goes the opposite way, and b and c are reversed and inverted)
				b = -c;
				c = -val;
				filters = cs.filter;
				style.filter = ""; //remove filters so that we can accurately measure offsetWidth/offsetHeight
				var w = this.t.offsetWidth,
					h = this.t.offsetHeight,
					clip = (cs.position !== "absolute"),
					m = "progid:DXImageTransform.Microsoft.Matrix(M11=" + a + ", M12=" + b + ", M21=" + c + ", M22=" + d,
					ox = t.x + (w * t.xPercent / 100),
					oy = t.y + (h * t.yPercent / 100),
					dx, dy;

				//if transformOrigin is being used, adjust the offset x and y
				if (t.ox != null) {
					dx = ((t.oxp) ? w * t.ox * 0.01 : t.ox) - w / 2;
					dy = ((t.oyp) ? h * t.oy * 0.01 : t.oy) - h / 2;
					ox += dx - (dx * a + dy * b);
					oy += dy - (dx * c + dy * d);
				}

				if (!clip) {
					m += ", sizingMethod='auto expand')";
				} else {
					dx = (w / 2);
					dy = (h / 2);
					//translate to ensure that transformations occur around the correct origin (default is center).
					m += ", Dx=" + (dx - (dx * a + dy * b) + ox) + ", Dy=" + (dy - (dx * c + dy * d) + oy) + ")";
				}
				if (filters.indexOf("DXImageTransform.Microsoft.Matrix(") !== -1) {
					style.filter = filters.replace(_ieSetMatrixExp, m);
				} else {
					style.filter = m + " " + filters; //we must always put the transform/matrix FIRST (before alpha(opacity=xx)) to avoid an IE bug that slices part of the object when rotation is applied with alpha.
				}

				//at the end or beginning of the tween, if the matrix is normal (1, 0, 0, 1) and opacity is 100 (or doesn't exist), remove the filter to improve browser performance.
				if (v === 0 || v === 1) if (a === 1) if (b === 0) if (c === 0) if (d === 1) if (!clip || m.indexOf("Dx=0, Dy=0") !== -1) if (!_opacityExp.test(filters) || parseFloat(RegExp.$1) === 100) if (filters.indexOf("gradient(" && filters.indexOf("Alpha")) === -1) {
					style.removeAttribute("filter");
				}

				//we must set the margins AFTER applying the filter in order to avoid some bugs in IE8 that could (in rare scenarios) cause them to be ignored intermittently (vibration).
				if (!clip) {
					var mult = (_ieVers < 8) ? 1 : -1, //in Internet Explorer 7 and before, the box model is broken, causing the browser to treat the width/height of the actual rotated filtered image as the width/height of the box itself, but Microsoft corrected that in IE8. We must use a negative offset in IE8 on the right/bottom
						marg, prop, dif;
					dx = t.ieOffsetX || 0;
					dy = t.ieOffsetY || 0;
					t.ieOffsetX = Math.round((w - ((a < 0 ? -a : a) * w + (b < 0 ? -b : b) * h)) / 2 + ox);
					t.ieOffsetY = Math.round((h - ((d < 0 ? -d : d) * h + (c < 0 ? -c : c) * w)) / 2 + oy);
					for (i = 0; i < 4; i++) {
						prop = _margins[i];
						marg = cs[prop];
						//we need to get the current margin in case it is being tweened separately (we want to respect that tween's changes)
						val = (marg.indexOf("px") !== -1) ? parseFloat(marg) : _convertToPixels(this.t, prop, parseFloat(marg), marg.replace(_suffixExp, "")) || 0;
						if (val !== t[prop]) {
							dif = (i < 2) ? -t.ieOffsetX : -t.ieOffsetY; //if another tween is controlling a margin, we cannot only apply the difference in the ieOffsets, so we essentially zero-out the dx and dy here in that case. We record the margin(s) later so that we can keep comparing them, making this code very flexible.
						} else {
							dif = (i < 2) ? dx - t.ieOffsetX : dy - t.ieOffsetY;
						}
						style[prop] = (t[prop] = Math.round( val - dif * ((i === 0 || i === 2) ? 1 : mult) )) + "px";
					}
				}
			},

			/* translates a super small decimal to a string WITHOUT scientific notation
			_safeDecimal = function(n) {
				var s = (n < 0 ? -n : n) + "",
					a = s.split("e-");
				return (n < 0 ? "-0." : "0.") + new Array(parseInt(a[1], 10) || 0).join("0") + a[0].split(".").join("");
			},
			*/

			_setTransformRatio = _internals.set3DTransformRatio = _internals.setTransformRatio = function(v) {
				var t = this.data, //refers to the element's _gsTransform object
					style = this.t.style,
					angle = t.rotation,
					rotationX = t.rotationX,
					rotationY = t.rotationY,
					sx = t.scaleX,
					sy = t.scaleY,
					sz = t.scaleZ,
					x = t.x,
					y = t.y,
					z = t.z,
					isSVG = t.svg,
					perspective = t.perspective,
					force3D = t.force3D,
					a11, a12, a13, a21, a22, a23, a31, a32, a33, a41, a42, a43,
					zOrigin, min, cos, sin, t1, t2, transform, comma, zero, skew, rnd;

				//check to see if we should render as 2D (and SVGs must use 2D when _useSVGTransformAttr is true)
				if (((((v === 1 || v === 0) && force3D === "auto" && (this.tween._totalTime === this.tween._totalDuration || !this.tween._totalTime)) || !force3D) && !z && !perspective && !rotationY && !rotationX) || (_useSVGTransformAttr && isSVG) || !_supports3D) { //on the final render (which could be 0 for a from tween), if there are no 3D aspects, render in 2D to free up memory and improve performance especially on mobile devices. Check the tween's totalTime/totalDuration too in order to make sure it doesn't happen between repeats if it's a repeating tween.

					//2D
					if (angle || t.skewX || isSVG) {
						angle *= _DEG2RAD;
						skew = t.skewX * _DEG2RAD;
						rnd = 100000;
						a11 = Math.cos(angle) * sx;
						a21 = Math.sin(angle) * sx;
						a12 = Math.sin(angle - skew) * -sy;
						a22 = Math.cos(angle - skew) * sy;
						if (skew && t.skewType === "simple") { //by default, we compensate skewing on the other axis to make it look more natural, but you can set the skewType to "simple" to use the uncompensated skewing that CSS does
							t1 = Math.tan(skew);
							t1 = Math.sqrt(1 + t1 * t1);
							a12 *= t1;
							a22 *= t1;
							if (t.skewY) {
								a11 *= t1;
								a21 *= t1;
							}
						}
						if (isSVG) {
							x += t.xOrigin - (t.xOrigin * a11 + t.yOrigin * a12);
							y += t.yOrigin - (t.xOrigin * a21 + t.yOrigin * a22);
							min = 0.000001;
							if (x < min) if (x > -min) {
								x = 0;
							}
							if (y < min) if (y > -min) {
								y = 0;
							}
						}
						transform = (((a11 * rnd) | 0) / rnd) + "," + (((a21 * rnd) | 0) / rnd) + "," + (((a12 * rnd) | 0) / rnd) + "," + (((a22 * rnd) | 0) / rnd) + "," + x + "," + y + ")";
						if (isSVG && _useSVGTransformAttr) {
							this.t.setAttribute("transform", "matrix(" + transform);
						} else {
							//some browsers have a hard time with very small values like 2.4492935982947064e-16 (notice the "e-" towards the end) and would render the object slightly off. So we round to 5 decimal places.
							style[_transformProp] = ((t.xPercent || t.yPercent) ? "translate(" + t.xPercent + "%," + t.yPercent + "%) matrix(" : "matrix(") + transform;
						}
					} else {
						style[_transformProp] = ((t.xPercent || t.yPercent) ? "translate(" + t.xPercent + "%," + t.yPercent + "%) matrix(" : "matrix(") + sx + ",0,0," + sy + "," + x + "," + y + ")";
					}
					return;

				}
				if (_isFirefox) { //Firefox has a bug (at least in v25) that causes it to render the transparent part of 32-bit PNG images as black when displayed inside an iframe and the 3D scale is very small and doesn't change sufficiently enough between renders (like if you use a Power4.easeInOut to scale from 0 to 1 where the beginning values only change a tiny amount to begin the tween before accelerating). In this case, we force the scale to be 0.00002 instead which is visually the same but works around the Firefox issue.
					min = 0.0001;
					if (sx < min && sx > -min) {
						sx = sz = 0.00002;
					}
					if (sy < min && sy > -min) {
						sy = sz = 0.00002;
					}
					if (perspective && !t.z && !t.rotationX && !t.rotationY) { //Firefox has a bug that causes elements to have an odd super-thin, broken/dotted black border on elements that have a perspective set but aren't utilizing 3D space (no rotationX, rotationY, or z).
						perspective = 0;
					}
				}
				if (angle || t.skewX) {
					angle *= _DEG2RAD;
					cos = a11 = Math.cos(angle);
					sin = a21 = Math.sin(angle);
					if (t.skewX) {
						angle -= t.skewX * _DEG2RAD;
						cos = Math.cos(angle);
						sin = Math.sin(angle);
						if (t.skewType === "simple") { //by default, we compensate skewing on the other axis to make it look more natural, but you can set the skewType to "simple" to use the uncompensated skewing that CSS does
							t1 = Math.tan(t.skewX * _DEG2RAD);
							t1 = Math.sqrt(1 + t1 * t1);
							cos *= t1;
							sin *= t1;
							if (t.skewY) {
								a11 *= t1;
								a21 *= t1;
							}
						}
					}
					a12 = -sin;
					a22 = cos;

				} else if (!rotationY && !rotationX && sz === 1 && !perspective && !isSVG) { //if we're only translating and/or 2D scaling, this is faster...
					style[_transformProp] = ((t.xPercent || t.yPercent) ? "translate(" + t.xPercent + "%," + t.yPercent + "%) translate3d(" : "translate3d(") + x + "px," + y + "px," + z +"px)" + ((sx !== 1 || sy !== 1) ? " scale(" + sx + "," + sy + ")" : "");
					return;
				} else {
					a11 = a22 = 1;
					a12 = a21 = 0;
				}
				// KEY  INDEX   AFFECTS
				// a11  0       rotation, rotationY, scaleX
				// a21  1       rotation, rotationY, scaleX
				// a31  2       rotationY, scaleX
				// a41  3       rotationY, scaleX
				// a12  4       rotation, skewX, rotationX, scaleY
				// a22  5       rotation, skewX, rotationX, scaleY
				// a32  6       rotationX, scaleY
				// a42  7       rotationX, scaleY
				// a13  8       rotationY, rotationX, scaleZ
				// a23  9       rotationY, rotationX, scaleZ
				// a33  10      rotationY, rotationX, scaleZ
				// a43  11      rotationY, rotationX, perspective, scaleZ
				// a14  12      x, zOrigin, svgOrigin
				// a24  13      y, zOrigin, svgOrigin
				// a34  14      z, zOrigin
				// a44  15
				// rotation: Math.atan2(a21, a11)
				// rotationY: Math.atan2(a13, a33) (or Math.atan2(a13, a11))
				// rotationX: Math.atan2(a32, a33)
				a33 = 1;
				a13 = a23 = a31 = a32 = a41 = a42 = 0;
				a43 = (perspective) ? -1 / perspective : 0;
				zOrigin = t.zOrigin;
				min = 0.000001; //threshold below which browsers use scientific notation which won't work.
				comma = ",";
				zero = "0";
				angle = rotationY * _DEG2RAD;
				if (angle) {
					cos = Math.cos(angle);
					sin = Math.sin(angle);
					a31 = -sin;
					a41 = a43*-sin;
					a13 = a11*sin;
					a23 = a21*sin;
					a33 = cos;
					a43 *= cos;
					a11 *= cos;
					a21 *= cos;
				}
				angle = rotationX * _DEG2RAD;
				if (angle) {
					cos = Math.cos(angle);
					sin = Math.sin(angle);
					t1 = a12*cos+a13*sin;
					t2 = a22*cos+a23*sin;
					a32 = a33*sin;
					a42 = a43*sin;
					a13 = a12*-sin+a13*cos;
					a23 = a22*-sin+a23*cos;
					a33 = a33*cos;
					a43 = a43*cos;
					a12 = t1;
					a22 = t2;
				}
				if (sz !== 1) {
					a13*=sz;
					a23*=sz;
					a33*=sz;
					a43*=sz;
				}
				if (sy !== 1) {
					a12*=sy;
					a22*=sy;
					a32*=sy;
					a42*=sy;
				}
				if (sx !== 1) {
					a11*=sx;
					a21*=sx;
					a31*=sx;
					a41*=sx;
				}

				if (zOrigin || isSVG) {
					if (zOrigin) {
						x += a13*-zOrigin;
						y += a23*-zOrigin;
						z += a33*-zOrigin+zOrigin;
					}
					if (isSVG) { //due to bugs in some browsers, we need to manage the transform-origin of SVG manually
						x += t.xOrigin - (t.xOrigin * a11 + t.yOrigin * a12);
						y += t.yOrigin - (t.xOrigin * a21 + t.yOrigin * a22);
					}
					if (x < min && x > -min) {
						x = zero;
					}
					if (y < min && y > -min) {
						y = zero;
					}
					if (z < min && z > -min) {
						z = 0; //don't use string because we calculate perspective later and need the number.
					}
				}

				//optimized way of concatenating all the values into a string. If we do it all in one shot, it's slower because of the way browsers have to create temp strings and the way it affects memory. If we do it piece-by-piece with +=, it's a bit slower too. We found that doing it in these sized chunks works best overall:
				transform = ((t.xPercent || t.yPercent) ? "translate(" + t.xPercent + "%," + t.yPercent + "%) matrix3d(" : "matrix3d(");
				transform += ((a11 < min && a11 > -min) ? zero : a11) + comma + ((a21 < min && a21 > -min) ? zero : a21) + comma + ((a31 < min && a31 > -min) ? zero : a31);
				transform += comma + ((a41 < min && a41 > -min) ? zero : a41) + comma + ((a12 < min && a12 > -min) ? zero : a12) + comma + ((a22 < min && a22 > -min) ? zero : a22);
				if (rotationX || rotationY) { //performance optimization (often there's no rotationX or rotationY, so we can skip these calculations)
					transform += comma + ((a32 < min && a32 > -min) ? zero : a32) + comma + ((a42 < min && a42 > -min) ? zero : a42) + comma + ((a13 < min && a13 > -min) ? zero : a13);
					transform += comma + ((a23 < min && a23 > -min) ? zero : a23) + comma + ((a33 < min && a33 > -min) ? zero : a33) + comma + ((a43 < min && a43 > -min) ? zero : a43) + comma;
				} else {
					transform += ",0,0,0,0,1,0,";
				}
				transform += x + comma + y + comma + z + comma + (perspective ? (1 + (-z / perspective)) : 1) + ")";

				style[_transformProp] = transform;
			};

		p = Transform.prototype;
		p.x = p.y = p.z = p.skewX = p.skewY = p.rotation = p.rotationX = p.rotationY = p.zOrigin = p.xPercent = p.yPercent = 0;
		p.scaleX = p.scaleY = p.scaleZ = 1;

		_registerComplexSpecialProp("transform,scale,scaleX,scaleY,scaleZ,x,y,z,rotation,rotationX,rotationY,rotationZ,skewX,skewY,shortRotation,shortRotationX,shortRotationY,shortRotationZ,transformOrigin,svgOrigin,transformPerspective,directionalRotation,parseTransform,force3D,skewType,xPercent,yPercent", {parser:function(t, e, p, cssp, pt, plugin, vars) {
			if (cssp._lastParsedTransform === vars) { return pt; } //only need to parse the transform once, and only if the browser supports it.
			cssp._lastParsedTransform = vars;
			var m1 = cssp._transform = _getTransform(t, _cs, true, vars.parseTransform),
				style = t.style,
				min = 0.000001,
				i = _transformProps.length,
				v = vars,
				endRotations = {},
				m2, skewY, copy, orig, has3D, hasChange, dr;
			if (typeof(v.transform) === "string" && _transformProp) { //for values like transform:"rotate(60deg) scale(0.5, 0.8)"
				copy = _tempDiv.style; //don't use the original target because it might be SVG in which case some browsers don't report computed style correctly.
				copy[_transformProp] = v.transform;
				copy.display = "block"; //if display is "none", the browser often refuses to report the transform properties correctly.
				copy.position = "absolute";
				_doc.body.appendChild(_tempDiv);
				m2 = _getTransform(_tempDiv, null, false);
				_doc.body.removeChild(_tempDiv);
			} else if (typeof(v) === "object") { //for values like scaleX, scaleY, rotation, x, y, skewX, and skewY or transform:{...} (object)
				m2 = {scaleX:_parseVal((v.scaleX != null) ? v.scaleX : v.scale, m1.scaleX),
					scaleY:_parseVal((v.scaleY != null) ? v.scaleY : v.scale, m1.scaleY),
					scaleZ:_parseVal(v.scaleZ, m1.scaleZ),
					x:_parseVal(v.x, m1.x),
					y:_parseVal(v.y, m1.y),
					z:_parseVal(v.z, m1.z),
					xPercent:_parseVal(v.xPercent, m1.xPercent),
					yPercent:_parseVal(v.yPercent, m1.yPercent),
					perspective:_parseVal(v.transformPerspective, m1.perspective)};
				dr = v.directionalRotation;
				if (dr != null) {
					if (typeof(dr) === "object") {
						for (copy in dr) {
							v[copy] = dr[copy];
						}
					} else {
						v.rotation = dr;
					}
				}
				if (typeof(v.x) === "string" && v.x.indexOf("%") !== -1) {
					m2.x = 0;
					m2.xPercent = _parseVal(v.x, m1.xPercent);
				}
				if (typeof(v.y) === "string" && v.y.indexOf("%") !== -1) {
					m2.y = 0;
					m2.yPercent = _parseVal(v.y, m1.yPercent);
				}

				m2.rotation = _parseAngle(("rotation" in v) ? v.rotation : ("shortRotation" in v) ? v.shortRotation + "_short" : ("rotationZ" in v) ? v.rotationZ : m1.rotation, m1.rotation, "rotation", endRotations);
				if (_supports3D) {
					m2.rotationX = _parseAngle(("rotationX" in v) ? v.rotationX : ("shortRotationX" in v) ? v.shortRotationX + "_short" : m1.rotationX || 0, m1.rotationX, "rotationX", endRotations);
					m2.rotationY = _parseAngle(("rotationY" in v) ? v.rotationY : ("shortRotationY" in v) ? v.shortRotationY + "_short" : m1.rotationY || 0, m1.rotationY, "rotationY", endRotations);
				}
				m2.skewX = (v.skewX == null) ? m1.skewX : _parseAngle(v.skewX, m1.skewX);

				//note: for performance reasons, we combine all skewing into the skewX and rotation values, ignoring skewY but we must still record it so that we can discern how much of the overall skew is attributed to skewX vs. skewY. Otherwise, if the skewY would always act relative (tween skewY to 10deg, for example, multiple times and if we always combine things into skewX, we can't remember that skewY was 10 from last time). Remember, a skewY of 10 degrees looks the same as a rotation of 10 degrees plus a skewX of -10 degrees.
				m2.skewY = (v.skewY == null) ? m1.skewY : _parseAngle(v.skewY, m1.skewY);
				if ((skewY = m2.skewY - m1.skewY)) {
					m2.skewX += skewY;
					m2.rotation += skewY;
				}
			}
			if (_supports3D && v.force3D != null) {
				m1.force3D = v.force3D;
				hasChange = true;
			}

			m1.skewType = v.skewType || m1.skewType || CSSPlugin.defaultSkewType;

			has3D = (m1.force3D || m1.z || m1.rotationX || m1.rotationY || m2.z || m2.rotationX || m2.rotationY || m2.perspective);
			if (!has3D && v.scale != null) {
				m2.scaleZ = 1; //no need to tween scaleZ.
			}

			while (--i > -1) {
				p = _transformProps[i];
				orig = m2[p] - m1[p];
				if (orig > min || orig < -min || v[p] != null || _forcePT[p] != null) {
					hasChange = true;
					pt = new CSSPropTween(m1, p, m1[p], orig, pt);
					if (p in endRotations) {
						pt.e = endRotations[p]; //directional rotations typically have compensated values during the tween, but we need to make sure they end at exactly what the user requested
					}
					pt.xs0 = 0; //ensures the value stays numeric in setRatio()
					pt.plugin = plugin;
					cssp._overwriteProps.push(pt.n);
				}
			}

			orig = v.transformOrigin;
			if (m1.svg && (orig || v.svgOrigin)) {
				_parseSVGOrigin(t, _parsePosition(orig), m2, v.svgOrigin);
				pt = new CSSPropTween(m1, "xOrigin", m1.xOrigin, m2.xOrigin - m1.xOrigin, pt, -1, "transformOrigin");
				pt.b = m1.xOrigin;
				pt.e = pt.xs0 = m2.xOrigin;
				pt = new CSSPropTween(m1, "yOrigin", m1.yOrigin, m2.yOrigin - m1.yOrigin, pt, -1, "transformOrigin");
				pt.b = m1.yOrigin;
				pt.e = pt.xs0 = m2.yOrigin;
				orig = _useSVGTransformAttr ? null : "0px 0px"; //certain browsers (like firefox) completely botch transform-origin, so we must remove it to prevent it from contaminating transforms. We manage it ourselves with xOrigin and yOrigin
			}
			if (orig || (_supports3D && has3D && m1.zOrigin)) { //if anything 3D is happening and there's a transformOrigin with a z component that's non-zero, we must ensure that the transformOrigin's z-component is set to 0 so that we can manually do those calculations to get around Safari bugs. Even if the user didn't specifically define a "transformOrigin" in this particular tween (maybe they did it via css directly).
				if (_transformProp) {
					hasChange = true;
					p = _transformOriginProp;
					orig = (orig || _getStyle(t, p, _cs, false, "50% 50%")) + ""; //cast as string to avoid errors
					pt = new CSSPropTween(style, p, 0, 0, pt, -1, "transformOrigin");
					pt.b = style[p];
					pt.plugin = plugin;
					if (_supports3D) {
						copy = m1.zOrigin;
						orig = orig.split(" ");
						m1.zOrigin = ((orig.length > 2 && !(copy !== 0 && orig[2] === "0px")) ? parseFloat(orig[2]) : copy) || 0; //Safari doesn't handle the z part of transformOrigin correctly, so we'll manually handle it in the _set3DTransformRatio() method.
						pt.xs0 = pt.e = orig[0] + " " + (orig[1] || "50%") + " 0px"; //we must define a z value of 0px specifically otherwise iOS 5 Safari will stick with the old one (if one was defined)!
						pt = new CSSPropTween(m1, "zOrigin", 0, 0, pt, -1, pt.n); //we must create a CSSPropTween for the _gsTransform.zOrigin so that it gets reset properly at the beginning if the tween runs backward (as opposed to just setting m1.zOrigin here)
						pt.b = copy;
						pt.xs0 = pt.e = m1.zOrigin;
					} else {
						pt.xs0 = pt.e = orig;
					}

					//for older versions of IE (6-8), we need to manually calculate things inside the setRatio() function. We record origin x and y (ox and oy) and whether or not the values are percentages (oxp and oyp).
				} else {
					_parsePosition(orig + "", m1);
				}
			}
			if (hasChange) {
				cssp._transformType = (!(m1.svg && _useSVGTransformAttr) && (has3D || this._transformType === 3)) ? 3 : 2; //quicker than calling cssp._enableTransforms();
			}
			return pt;
		}, prefix:true});

		_registerComplexSpecialProp("boxShadow", {defaultValue:"0px 0px 0px 0px #999", prefix:true, color:true, multi:true, keyword:"inset"});

		_registerComplexSpecialProp("borderRadius", {defaultValue:"0px", parser:function(t, e, p, cssp, pt, plugin) {
			e = this.format(e);
			var props = ["borderTopLeftRadius","borderTopRightRadius","borderBottomRightRadius","borderBottomLeftRadius"],
				style = t.style,
				ea1, i, es2, bs2, bs, es, bn, en, w, h, esfx, bsfx, rel, hn, vn, em;
			w = parseFloat(t.offsetWidth);
			h = parseFloat(t.offsetHeight);
			ea1 = e.split(" ");
			for (i = 0; i < props.length; i++) { //if we're dealing with percentages, we must convert things separately for the horizontal and vertical axis!
				if (this.p.indexOf("border")) { //older browsers used a prefix
					props[i] = _checkPropPrefix(props[i]);
				}
				bs = bs2 = _getStyle(t, props[i], _cs, false, "0px");
				if (bs.indexOf(" ") !== -1) {
					bs2 = bs.split(" ");
					bs = bs2[0];
					bs2 = bs2[1];
				}
				es = es2 = ea1[i];
				bn = parseFloat(bs);
				bsfx = bs.substr((bn + "").length);
				rel = (es.charAt(1) === "=");
				if (rel) {
					en = parseInt(es.charAt(0)+"1", 10);
					es = es.substr(2);
					en *= parseFloat(es);
					esfx = es.substr((en + "").length - (en < 0 ? 1 : 0)) || "";
				} else {
					en = parseFloat(es);
					esfx = es.substr((en + "").length);
				}
				if (esfx === "") {
					esfx = _suffixMap[p] || bsfx;
				}
				if (esfx !== bsfx) {
					hn = _convertToPixels(t, "borderLeft", bn, bsfx); //horizontal number (we use a bogus "borderLeft" property just because the _convertToPixels() method searches for the keywords "Left", "Right", "Top", and "Bottom" to determine of it's a horizontal or vertical property, and we need "border" in the name so that it knows it should measure relative to the element itself, not its parent.
					vn = _convertToPixels(t, "borderTop", bn, bsfx); //vertical number
					if (esfx === "%") {
						bs = (hn / w * 100) + "%";
						bs2 = (vn / h * 100) + "%";
					} else if (esfx === "em") {
						em = _convertToPixels(t, "borderLeft", 1, "em");
						bs = (hn / em) + "em";
						bs2 = (vn / em) + "em";
					} else {
						bs = hn + "px";
						bs2 = vn + "px";
					}
					if (rel) {
						es = (parseFloat(bs) + en) + esfx;
						es2 = (parseFloat(bs2) + en) + esfx;
					}
				}
				pt = _parseComplex(style, props[i], bs + " " + bs2, es + " " + es2, false, "0px", pt);
			}
			return pt;
		}, prefix:true, formatter:_getFormatter("0px 0px 0px 0px", false, true)});
		_registerComplexSpecialProp("backgroundPosition", {defaultValue:"0 0", parser:function(t, e, p, cssp, pt, plugin) {
			var bp = "background-position",
				cs = (_cs || _getComputedStyle(t, null)),
				bs = this.format( ((cs) ? _ieVers ? cs.getPropertyValue(bp + "-x") + " " + cs.getPropertyValue(bp + "-y") : cs.getPropertyValue(bp) : t.currentStyle.backgroundPositionX + " " + t.currentStyle.backgroundPositionY) || "0 0"), //Internet Explorer doesn't report background-position correctly - we must query background-position-x and background-position-y and combine them (even in IE10). Before IE9, we must do the same with the currentStyle object and use camelCase
				es = this.format(e),
				ba, ea, i, pct, overlap, src;
			if ((bs.indexOf("%") !== -1) !== (es.indexOf("%") !== -1)) {
				src = _getStyle(t, "backgroundImage").replace(_urlExp, "");
				if (src && src !== "none") {
					ba = bs.split(" ");
					ea = es.split(" ");
					_tempImg.setAttribute("src", src); //set the temp IMG's src to the background-image so that we can measure its width/height
					i = 2;
					while (--i > -1) {
						bs = ba[i];
						pct = (bs.indexOf("%") !== -1);
						if (pct !== (ea[i].indexOf("%") !== -1)) {
							overlap = (i === 0) ? t.offsetWidth - _tempImg.width : t.offsetHeight - _tempImg.height;
							ba[i] = pct ? (parseFloat(bs) / 100 * overlap) + "px" : (parseFloat(bs) / overlap * 100) + "%";
						}
					}
					bs = ba.join(" ");
				}
			}
			return this.parseComplex(t.style, bs, es, pt, plugin);
		}, formatter:_parsePosition});
		_registerComplexSpecialProp("backgroundSize", {defaultValue:"0 0", formatter:_parsePosition});
		_registerComplexSpecialProp("perspective", {defaultValue:"0px", prefix:true});
		_registerComplexSpecialProp("perspectiveOrigin", {defaultValue:"50% 50%", prefix:true});
		_registerComplexSpecialProp("transformStyle", {prefix:true});
		_registerComplexSpecialProp("backfaceVisibility", {prefix:true});
		_registerComplexSpecialProp("userSelect", {prefix:true});
		_registerComplexSpecialProp("margin", {parser:_getEdgeParser("marginTop,marginRight,marginBottom,marginLeft")});
		_registerComplexSpecialProp("padding", {parser:_getEdgeParser("paddingTop,paddingRight,paddingBottom,paddingLeft")});
		_registerComplexSpecialProp("clip", {defaultValue:"rect(0px,0px,0px,0px)", parser:function(t, e, p, cssp, pt, plugin){
			var b, cs, delim;
			if (_ieVers < 9) { //IE8 and earlier don't report a "clip" value in the currentStyle - instead, the values are split apart into clipTop, clipRight, clipBottom, and clipLeft. Also, in IE7 and earlier, the values inside rect() are space-delimited, not comma-delimited.
				cs = t.currentStyle;
				delim = _ieVers < 8 ? " " : ",";
				b = "rect(" + cs.clipTop + delim + cs.clipRight + delim + cs.clipBottom + delim + cs.clipLeft + ")";
				e = this.format(e).split(",").join(delim);
			} else {
				b = this.format(_getStyle(t, this.p, _cs, false, this.dflt));
				e = this.format(e);
			}
			return this.parseComplex(t.style, b, e, pt, plugin);
		}});
		_registerComplexSpecialProp("textShadow", {defaultValue:"0px 0px 0px #999", color:true, multi:true});
		_registerComplexSpecialProp("autoRound,strictUnits", {parser:function(t, e, p, cssp, pt) {return pt;}}); //just so that we can ignore these properties (not tween them)
		_registerComplexSpecialProp("border", {defaultValue:"0px solid #000", parser:function(t, e, p, cssp, pt, plugin) {
				return this.parseComplex(t.style, this.format(_getStyle(t, "borderTopWidth", _cs, false, "0px") + " " + _getStyle(t, "borderTopStyle", _cs, false, "solid") + " " + _getStyle(t, "borderTopColor", _cs, false, "#000")), this.format(e), pt, plugin);
			}, color:true, formatter:function(v) {
				var a = v.split(" ");
				return a[0] + " " + (a[1] || "solid") + " " + (v.match(_colorExp) || ["#000"])[0];
			}});
		_registerComplexSpecialProp("borderWidth", {parser:_getEdgeParser("borderTopWidth,borderRightWidth,borderBottomWidth,borderLeftWidth")}); //Firefox doesn't pick up on borderWidth set in style sheets (only inline).
		_registerComplexSpecialProp("float,cssFloat,styleFloat", {parser:function(t, e, p, cssp, pt, plugin) {
			var s = t.style,
				prop = ("cssFloat" in s) ? "cssFloat" : "styleFloat";
			return new CSSPropTween(s, prop, 0, 0, pt, -1, p, false, 0, s[prop], e);
		}});

		//opacity-related
		var _setIEOpacityRatio = function(v) {
				var t = this.t, //refers to the element's style property
					filters = t.filter || _getStyle(this.data, "filter") || "",
					val = (this.s + this.c * v) | 0,
					skip;
				if (val === 100) { //for older versions of IE that need to use a filter to apply opacity, we should remove the filter if opacity hits 1 in order to improve performance, but make sure there isn't a transform (matrix) or gradient in the filters.
					if (filters.indexOf("atrix(") === -1 && filters.indexOf("radient(") === -1 && filters.indexOf("oader(") === -1) {
						t.removeAttribute("filter");
						skip = (!_getStyle(this.data, "filter")); //if a class is applied that has an alpha filter, it will take effect (we don't want that), so re-apply our alpha filter in that case. We must first remove it and then check.
					} else {
						t.filter = filters.replace(_alphaFilterExp, "");
						skip = true;
					}
				}
				if (!skip) {
					if (this.xn1) {
						t.filter = filters = filters || ("alpha(opacity=" + val + ")"); //works around bug in IE7/8 that prevents changes to "visibility" from being applied properly if the filter is changed to a different alpha on the same frame.
					}
					if (filters.indexOf("pacity") === -1) { //only used if browser doesn't support the standard opacity style property (IE 7 and 8). We omit the "O" to avoid case-sensitivity issues
						if (val !== 0 || !this.xn1) { //bugs in IE7/8 won't render the filter properly if opacity is ADDED on the same frame/render as "visibility" changes (this.xn1 is 1 if this tween is an "autoAlpha" tween)
							t.filter = filters + " alpha(opacity=" + val + ")"; //we round the value because otherwise, bugs in IE7/8 can prevent "visibility" changes from being applied properly.
						}
					} else {
						t.filter = filters.replace(_opacityExp, "opacity=" + val);
					}
				}
			};
		_registerComplexSpecialProp("opacity,alpha,autoAlpha", {defaultValue:"1", parser:function(t, e, p, cssp, pt, plugin) {
			var b = parseFloat(_getStyle(t, "opacity", _cs, false, "1")),
				style = t.style,
				isAutoAlpha = (p === "autoAlpha");
			if (typeof(e) === "string" && e.charAt(1) === "=") {
				e = ((e.charAt(0) === "-") ? -1 : 1) * parseFloat(e.substr(2)) + b;
			}
			if (isAutoAlpha && b === 1 && _getStyle(t, "visibility", _cs) === "hidden" && e !== 0) { //if visibility is initially set to "hidden", we should interpret that as intent to make opacity 0 (a convenience)
				b = 0;
			}
			if (_supportsOpacity) {
				pt = new CSSPropTween(style, "opacity", b, e - b, pt);
			} else {
				pt = new CSSPropTween(style, "opacity", b * 100, (e - b) * 100, pt);
				pt.xn1 = isAutoAlpha ? 1 : 0; //we need to record whether or not this is an autoAlpha so that in the setRatio(), we know to duplicate the setting of the alpha in order to work around a bug in IE7 and IE8 that prevents changes to "visibility" from taking effect if the filter is changed to a different alpha(opacity) at the same time. Setting it to the SAME value first, then the new value works around the IE7/8 bug.
				style.zoom = 1; //helps correct an IE issue.
				pt.type = 2;
				pt.b = "alpha(opacity=" + pt.s + ")";
				pt.e = "alpha(opacity=" + (pt.s + pt.c) + ")";
				pt.data = t;
				pt.plugin = plugin;
				pt.setRatio = _setIEOpacityRatio;
			}
			if (isAutoAlpha) { //we have to create the "visibility" PropTween after the opacity one in the linked list so that they run in the order that works properly in IE8 and earlier
				pt = new CSSPropTween(style, "visibility", 0, 0, pt, -1, null, false, 0, ((b !== 0) ? "inherit" : "hidden"), ((e === 0) ? "hidden" : "inherit"));
				pt.xs0 = "inherit";
				cssp._overwriteProps.push(pt.n);
				cssp._overwriteProps.push(p);
			}
			return pt;
		}});


		var _removeProp = function(s, p) {
				if (p) {
					if (s.removeProperty) {
						if (p.substr(0,2) === "ms" || p.substr(0,6) === "webkit") { //Microsoft and some Webkit browsers don't conform to the standard of capitalizing the first prefix character, so we adjust so that when we prefix the caps with a dash, it's correct (otherwise it'd be "ms-transform" instead of "-ms-transform" for IE9, for example)
							p = "-" + p;
						}
						s.removeProperty(p.replace(_capsExp, "-$1").toLowerCase());
					} else { //note: old versions of IE use "removeAttribute()" instead of "removeProperty()"
						s.removeAttribute(p);
					}
				}
			},
			_setClassNameRatio = function(v) {
				this.t._gsClassPT = this;
				if (v === 1 || v === 0) {
					this.t.setAttribute("class", (v === 0) ? this.b : this.e);
					var mpt = this.data, //first MiniPropTween
						s = this.t.style;
					while (mpt) {
						if (!mpt.v) {
							_removeProp(s, mpt.p);
						} else {
							s[mpt.p] = mpt.v;
						}
						mpt = mpt._next;
					}
					if (v === 1 && this.t._gsClassPT === this) {
						this.t._gsClassPT = null;
					}
				} else if (this.t.getAttribute("class") !== this.e) {
					this.t.setAttribute("class", this.e);
				}
			};
		_registerComplexSpecialProp("className", {parser:function(t, e, p, cssp, pt, plugin, vars) {
			var b = t.getAttribute("class") || "", //don't use t.className because it doesn't work consistently on SVG elements; getAttribute("class") and setAttribute("class", value") is more reliable.
				cssText = t.style.cssText,
				difData, bs, cnpt, cnptLookup, mpt;
			pt = cssp._classNamePT = new CSSPropTween(t, p, 0, 0, pt, 2);
			pt.setRatio = _setClassNameRatio;
			pt.pr = -11;
			_hasPriority = true;
			pt.b = b;
			bs = _getAllStyles(t, _cs);
			//if there's a className tween already operating on the target, force it to its end so that the necessary inline styles are removed and the class name is applied before we determine the end state (we don't want inline styles interfering that were there just for class-specific values)
			cnpt = t._gsClassPT;
			if (cnpt) {
				cnptLookup = {};
				mpt = cnpt.data; //first MiniPropTween which stores the inline styles - we need to force these so that the inline styles don't contaminate things. Otherwise, there's a small chance that a tween could start and the inline values match the destination values and they never get cleaned.
				while (mpt) {
					cnptLookup[mpt.p] = 1;
					mpt = mpt._next;
				}
				cnpt.setRatio(1);
			}
			t._gsClassPT = pt;
			pt.e = (e.charAt(1) !== "=") ? e : b.replace(new RegExp("\\s*\\b" + e.substr(2) + "\\b"), "") + ((e.charAt(0) === "+") ? " " + e.substr(2) : "");
			t.setAttribute("class", pt.e);
			difData = _cssDif(t, bs, _getAllStyles(t), vars, cnptLookup);
			t.setAttribute("class", b);
			pt.data = difData.firstMPT;
			t.style.cssText = cssText; //we recorded cssText before we swapped classes and ran _getAllStyles() because in cases when a className tween is overwritten, we remove all the related tweening properties from that class change (otherwise class-specific stuff can't override properties we've directly set on the target's style object due to specificity).
			pt = pt.xfirst = cssp.parse(t, difData.difs, pt, plugin); //we record the CSSPropTween as the xfirst so that we can handle overwriting propertly (if "className" gets overwritten, we must kill all the properties associated with the className part of the tween, so we can loop through from xfirst to the pt itself)
			return pt;
		}});


		var _setClearPropsRatio = function(v) {
			if (v === 1 || v === 0) if (this.data._totalTime === this.data._totalDuration && this.data.data !== "isFromStart") { //this.data refers to the tween. Only clear at the END of the tween (remember, from() tweens make the ratio go from 1 to 0, so we can't just check that and if the tween is the zero-duration one that's created internally to render the starting values in a from() tween, ignore that because otherwise, for example, from(...{height:100, clearProps:"height", delay:1}) would wipe the height at the beginning of the tween and after 1 second, it'd kick back in).
				var s = this.t.style,
					transformParse = _specialProps.transform.parse,
					a, p, i, clearTransform, transform;
				if (this.e === "all") {
					s.cssText = "";
					clearTransform = true;
				} else {
					a = this.e.split(" ").join("").split(",");
					i = a.length;
					while (--i > -1) {
						p = a[i];
						if (_specialProps[p]) {
							if (_specialProps[p].parse === transformParse) {
								clearTransform = true;
							} else {
								p = (p === "transformOrigin") ? _transformOriginProp : _specialProps[p].p; //ensures that special properties use the proper browser-specific property name, like "scaleX" might be "-webkit-transform" or "boxShadow" might be "-moz-box-shadow"
							}
						}
						_removeProp(s, p);
					}
				}
				if (clearTransform) {
					_removeProp(s, _transformProp);
					transform = this.t._gsTransform;
					if (transform) {
						if (transform.svg) {
							this.t.removeAttribute("data-svg-origin");
						}
						delete this.t._gsTransform;
					}
				}

			}
		};
		_registerComplexSpecialProp("clearProps", {parser:function(t, e, p, cssp, pt) {
			pt = new CSSPropTween(t, p, 0, 0, pt, 2);
			pt.setRatio = _setClearPropsRatio;
			pt.e = e;
			pt.pr = -10;
			pt.data = cssp._tween;
			_hasPriority = true;
			return pt;
		}});

		p = "bezier,throwProps,physicsProps,physics2D".split(",");
		i = p.length;
		while (i--) {
			_registerPluginProp(p[i]);
		}








		p = CSSPlugin.prototype;
		p._firstPT = p._lastParsedTransform = p._transform = null;

		//gets called when the tween renders for the first time. This kicks everything off, recording start/end values, etc.
		p._onInitTween = function(target, vars, tween) {
			if (!target.nodeType) { //css is only for dom elements
				return false;
			}
			this._target = target;
			this._tween = tween;
			this._vars = vars;
			_autoRound = vars.autoRound;
			_hasPriority = false;
			_suffixMap = vars.suffixMap || CSSPlugin.suffixMap;
			_cs = _getComputedStyle(target, "");
			_overwriteProps = this._overwriteProps;
			var style = target.style,
				v, pt, pt2, first, last, next, zIndex, tpt, threeD;
			if (_reqSafariFix) if (style.zIndex === "") {
				v = _getStyle(target, "zIndex", _cs);
				if (v === "auto" || v === "") {
					//corrects a bug in [non-Android] Safari that prevents it from repainting elements in their new positions if they don't have a zIndex set. We also can't just apply this inside _parseTransform() because anything that's moved in any way (like using "left" or "top" instead of transforms like "x" and "y") can be affected, so it is best to ensure that anything that's tweening has a z-index. Setting "WebkitPerspective" to a non-zero value worked too except that on iOS Safari things would flicker randomly. Plus zIndex is less memory-intensive.
					this._addLazySet(style, "zIndex", 0);
				}
			}

			if (typeof(vars) === "string") {
				first = style.cssText;
				v = _getAllStyles(target, _cs);
				style.cssText = first + ";" + vars;
				v = _cssDif(target, v, _getAllStyles(target)).difs;
				if (!_supportsOpacity && _opacityValExp.test(vars)) {
					v.opacity = parseFloat( RegExp.$1 );
				}
				vars = v;
				style.cssText = first;
			}

			if (vars.className) { //className tweens will combine any differences they find in the css with the vars that are passed in, so {className:"myClass", scale:0.5, left:20} would work.
				this._firstPT = pt = _specialProps.className.parse(target, vars.className, "className", this, null, null, vars);
			} else {
				this._firstPT = pt = this.parse(target, vars, null);
			}

			if (this._transformType) {
				threeD = (this._transformType === 3);
				if (!_transformProp) {
					style.zoom = 1; //helps correct an IE issue.
				} else if (_isSafari) {
					_reqSafariFix = true;
					//if zIndex isn't set, iOS Safari doesn't repaint things correctly sometimes (seemingly at random).
					if (style.zIndex === "") {
						zIndex = _getStyle(target, "zIndex", _cs);
						if (zIndex === "auto" || zIndex === "") {
							this._addLazySet(style, "zIndex", 0);
						}
					}
					//Setting WebkitBackfaceVisibility corrects 3 bugs:
					// 1) [non-Android] Safari skips rendering changes to "top" and "left" that are made on the same frame/render as a transform update.
					// 2) iOS Safari sometimes neglects to repaint elements in their new positions. Setting "WebkitPerspective" to a non-zero value worked too except that on iOS Safari things would flicker randomly.
					// 3) Safari sometimes displayed odd artifacts when tweening the transform (or WebkitTransform) property, like ghosts of the edges of the element remained. Definitely a browser bug.
					//Note: we allow the user to override the auto-setting by defining WebkitBackfaceVisibility in the vars of the tween.
					if (_isSafariLT6) {
						this._addLazySet(style, "WebkitBackfaceVisibility", this._vars.WebkitBackfaceVisibility || (threeD ? "visible" : "hidden"));
					}
				}
				pt2 = pt;
				while (pt2 && pt2._next) {
					pt2 = pt2._next;
				}
				tpt = new CSSPropTween(target, "transform", 0, 0, null, 2);
				this._linkCSSP(tpt, null, pt2);
				tpt.setRatio = _transformProp ? _setTransformRatio : _setIETransformRatio;
				tpt.data = this._transform || _getTransform(target, _cs, true);
				tpt.tween = tween;
				tpt.pr = -1; //ensures that the transforms get applied after the components are updated.
				_overwriteProps.pop(); //we don't want to force the overwrite of all "transform" tweens of the target - we only care about individual transform properties like scaleX, rotation, etc. The CSSPropTween constructor automatically adds the property to _overwriteProps which is why we need to pop() here.
			}

			if (_hasPriority) {
				//reorders the linked list in order of pr (priority)
				while (pt) {
					next = pt._next;
					pt2 = first;
					while (pt2 && pt2.pr > pt.pr) {
						pt2 = pt2._next;
					}
					if ((pt._prev = pt2 ? pt2._prev : last)) {
						pt._prev._next = pt;
					} else {
						first = pt;
					}
					if ((pt._next = pt2)) {
						pt2._prev = pt;
					} else {
						last = pt;
					}
					pt = next;
				}
				this._firstPT = first;
			}
			return true;
		};


		p.parse = function(target, vars, pt, plugin) {
			var style = target.style,
				p, sp, bn, en, bs, es, bsfx, esfx, isStr, rel;
			for (p in vars) {
				es = vars[p]; //ending value string
				sp = _specialProps[p]; //SpecialProp lookup.
				if (sp) {
					pt = sp.parse(target, es, p, this, pt, plugin, vars);

				} else {
					bs = _getStyle(target, p, _cs) + "";
					isStr = (typeof(es) === "string");
					if (p === "color" || p === "fill" || p === "stroke" || p.indexOf("Color") !== -1 || (isStr && _rgbhslExp.test(es))) { //Opera uses background: to define color sometimes in addition to backgroundColor:
						if (!isStr) {
							es = _parseColor(es);
							es = ((es.length > 3) ? "rgba(" : "rgb(") + es.join(",") + ")";
						}
						pt = _parseComplex(style, p, bs, es, true, "transparent", pt, 0, plugin);

					} else if (isStr && (es.indexOf(" ") !== -1 || es.indexOf(",") !== -1)) {
						pt = _parseComplex(style, p, bs, es, true, null, pt, 0, plugin);

					} else {
						bn = parseFloat(bs);
						bsfx = (bn || bn === 0) ? bs.substr((bn + "").length) : ""; //remember, bs could be non-numeric like "normal" for fontWeight, so we should default to a blank suffix in that case.

						if (bs === "" || bs === "auto") {
							if (p === "width" || p === "height") {
								bn = _getDimension(target, p, _cs);
								bsfx = "px";
							} else if (p === "left" || p === "top") {
								bn = _calculateOffset(target, p, _cs);
								bsfx = "px";
							} else {
								bn = (p !== "opacity") ? 0 : 1;
								bsfx = "";
							}
						}

						rel = (isStr && es.charAt(1) === "=");
						if (rel) {
							en = parseInt(es.charAt(0) + "1", 10);
							es = es.substr(2);
							en *= parseFloat(es);
							esfx = es.replace(_suffixExp, "");
						} else {
							en = parseFloat(es);
							esfx = isStr ? es.replace(_suffixExp, "") : "";
						}

						if (esfx === "") {
							esfx = (p in _suffixMap) ? _suffixMap[p] : bsfx; //populate the end suffix, prioritizing the map, then if none is found, use the beginning suffix.
						}

						es = (en || en === 0) ? (rel ? en + bn : en) + esfx : vars[p]; //ensures that any += or -= prefixes are taken care of. Record the end value before normalizing the suffix because we always want to end the tween on exactly what they intended even if it doesn't match the beginning value's suffix.

						//if the beginning/ending suffixes don't match, normalize them...
						if (bsfx !== esfx) if (esfx !== "") if (en || en === 0) if (bn) { //note: if the beginning value (bn) is 0, we don't need to convert units!
							bn = _convertToPixels(target, p, bn, bsfx);
							if (esfx === "%") {
								bn /= _convertToPixels(target, p, 100, "%") / 100;
								if (vars.strictUnits !== true) { //some browsers report only "px" values instead of allowing "%" with getComputedStyle(), so we assume that if we're tweening to a %, we should start there too unless strictUnits:true is defined. This approach is particularly useful for responsive designs that use from() tweens.
									bs = bn + "%";
								}

							} else if (esfx === "em") {
								bn /= _convertToPixels(target, p, 1, "em");

							//otherwise convert to pixels.
							} else if (esfx !== "px") {
								en = _convertToPixels(target, p, en, esfx);
								esfx = "px"; //we don't use bsfx after this, so we don't need to set it to px too.
							}
							if (rel) if (en || en === 0) {
								es = (en + bn) + esfx; //the changes we made affect relative calculations, so adjust the end value here.
							}
						}

						if (rel) {
							en += bn;
						}

						if ((bn || bn === 0) && (en || en === 0)) { //faster than isNaN(). Also, previously we required en !== bn but that doesn't really gain much performance and it prevents _parseToProxy() from working properly if beginning and ending values match but need to get tweened by an external plugin anyway. For example, a bezier tween where the target starts at left:0 and has these points: [{left:50},{left:0}] wouldn't work properly because when parsing the last point, it'd match the first (current) one and a non-tweening CSSPropTween would be recorded when we actually need a normal tween (type:0) so that things get updated during the tween properly.
							pt = new CSSPropTween(style, p, bn, en - bn, pt, 0, p, (_autoRound !== false && (esfx === "px" || p === "zIndex")), 0, bs, es);
							pt.xs0 = esfx;
							//DEBUG: _log("tween "+p+" from "+pt.b+" ("+bn+esfx+") to "+pt.e+" with suffix: "+pt.xs0);
						} else if (style[p] === undefined || !es && (es + "" === "NaN" || es == null)) {
							_log("invalid " + p + " tween value: " + vars[p]);
						} else {
							pt = new CSSPropTween(style, p, en || bn || 0, 0, pt, -1, p, false, 0, bs, es);
							pt.xs0 = (es === "none" && (p === "display" || p.indexOf("Style") !== -1)) ? bs : es; //intermediate value should typically be set immediately (end value) except for "display" or things like borderTopStyle, borderBottomStyle, etc. which should use the beginning value during the tween.
							//DEBUG: _log("non-tweening value "+p+": "+pt.xs0);
						}
					}
				}
				if (plugin) if (pt && !pt.plugin) {
					pt.plugin = plugin;
				}
			}
			return pt;
		};


		//gets called every time the tween updates, passing the new ratio (typically a value between 0 and 1, but not always (for example, if an Elastic.easeOut is used, the value can jump above 1 mid-tween). It will always start and 0 and end at 1.
		p.setRatio = function(v) {
			var pt = this._firstPT,
				min = 0.000001,
				val, str, i;
			//at the end of the tween, we set the values to exactly what we received in order to make sure non-tweening values (like "position" or "float" or whatever) are set and so that if the beginning/ending suffixes (units) didn't match and we normalized to px, the value that the user passed in is used here. We check to see if the tween is at its beginning in case it's a from() tween in which case the ratio will actually go from 1 to 0 over the course of the tween (backwards).
			if (v === 1 && (this._tween._time === this._tween._duration || this._tween._time === 0)) {
				while (pt) {
					if (pt.type !== 2) {
						pt.t[pt.p] = pt.e;
					} else {
						pt.setRatio(v);
					}
					pt = pt._next;
				}

			} else if (v || !(this._tween._time === this._tween._duration || this._tween._time === 0) || this._tween._rawPrevTime === -0.000001) {
				while (pt) {
					val = pt.c * v + pt.s;
					if (pt.r) {
						val = Math.round(val);
					} else if (val < min) if (val > -min) {
						val = 0;
					}
					if (!pt.type) {
						pt.t[pt.p] = val + pt.xs0;
					} else if (pt.type === 1) { //complex value (one that typically has multiple numbers inside a string, like "rect(5px,10px,20px,25px)"
						i = pt.l;
						if (i === 2) {
							pt.t[pt.p] = pt.xs0 + val + pt.xs1 + pt.xn1 + pt.xs2;
						} else if (i === 3) {
							pt.t[pt.p] = pt.xs0 + val + pt.xs1 + pt.xn1 + pt.xs2 + pt.xn2 + pt.xs3;
						} else if (i === 4) {
							pt.t[pt.p] = pt.xs0 + val + pt.xs1 + pt.xn1 + pt.xs2 + pt.xn2 + pt.xs3 + pt.xn3 + pt.xs4;
						} else if (i === 5) {
							pt.t[pt.p] = pt.xs0 + val + pt.xs1 + pt.xn1 + pt.xs2 + pt.xn2 + pt.xs3 + pt.xn3 + pt.xs4 + pt.xn4 + pt.xs5;
						} else {
							str = pt.xs0 + val + pt.xs1;
							for (i = 1; i < pt.l; i++) {
								str += pt["xn"+i] + pt["xs"+(i+1)];
							}
							pt.t[pt.p] = str;
						}

					} else if (pt.type === -1) { //non-tweening value
						pt.t[pt.p] = pt.xs0;

					} else if (pt.setRatio) { //custom setRatio() for things like SpecialProps, external plugins, etc.
						pt.setRatio(v);
					}
					pt = pt._next;
				}

			//if the tween is reversed all the way back to the beginning, we need to restore the original values which may have different units (like % instead of px or em or whatever).
			} else {
				while (pt) {
					if (pt.type !== 2) {
						pt.t[pt.p] = pt.b;
					} else {
						pt.setRatio(v);
					}
					pt = pt._next;
				}
			}
		};

		/**
		 * @private
		 * Forces rendering of the target's transforms (rotation, scale, etc.) whenever the CSSPlugin's setRatio() is called.
		 * Basically, this tells the CSSPlugin to create a CSSPropTween (type 2) after instantiation that runs last in the linked
		 * list and calls the appropriate (3D or 2D) rendering function. We separate this into its own method so that we can call
		 * it from other plugins like BezierPlugin if, for example, it needs to apply an autoRotation and this CSSPlugin
		 * doesn't have any transform-related properties of its own. You can call this method as many times as you
		 * want and it won't create duplicate CSSPropTweens.
		 *
		 * @param {boolean} threeD if true, it should apply 3D tweens (otherwise, just 2D ones are fine and typically faster)
		 */
		p._enableTransforms = function(threeD) {
			this._transform = this._transform || _getTransform(this._target, _cs, true); //ensures that the element has a _gsTransform property with the appropriate values.
			this._transformType = (!(this._transform.svg && _useSVGTransformAttr) && (threeD || this._transformType === 3)) ? 3 : 2;
		};

		var lazySet = function(v) {
			this.t[this.p] = this.e;
			this.data._linkCSSP(this, this._next, null, true); //we purposefully keep this._next even though it'd make sense to null it, but this is a performance optimization, as this happens during the while (pt) {} loop in setRatio() at the bottom of which it sets pt = pt._next, so if we null it, the linked list will be broken in that loop.
		};
		/** @private Gives us a way to set a value on the first render (and only the first render). **/
		p._addLazySet = function(t, p, v) {
			var pt = this._firstPT = new CSSPropTween(t, p, 0, 0, this._firstPT, 2);
			pt.e = v;
			pt.setRatio = lazySet;
			pt.data = this;
		};

		/** @private **/
		p._linkCSSP = function(pt, next, prev, remove) {
			if (pt) {
				if (next) {
					next._prev = pt;
				}
				if (pt._next) {
					pt._next._prev = pt._prev;
				}
				if (pt._prev) {
					pt._prev._next = pt._next;
				} else if (this._firstPT === pt) {
					this._firstPT = pt._next;
					remove = true; //just to prevent resetting this._firstPT 5 lines down in case pt._next is null. (optimized for speed)
				}
				if (prev) {
					prev._next = pt;
				} else if (!remove && this._firstPT === null) {
					this._firstPT = pt;
				}
				pt._next = next;
				pt._prev = prev;
			}
			return pt;
		};

		//we need to make sure that if alpha or autoAlpha is killed, opacity is too. And autoAlpha affects the "visibility" property.
		p._kill = function(lookup) {
			var copy = lookup,
				pt, p, xfirst;
			if (lookup.autoAlpha || lookup.alpha) {
				copy = {};
				for (p in lookup) { //copy the lookup so that we're not changing the original which may be passed elsewhere.
					copy[p] = lookup[p];
				}
				copy.opacity = 1;
				if (copy.autoAlpha) {
					copy.visibility = 1;
				}
			}
			if (lookup.className && (pt = this._classNamePT)) { //for className tweens, we need to kill any associated CSSPropTweens too; a linked list starts at the className's "xfirst".
				xfirst = pt.xfirst;
				if (xfirst && xfirst._prev) {
					this._linkCSSP(xfirst._prev, pt._next, xfirst._prev._prev); //break off the prev
				} else if (xfirst === this._firstPT) {
					this._firstPT = pt._next;
				}
				if (pt._next) {
					this._linkCSSP(pt._next, pt._next._next, xfirst._prev);
				}
				this._classNamePT = null;
			}
			return TweenPlugin.prototype._kill.call(this, copy);
		};



		//used by cascadeTo() for gathering all the style properties of each child element into an array for comparison.
		var _getChildStyles = function(e, props, targets) {
				var children, i, child, type;
				if (e.slice) {
					i = e.length;
					while (--i > -1) {
						_getChildStyles(e[i], props, targets);
					}
					return;
				}
				children = e.childNodes;
				i = children.length;
				while (--i > -1) {
					child = children[i];
					type = child.type;
					if (child.style) {
						props.push(_getAllStyles(child));
						if (targets) {
							targets.push(child);
						}
					}
					if ((type === 1 || type === 9 || type === 11) && child.childNodes.length) {
						_getChildStyles(child, props, targets);
					}
				}
			};

		/**
		 * Typically only useful for className tweens that may affect child elements, this method creates a TweenLite
		 * and then compares the style properties of all the target's child elements at the tween's start and end, and
		 * if any are different, it also creates tweens for those and returns an array containing ALL of the resulting
		 * tweens (so that you can easily add() them to a TimelineLite, for example). The reason this functionality is
		 * wrapped into a separate static method of CSSPlugin instead of being integrated into all regular className tweens
		 * is because it creates entirely new tweens that may have completely different targets than the original tween,
		 * so if they were all lumped into the original tween instance, it would be inconsistent with the rest of the API
		 * and it would create other problems. For example:
		 *  - If I create a tween of elementA, that tween instance may suddenly change its target to include 50 other elements (unintuitive if I specifically defined the target I wanted)
		 *  - We can't just create new independent tweens because otherwise, what happens if the original/parent tween is reversed or pause or dropped into a TimelineLite for tight control? You'd expect that tween's behavior to affect all the others.
		 *  - Analyzing every style property of every child before and after the tween is an expensive operation when there are many children, so this behavior shouldn't be imposed on all className tweens by default, especially since it's probably rare that this extra functionality is needed.
		 *
		 * @param {Object} target object to be tweened
		 * @param {number} Duration in seconds (or frames for frames-based tweens)
		 * @param {Object} Object containing the end values, like {className:"newClass", ease:Linear.easeNone}
		 * @return {Array} An array of TweenLite instances
		 */
		CSSPlugin.cascadeTo = function(target, duration, vars) {
			var tween = TweenLite.to(target, duration, vars),
				results = [tween],
				b = [],
				e = [],
				targets = [],
				_reservedProps = TweenLite._internals.reservedProps,
				i, difs, p, from;
			target = tween._targets || tween.target;
			_getChildStyles(target, b, targets);
			tween.render(duration, true, true);
			_getChildStyles(target, e);
			tween.render(0, true, true);
			tween._enabled(true);
			i = targets.length;
			while (--i > -1) {
				difs = _cssDif(targets[i], b[i], e[i]);
				if (difs.firstMPT) {
					difs = difs.difs;
					for (p in vars) {
						if (_reservedProps[p]) {
							difs[p] = vars[p];
						}
					}
					from = {};
					for (p in difs) {
						from[p] = b[i][p];
					}
					results.push(TweenLite.fromTo(targets[i], duration, from, difs));
				}
			}
			return results;
		};

		TweenPlugin.activate([CSSPlugin]);
		return CSSPlugin;

	}, true);
	
}); if (_gsScope._gsDefine) { _gsScope._gsQueue.pop()(); }

//export to AMD/RequireJS and CommonJS/Node (precursor to full modular build system coming at a later date)
(function(name) {
	"use strict";
	var getGlobal = function() {
		return (_gsScope.GreenSockGlobals || _gsScope)[name];
	};
	if (typeof(define) === "function" && define.amd) { //AMD
		define(["TweenLite"], getGlobal);
	} else if (typeof(module) !== "undefined" && module.exports) { //node
		require("../TweenLite.js");
		module.exports = getGlobal();
	}
}("CSSPlugin"));

/*!
 * VERSION: 1.7.5
 * DATE: 2015-02-26
 * UPDATES AND DOCS AT: http://greensock.com
 *
 * @license Copyright (c) 2008-2015, GreenSock. All rights reserved.
 * This work is subject to the terms at http://greensock.com/standard-license or for
 * Club GreenSock members, the software agreement that was issued with your membership.
 * 
 * @author: Jack Doyle, jack@greensock.com
 **/
var _gsScope = (typeof(module) !== "undefined" && module.exports && typeof(global) !== "undefined") ? global : this || window; //helps ensure compatibility with AMD/RequireJS and CommonJS/Node
(_gsScope._gsQueue || (_gsScope._gsQueue = [])).push( function() {

	"use strict";

	var _doc = document.documentElement,
		_window = window,
		_max = function(element, axis) {
			var dim = (axis === "x") ? "Width" : "Height",
				scroll = "scroll" + dim,
				client = "client" + dim,
				body = document.body;
			return (element === _window || element === _doc || element === body) ? Math.max(_doc[scroll], body[scroll]) - (_window["inner" + dim] || _doc[client] || body[client]) : element[scroll] - element["offset" + dim];
		},

		ScrollToPlugin = _gsScope._gsDefine.plugin({
			propName: "scrollTo",
			API: 2,
			version:"1.7.5",

			//called when the tween renders for the first time. This is where initial values should be recorded and any setup routines should run.
			init: function(target, value, tween) {
				this._wdw = (target === _window);
				this._target = target;
				this._tween = tween;
				if (typeof(value) !== "object") {
					value = {y:value}; //if we don't receive an object as the parameter, assume the user intends "y".
				}
				this.vars = value;
				this._autoKill = (value.autoKill !== false);
				this.x = this.xPrev = this.getX();
				this.y = this.yPrev = this.getY();
				if (value.x != null) {
					this._addTween(this, "x", this.x, (value.x === "max") ? _max(target, "x") : value.x, "scrollTo_x", true);
					this._overwriteProps.push("scrollTo_x");
				} else {
					this.skipX = true;
				}
				if (value.y != null) {
					this._addTween(this, "y", this.y, (value.y === "max") ? _max(target, "y") : value.y, "scrollTo_y", true);
					this._overwriteProps.push("scrollTo_y");
				} else {
					this.skipY = true;
				}
				return true;
			},

			//called each time the values should be updated, and the ratio gets passed as the only parameter (typically it's a value between 0 and 1, but it can exceed those when using an ease like Elastic.easeOut or Back.easeOut, etc.)
			set: function(v) {
				this._super.setRatio.call(this, v);

				var x = (this._wdw || !this.skipX) ? this.getX() : this.xPrev,
					y = (this._wdw || !this.skipY) ? this.getY() : this.yPrev,
					yDif = y - this.yPrev,
					xDif = x - this.xPrev;

				if (this._autoKill) {
					//note: iOS has a bug that throws off the scroll by several pixels, so we need to check if it's within 7 pixels of the previous one that we set instead of just looking for an exact match.
					if (!this.skipX && (xDif > 7 || xDif < -7) && x < _max(this._target, "x")) {
						this.skipX = true; //if the user scrolls separately, we should stop tweening!
					}
					if (!this.skipY && (yDif > 7 || yDif < -7) && y < _max(this._target, "y")) {
						this.skipY = true; //if the user scrolls separately, we should stop tweening!
					}
					if (this.skipX && this.skipY) {
						this._tween.kill();
						if (this.vars.onAutoKill) {
							this.vars.onAutoKill.apply(this.vars.onAutoKillScope || this._tween, this.vars.onAutoKillParams || []);
						}
					}
				}
				if (this._wdw) {
					_window.scrollTo((!this.skipX) ? this.x : x, (!this.skipY) ? this.y : y);
				} else {
					if (!this.skipY) {
						this._target.scrollTop = this.y;
					}
					if (!this.skipX) {
						this._target.scrollLeft = this.x;
					}
				}
				this.xPrev = this.x;
				this.yPrev = this.y;
			}

		}),
		p = ScrollToPlugin.prototype;

	ScrollToPlugin.max = _max;

	p.getX = function() {
		return (!this._wdw) ? this._target.scrollLeft : (_window.pageXOffset != null) ? _window.pageXOffset : (_doc.scrollLeft != null) ? _doc.scrollLeft : document.body.scrollLeft;
	};

	p.getY = function() {
		return (!this._wdw) ? this._target.scrollTop : (_window.pageYOffset != null) ? _window.pageYOffset : (_doc.scrollTop != null) ? _doc.scrollTop : document.body.scrollTop;
	};

	p._kill = function(lookup) {
		if (lookup.scrollTo_x) {
			this.skipX = true;
		}
		if (lookup.scrollTo_y) {
			this.skipY = true;
		}
		return this._super._kill.call(this, lookup);
	};

}); if (_gsScope._gsDefine) { _gsScope._gsQueue.pop()(); }
/*!
 * VERSION: beta 1.15.2
 * DATE: 2015-01-27
 * UPDATES AND DOCS AT: http://greensock.com
 *
 * @license Copyright (c) 2008-2015, GreenSock. All rights reserved.
 * This work is subject to the terms at http://greensock.com/standard-license or for
 * Club GreenSock members, the software agreement that was issued with your membership.
 * 
 * @author: Jack Doyle, jack@greensock.com
 **/
var _gsScope = (typeof(module) !== "undefined" && module.exports && typeof(global) !== "undefined") ? global : this || window; //helps ensure compatibility with AMD/RequireJS and CommonJS/Node
(_gsScope._gsQueue || (_gsScope._gsQueue = [])).push( function() {

	"use strict";

	_gsScope._gsDefine("easing.Back", ["easing.Ease"], function(Ease) {
		
		var w = (_gsScope.GreenSockGlobals || _gsScope),
			gs = w.com.greensock,
			_2PI = Math.PI * 2,
			_HALF_PI = Math.PI / 2,
			_class = gs._class,
			_create = function(n, f) {
				var C = _class("easing." + n, function(){}, true),
					p = C.prototype = new Ease();
				p.constructor = C;
				p.getRatio = f;
				return C;
			},
			_easeReg = Ease.register || function(){}, //put an empty function in place just as a safety measure in case someone loads an OLD version of TweenLite.js where Ease.register doesn't exist.
			_wrap = function(name, EaseOut, EaseIn, EaseInOut, aliases) {
				var C = _class("easing."+name, {
					easeOut:new EaseOut(),
					easeIn:new EaseIn(),
					easeInOut:new EaseInOut()
				}, true);
				_easeReg(C, name);
				return C;
			},
			EasePoint = function(time, value, next) {
				this.t = time;
				this.v = value;
				if (next) {
					this.next = next;
					next.prev = this;
					this.c = next.v - value;
					this.gap = next.t - time;
				}
			},

			//Back
			_createBack = function(n, f) {
				var C = _class("easing." + n, function(overshoot) {
						this._p1 = (overshoot || overshoot === 0) ? overshoot : 1.70158;
						this._p2 = this._p1 * 1.525;
					}, true), 
					p = C.prototype = new Ease();
				p.constructor = C;
				p.getRatio = f;
				p.config = function(overshoot) {
					return new C(overshoot);
				};
				return C;
			},

			Back = _wrap("Back",
				_createBack("BackOut", function(p) {
					return ((p = p - 1) * p * ((this._p1 + 1) * p + this._p1) + 1);
				}),
				_createBack("BackIn", function(p) {
					return p * p * ((this._p1 + 1) * p - this._p1);
				}),
				_createBack("BackInOut", function(p) {
					return ((p *= 2) < 1) ? 0.5 * p * p * ((this._p2 + 1) * p - this._p2) : 0.5 * ((p -= 2) * p * ((this._p2 + 1) * p + this._p2) + 2);
				})
			),


			//SlowMo
			SlowMo = _class("easing.SlowMo", function(linearRatio, power, yoyoMode) {
				power = (power || power === 0) ? power : 0.7;
				if (linearRatio == null) {
					linearRatio = 0.7;
				} else if (linearRatio > 1) {
					linearRatio = 1;
				}
				this._p = (linearRatio !== 1) ? power : 0;
				this._p1 = (1 - linearRatio) / 2;
				this._p2 = linearRatio;
				this._p3 = this._p1 + this._p2;
				this._calcEnd = (yoyoMode === true);
			}, true),
			p = SlowMo.prototype = new Ease(),
			SteppedEase, RoughEase, _createElastic;
			
		p.constructor = SlowMo;
		p.getRatio = function(p) {
			var r = p + (0.5 - p) * this._p;
			if (p < this._p1) {
				return this._calcEnd ? 1 - ((p = 1 - (p / this._p1)) * p) : r - ((p = 1 - (p / this._p1)) * p * p * p * r);
			} else if (p > this._p3) {
				return this._calcEnd ? 1 - (p = (p - this._p3) / this._p1) * p : r + ((p - r) * (p = (p - this._p3) / this._p1) * p * p * p);
			}
			return this._calcEnd ? 1 : r;
		};
		SlowMo.ease = new SlowMo(0.7, 0.7);
		
		p.config = SlowMo.config = function(linearRatio, power, yoyoMode) {
			return new SlowMo(linearRatio, power, yoyoMode);
		};


		//SteppedEase
		SteppedEase = _class("easing.SteppedEase", function(steps) {
				steps = steps || 1;
				this._p1 = 1 / steps;
				this._p2 = steps + 1;
			}, true);
		p = SteppedEase.prototype = new Ease();	
		p.constructor = SteppedEase;
		p.getRatio = function(p) {
			if (p < 0) {
				p = 0;
			} else if (p >= 1) {
				p = 0.999999999;
			}
			return ((this._p2 * p) >> 0) * this._p1;
		};
		p.config = SteppedEase.config = function(steps) {
			return new SteppedEase(steps);
		};


		//RoughEase
		RoughEase = _class("easing.RoughEase", function(vars) {
			vars = vars || {};
			var taper = vars.taper || "none",
				a = [],
				cnt = 0,
				points = (vars.points || 20) | 0,
				i = points,
				randomize = (vars.randomize !== false),
				clamp = (vars.clamp === true),
				template = (vars.template instanceof Ease) ? vars.template : null,
				strength = (typeof(vars.strength) === "number") ? vars.strength * 0.4 : 0.4,
				x, y, bump, invX, obj, pnt;
			while (--i > -1) {
				x = randomize ? Math.random() : (1 / points) * i;
				y = template ? template.getRatio(x) : x;
				if (taper === "none") {
					bump = strength;
				} else if (taper === "out") {
					invX = 1 - x;
					bump = invX * invX * strength;
				} else if (taper === "in") {
					bump = x * x * strength;
				} else if (x < 0.5) {  //"both" (start)
					invX = x * 2;
					bump = invX * invX * 0.5 * strength;
				} else {				//"both" (end)
					invX = (1 - x) * 2;
					bump = invX * invX * 0.5 * strength;
				}
				if (randomize) {
					y += (Math.random() * bump) - (bump * 0.5);
				} else if (i % 2) {
					y += bump * 0.5;
				} else {
					y -= bump * 0.5;
				}
				if (clamp) {
					if (y > 1) {
						y = 1;
					} else if (y < 0) {
						y = 0;
					}
				}
				a[cnt++] = {x:x, y:y};
			}
			a.sort(function(a, b) {
				return a.x - b.x;
			});

			pnt = new EasePoint(1, 1, null);
			i = points;
			while (--i > -1) {
				obj = a[i];
				pnt = new EasePoint(obj.x, obj.y, pnt);
			}

			this._prev = new EasePoint(0, 0, (pnt.t !== 0) ? pnt : pnt.next);
		}, true);
		p = RoughEase.prototype = new Ease();
		p.constructor = RoughEase;
		p.getRatio = function(p) {
			var pnt = this._prev;
			if (p > pnt.t) {
				while (pnt.next && p >= pnt.t) {
					pnt = pnt.next;
				}
				pnt = pnt.prev;
			} else {
				while (pnt.prev && p <= pnt.t) {
					pnt = pnt.prev;
				}
			}
			this._prev = pnt;
			return (pnt.v + ((p - pnt.t) / pnt.gap) * pnt.c);
		};
		p.config = function(vars) {
			return new RoughEase(vars);
		};
		RoughEase.ease = new RoughEase();


		//Bounce
		_wrap("Bounce",
			_create("BounceOut", function(p) {
				if (p < 1 / 2.75) {
					return 7.5625 * p * p;
				} else if (p < 2 / 2.75) {
					return 7.5625 * (p -= 1.5 / 2.75) * p + 0.75;
				} else if (p < 2.5 / 2.75) {
					return 7.5625 * (p -= 2.25 / 2.75) * p + 0.9375;
				}
				return 7.5625 * (p -= 2.625 / 2.75) * p + 0.984375;
			}),
			_create("BounceIn", function(p) {
				if ((p = 1 - p) < 1 / 2.75) {
					return 1 - (7.5625 * p * p);
				} else if (p < 2 / 2.75) {
					return 1 - (7.5625 * (p -= 1.5 / 2.75) * p + 0.75);
				} else if (p < 2.5 / 2.75) {
					return 1 - (7.5625 * (p -= 2.25 / 2.75) * p + 0.9375);
				}
				return 1 - (7.5625 * (p -= 2.625 / 2.75) * p + 0.984375);
			}),
			_create("BounceInOut", function(p) {
				var invert = (p < 0.5);
				if (invert) {
					p = 1 - (p * 2);
				} else {
					p = (p * 2) - 1;
				}
				if (p < 1 / 2.75) {
					p = 7.5625 * p * p;
				} else if (p < 2 / 2.75) {
					p = 7.5625 * (p -= 1.5 / 2.75) * p + 0.75;
				} else if (p < 2.5 / 2.75) {
					p = 7.5625 * (p -= 2.25 / 2.75) * p + 0.9375;
				} else {
					p = 7.5625 * (p -= 2.625 / 2.75) * p + 0.984375;
				}
				return invert ? (1 - p) * 0.5 : p * 0.5 + 0.5;
			})
		);


		//CIRC
		_wrap("Circ",
			_create("CircOut", function(p) {
				return Math.sqrt(1 - (p = p - 1) * p);
			}),
			_create("CircIn", function(p) {
				return -(Math.sqrt(1 - (p * p)) - 1);
			}),
			_create("CircInOut", function(p) {
				return ((p*=2) < 1) ? -0.5 * (Math.sqrt(1 - p * p) - 1) : 0.5 * (Math.sqrt(1 - (p -= 2) * p) + 1);
			})
		);


		//Elastic
		_createElastic = function(n, f, def) {
			var C = _class("easing." + n, function(amplitude, period) {
					this._p1 = (amplitude >= 1) ? amplitude : 1; //note: if amplitude is < 1, we simply adjust the period for a more natural feel. Otherwise the math doesn't work right and the curve starts at 1.
					this._p2 = (period || def) / (amplitude < 1 ? amplitude : 1);
					this._p3 = this._p2 / _2PI * (Math.asin(1 / this._p1) || 0);
					this._p2 = _2PI / this._p2; //precalculate to optimize
				}, true),
				p = C.prototype = new Ease();
			p.constructor = C;
			p.getRatio = f;
			p.config = function(amplitude, period) {
				return new C(amplitude, period);
			};
			return C;
		};
		_wrap("Elastic",
			_createElastic("ElasticOut", function(p) {
				return this._p1 * Math.pow(2, -10 * p) * Math.sin( (p - this._p3) * this._p2 ) + 1;
			}, 0.3),
			_createElastic("ElasticIn", function(p) {
				return -(this._p1 * Math.pow(2, 10 * (p -= 1)) * Math.sin( (p - this._p3) * this._p2 ));
			}, 0.3),
			_createElastic("ElasticInOut", function(p) {
				return ((p *= 2) < 1) ? -0.5 * (this._p1 * Math.pow(2, 10 * (p -= 1)) * Math.sin( (p - this._p3) * this._p2)) : this._p1 * Math.pow(2, -10 *(p -= 1)) * Math.sin( (p - this._p3) * this._p2 ) * 0.5 + 1;
			}, 0.45)
		);


		//Expo
		_wrap("Expo",
			_create("ExpoOut", function(p) {
				return 1 - Math.pow(2, -10 * p);
			}),
			_create("ExpoIn", function(p) {
				return Math.pow(2, 10 * (p - 1)) - 0.001;
			}),
			_create("ExpoInOut", function(p) {
				return ((p *= 2) < 1) ? 0.5 * Math.pow(2, 10 * (p - 1)) : 0.5 * (2 - Math.pow(2, -10 * (p - 1)));
			})
		);


		//Sine
		_wrap("Sine",
			_create("SineOut", function(p) {
				return Math.sin(p * _HALF_PI);
			}),
			_create("SineIn", function(p) {
				return -Math.cos(p * _HALF_PI) + 1;
			}),
			_create("SineInOut", function(p) {
				return -0.5 * (Math.cos(Math.PI * p) - 1);
			})
		);

		_class("easing.EaseLookup", {
				find:function(s) {
					return Ease.map[s];
				}
			}, true);

		//register the non-standard eases
		_easeReg(w.SlowMo, "SlowMo", "ease,");
		_easeReg(RoughEase, "RoughEase", "ease,");
		_easeReg(SteppedEase, "SteppedEase", "ease,");
		
		return Back;
		
	}, true);

}); if (_gsScope._gsDefine) { _gsScope._gsQueue.pop()(); }
/*!
 * VERSION: 1.16.1
 * DATE: 2015-03-13
 * UPDATES AND DOCS AT: http://greensock.com
 *
 * @license Copyright (c) 2008-2015, GreenSock. All rights reserved.
 * This work is subject to the terms at http://greensock.com/standard-license or for
 * Club GreenSock members, the software agreement that was issued with your membership.
 * 
 * @author: Jack Doyle, jack@greensock.com
 */
(function(window, moduleName) {

		"use strict";
		var _globals = window.GreenSockGlobals = window.GreenSockGlobals || window;
		if (_globals.TweenLite) {
			return; //in case the core set of classes is already loaded, don't instantiate twice.
		}
		var _namespace = function(ns) {
				var a = ns.split("."),
					p = _globals, i;
				for (i = 0; i < a.length; i++) {
					p[a[i]] = p = p[a[i]] || {};
				}
				return p;
			},
			gs = _namespace("com.greensock"),
			_tinyNum = 0.0000000001,
			_slice = function(a) { //don't use Array.prototype.slice.call(target, 0) because that doesn't work in IE8 with a NodeList that's returned by querySelectorAll()
				var b = [],
					l = a.length,
					i;
				for (i = 0; i !== l; b.push(a[i++]));
				return b;
			},
			_emptyFunc = function() {},
			_isArray = (function() { //works around issues in iframe environments where the Array global isn't shared, thus if the object originates in a different window/iframe, "(obj instanceof Array)" will evaluate false. We added some speed optimizations to avoid Object.prototype.toString.call() unless it's absolutely necessary because it's VERY slow (like 20x slower)
				var toString = Object.prototype.toString,
					array = toString.call([]);
				return function(obj) {
					return obj != null && (obj instanceof Array || (typeof(obj) === "object" && !!obj.push && toString.call(obj) === array));
				};
			}()),
			a, i, p, _ticker, _tickerActive,
			_defLookup = {},

			/**
			 * @constructor
			 * Defines a GreenSock class, optionally with an array of dependencies that must be instantiated first and passed into the definition.
			 * This allows users to load GreenSock JS files in any order even if they have interdependencies (like CSSPlugin extends TweenPlugin which is
			 * inside TweenLite.js, but if CSSPlugin is loaded first, it should wait to run its code until TweenLite.js loads and instantiates TweenPlugin
			 * and then pass TweenPlugin to CSSPlugin's definition). This is all done automatically and internally.
			 *
			 * Every definition will be added to a "com.greensock" global object (typically window, but if a window.GreenSockGlobals object is found,
			 * it will go there as of v1.7). For example, TweenLite will be found at window.com.greensock.TweenLite and since it's a global class that should be available anywhere,
			 * it is ALSO referenced at window.TweenLite. However some classes aren't considered global, like the base com.greensock.core.Animation class, so
			 * those will only be at the package like window.com.greensock.core.Animation. Again, if you define a GreenSockGlobals object on the window, everything
			 * gets tucked neatly inside there instead of on the window directly. This allows you to do advanced things like load multiple versions of GreenSock
			 * files and put them into distinct objects (imagine a banner ad uses a newer version but the main site uses an older one). In that case, you could
			 * sandbox the banner one like:
			 *
			 * <script>
			 *     var gs = window.GreenSockGlobals = {}; //the newer version we're about to load could now be referenced in a "gs" object, like gs.TweenLite.to(...). Use whatever alias you want as long as it's unique, "gs" or "banner" or whatever.
			 * </script>
			 * <script src="js/greensock/v1.7/TweenMax.js"></script>
			 * <script>
			 *     window.GreenSockGlobals = window._gsQueue = window._gsDefine = null; //reset it back to null (along with the special _gsQueue variable) so that the next load of TweenMax affects the window and we can reference things directly like TweenLite.to(...)
			 * </script>
			 * <script src="js/greensock/v1.6/TweenMax.js"></script>
			 * <script>
			 *     gs.TweenLite.to(...); //would use v1.7
			 *     TweenLite.to(...); //would use v1.6
			 * </script>
			 *
			 * @param {!string} ns The namespace of the class definition, leaving off "com.greensock." as that's assumed. For example, "TweenLite" or "plugins.CSSPlugin" or "easing.Back".
			 * @param {!Array.<string>} dependencies An array of dependencies (described as their namespaces minus "com.greensock." prefix). For example ["TweenLite","plugins.TweenPlugin","core.Animation"]
			 * @param {!function():Object} func The function that should be called and passed the resolved dependencies which will return the actual class for this definition.
			 * @param {boolean=} global If true, the class will be added to the global scope (typically window unless you define a window.GreenSockGlobals object)
			 */
			Definition = function(ns, dependencies, func, global) {
				this.sc = (_defLookup[ns]) ? _defLookup[ns].sc : []; //subclasses
				_defLookup[ns] = this;
				this.gsClass = null;
				this.func = func;
				var _classes = [];
				this.check = function(init) {
					var i = dependencies.length,
						missing = i,
						cur, a, n, cl;
					while (--i > -1) {
						if ((cur = _defLookup[dependencies[i]] || new Definition(dependencies[i], [])).gsClass) {
							_classes[i] = cur.gsClass;
							missing--;
						} else if (init) {
							cur.sc.push(this);
						}
					}
					if (missing === 0 && func) {
						a = ("com.greensock." + ns).split(".");
						n = a.pop();
						cl = _namespace(a.join("."))[n] = this.gsClass = func.apply(func, _classes);

						//exports to multiple environments
						if (global) {
							_globals[n] = cl; //provides a way to avoid global namespace pollution. By default, the main classes like TweenLite, Power1, Strong, etc. are added to window unless a GreenSockGlobals is defined. So if you want to have things added to a custom object instead, just do something like window.GreenSockGlobals = {} before loading any GreenSock files. You can even set up an alias like window.GreenSockGlobals = windows.gs = {} so that you can access everything like gs.TweenLite. Also remember that ALL classes are added to the window.com.greensock object (in their respective packages, like com.greensock.easing.Power1, com.greensock.TweenLite, etc.)
							if (typeof(define) === "function" && define.amd){ //AMD
								define((window.GreenSockAMDPath ? window.GreenSockAMDPath + "/" : "") + ns.split(".").pop(), [], function() { return cl; });
							} else if (ns === moduleName && typeof(module) !== "undefined" && module.exports){ //node
								module.exports = cl;
							}
						}
						for (i = 0; i < this.sc.length; i++) {
							this.sc[i].check();
						}
					}
				};
				this.check(true);
			},

			//used to create Definition instances (which basically registers a class that has dependencies).
			_gsDefine = window._gsDefine = function(ns, dependencies, func, global) {
				return new Definition(ns, dependencies, func, global);
			},

			//a quick way to create a class that doesn't have any dependencies. Returns the class, but first registers it in the GreenSock namespace so that other classes can grab it (other classes might be dependent on the class).
			_class = gs._class = function(ns, func, global) {
				func = func || function() {};
				_gsDefine(ns, [], function(){ return func; }, global);
				return func;
			};

		_gsDefine.globals = _globals;



/*
 * ----------------------------------------------------------------
 * Ease
 * ----------------------------------------------------------------
 */
		var _baseParams = [0, 0, 1, 1],
			_blankArray = [],
			Ease = _class("easing.Ease", function(func, extraParams, type, power) {
				this._func = func;
				this._type = type || 0;
				this._power = power || 0;
				this._params = extraParams ? _baseParams.concat(extraParams) : _baseParams;
			}, true),
			_easeMap = Ease.map = {},
			_easeReg = Ease.register = function(ease, names, types, create) {
				var na = names.split(","),
					i = na.length,
					ta = (types || "easeIn,easeOut,easeInOut").split(","),
					e, name, j, type;
				while (--i > -1) {
					name = na[i];
					e = create ? _class("easing."+name, null, true) : gs.easing[name] || {};
					j = ta.length;
					while (--j > -1) {
						type = ta[j];
						_easeMap[name + "." + type] = _easeMap[type + name] = e[type] = ease.getRatio ? ease : ease[type] || new ease();
					}
				}
			};

		p = Ease.prototype;
		p._calcEnd = false;
		p.getRatio = function(p) {
			if (this._func) {
				this._params[0] = p;
				return this._func.apply(null, this._params);
			}
			var t = this._type,
				pw = this._power,
				r = (t === 1) ? 1 - p : (t === 2) ? p : (p < 0.5) ? p * 2 : (1 - p) * 2;
			if (pw === 1) {
				r *= r;
			} else if (pw === 2) {
				r *= r * r;
			} else if (pw === 3) {
				r *= r * r * r;
			} else if (pw === 4) {
				r *= r * r * r * r;
			}
			return (t === 1) ? 1 - r : (t === 2) ? r : (p < 0.5) ? r / 2 : 1 - (r / 2);
		};

		//create all the standard eases like Linear, Quad, Cubic, Quart, Quint, Strong, Power0, Power1, Power2, Power3, and Power4 (each with easeIn, easeOut, and easeInOut)
		a = ["Linear","Quad","Cubic","Quart","Quint,Strong"];
		i = a.length;
		while (--i > -1) {
			p = a[i]+",Power"+i;
			_easeReg(new Ease(null,null,1,i), p, "easeOut", true);
			_easeReg(new Ease(null,null,2,i), p, "easeIn" + ((i === 0) ? ",easeNone" : ""));
			_easeReg(new Ease(null,null,3,i), p, "easeInOut");
		}
		_easeMap.linear = gs.easing.Linear.easeIn;
		_easeMap.swing = gs.easing.Quad.easeInOut; //for jQuery folks


/*
 * ----------------------------------------------------------------
 * EventDispatcher
 * ----------------------------------------------------------------
 */
		var EventDispatcher = _class("events.EventDispatcher", function(target) {
			this._listeners = {};
			this._eventTarget = target || this;
		});
		p = EventDispatcher.prototype;

		p.addEventListener = function(type, callback, scope, useParam, priority) {
			priority = priority || 0;
			var list = this._listeners[type],
				index = 0,
				listener, i;
			if (list == null) {
				this._listeners[type] = list = [];
			}
			i = list.length;
			while (--i > -1) {
				listener = list[i];
				if (listener.c === callback && listener.s === scope) {
					list.splice(i, 1);
				} else if (index === 0 && listener.pr < priority) {
					index = i + 1;
				}
			}
			list.splice(index, 0, {c:callback, s:scope, up:useParam, pr:priority});
			if (this === _ticker && !_tickerActive) {
				_ticker.wake();
			}
		};

		p.removeEventListener = function(type, callback) {
			var list = this._listeners[type], i;
			if (list) {
				i = list.length;
				while (--i > -1) {
					if (list[i].c === callback) {
						list.splice(i, 1);
						return;
					}
				}
			}
		};

		p.dispatchEvent = function(type) {
			var list = this._listeners[type],
				i, t, listener;
			if (list) {
				i = list.length;
				t = this._eventTarget;
				while (--i > -1) {
					listener = list[i];
					if (listener) {
						if (listener.up) {
							listener.c.call(listener.s || t, {type:type, target:t});
						} else {
							listener.c.call(listener.s || t);
						}
					}
				}
			}
		};


/*
 * ----------------------------------------------------------------
 * Ticker
 * ----------------------------------------------------------------
 */
 		var _reqAnimFrame = window.requestAnimationFrame,
			_cancelAnimFrame = window.cancelAnimationFrame,
			_getTime = Date.now || function() {return new Date().getTime();},
			_lastUpdate = _getTime();

		//now try to determine the requestAnimationFrame and cancelAnimationFrame functions and if none are found, we'll use a setTimeout()/clearTimeout() polyfill.
		a = ["ms","moz","webkit","o"];
		i = a.length;
		while (--i > -1 && !_reqAnimFrame) {
			_reqAnimFrame = window[a[i] + "RequestAnimationFrame"];
			_cancelAnimFrame = window[a[i] + "CancelAnimationFrame"] || window[a[i] + "CancelRequestAnimationFrame"];
		}

		_class("Ticker", function(fps, useRAF) {
			var _self = this,
				_startTime = _getTime(),
				_useRAF = (useRAF !== false && _reqAnimFrame),
				_lagThreshold = 500,
				_adjustedLag = 33,
				_tickWord = "tick", //helps reduce gc burden
				_fps, _req, _id, _gap, _nextTime,
				_tick = function(manual) {
					var elapsed = _getTime() - _lastUpdate,
						overlap, dispatch;
					if (elapsed > _lagThreshold) {
						_startTime += elapsed - _adjustedLag;
					}
					_lastUpdate += elapsed;
					_self.time = (_lastUpdate - _startTime) / 1000;
					overlap = _self.time - _nextTime;
					if (!_fps || overlap > 0 || manual === true) {
						_self.frame++;
						_nextTime += overlap + (overlap >= _gap ? 0.004 : _gap - overlap);
						dispatch = true;
					}
					if (manual !== true) { //make sure the request is made before we dispatch the "tick" event so that timing is maintained. Otherwise, if processing the "tick" requires a bunch of time (like 15ms) and we're using a setTimeout() that's based on 16.7ms, it'd technically take 31.7ms between frames otherwise.
						_id = _req(_tick);
					}
					if (dispatch) {
						_self.dispatchEvent(_tickWord);
					}
				};

			EventDispatcher.call(_self);
			_self.time = _self.frame = 0;
			_self.tick = function() {
				_tick(true);
			};

			_self.lagSmoothing = function(threshold, adjustedLag) {
				_lagThreshold = threshold || (1 / _tinyNum); //zero should be interpreted as basically unlimited
				_adjustedLag = Math.min(adjustedLag, _lagThreshold, 0);
			};

			_self.sleep = function() {
				if (_id == null) {
					return;
				}
				if (!_useRAF || !_cancelAnimFrame) {
					clearTimeout(_id);
				} else {
					_cancelAnimFrame(_id);
				}
				_req = _emptyFunc;
				_id = null;
				if (_self === _ticker) {
					_tickerActive = false;
				}
			};

			_self.wake = function() {
				if (_id !== null) {
					_self.sleep();
				} else if (_self.frame > 10) { //don't trigger lagSmoothing if we're just waking up, and make sure that at least 10 frames have elapsed because of the iOS bug that we work around below with the 1.5-second setTimout().
					_lastUpdate = _getTime() - _lagThreshold + 5;
				}
				_req = (_fps === 0) ? _emptyFunc : (!_useRAF || !_reqAnimFrame) ? function(f) { return setTimeout(f, ((_nextTime - _self.time) * 1000 + 1) | 0); } : _reqAnimFrame;
				if (_self === _ticker) {
					_tickerActive = true;
				}
				_tick(2);
			};

			_self.fps = function(value) {
				if (!arguments.length) {
					return _fps;
				}
				_fps = value;
				_gap = 1 / (_fps || 60);
				_nextTime = this.time + _gap;
				_self.wake();
			};

			_self.useRAF = function(value) {
				if (!arguments.length) {
					return _useRAF;
				}
				_self.sleep();
				_useRAF = value;
				_self.fps(_fps);
			};
			_self.fps(fps);

			//a bug in iOS 6 Safari occasionally prevents the requestAnimationFrame from working initially, so we use a 1.5-second timeout that automatically falls back to setTimeout() if it senses this condition.
			setTimeout(function() {
				if (_useRAF && _self.frame < 5) {
					_self.useRAF(false);
				}
			}, 1500);
		});

		p = gs.Ticker.prototype = new gs.events.EventDispatcher();
		p.constructor = gs.Ticker;


/*
 * ----------------------------------------------------------------
 * Animation
 * ----------------------------------------------------------------
 */
		var Animation = _class("core.Animation", function(duration, vars) {
				this.vars = vars = vars || {};
				this._duration = this._totalDuration = duration || 0;
				this._delay = Number(vars.delay) || 0;
				this._timeScale = 1;
				this._active = (vars.immediateRender === true);
				this.data = vars.data;
				this._reversed = (vars.reversed === true);

				if (!_rootTimeline) {
					return;
				}
				if (!_tickerActive) { //some browsers (like iOS 6 Safari) shut down JavaScript execution when the tab is disabled and they [occasionally] neglect to start up requestAnimationFrame again when returning - this code ensures that the engine starts up again properly.
					_ticker.wake();
				}

				var tl = this.vars.useFrames ? _rootFramesTimeline : _rootTimeline;
				tl.add(this, tl._time);

				if (this.vars.paused) {
					this.paused(true);
				}
			});

		_ticker = Animation.ticker = new gs.Ticker();
		p = Animation.prototype;
		p._dirty = p._gc = p._initted = p._paused = false;
		p._totalTime = p._time = 0;
		p._rawPrevTime = -1;
		p._next = p._last = p._onUpdate = p._timeline = p.timeline = null;
		p._paused = false;


		//some browsers (like iOS) occasionally drop the requestAnimationFrame event when the user switches to a different tab and then comes back again, so we use a 2-second setTimeout() to sense if/when that condition occurs and then wake() the ticker.
		var _checkTimeout = function() {
				if (_tickerActive && _getTime() - _lastUpdate > 2000) {
					_ticker.wake();
				}
				setTimeout(_checkTimeout, 2000);
			};
		_checkTimeout();


		p.play = function(from, suppressEvents) {
			if (from != null) {
				this.seek(from, suppressEvents);
			}
			return this.reversed(false).paused(false);
		};

		p.pause = function(atTime, suppressEvents) {
			if (atTime != null) {
				this.seek(atTime, suppressEvents);
			}
			return this.paused(true);
		};

		p.resume = function(from, suppressEvents) {
			if (from != null) {
				this.seek(from, suppressEvents);
			}
			return this.paused(false);
		};

		p.seek = function(time, suppressEvents) {
			return this.totalTime(Number(time), suppressEvents !== false);
		};

		p.restart = function(includeDelay, suppressEvents) {
			return this.reversed(false).paused(false).totalTime(includeDelay ? -this._delay : 0, (suppressEvents !== false), true);
		};

		p.reverse = function(from, suppressEvents) {
			if (from != null) {
				this.seek((from || this.totalDuration()), suppressEvents);
			}
			return this.reversed(true).paused(false);
		};

		p.render = function(time, suppressEvents, force) {
			//stub - we override this method in subclasses.
		};

		p.invalidate = function() {
			this._time = this._totalTime = 0;
			this._initted = this._gc = false;
			this._rawPrevTime = -1;
			if (this._gc || !this.timeline) {
				this._enabled(true);
			}
			return this;
		};

		p.isActive = function() {
			var tl = this._timeline, //the 2 root timelines won't have a _timeline; they're always active.
				startTime = this._startTime,
				rawTime;
			return (!tl || (!this._gc && !this._paused && tl.isActive() && (rawTime = tl.rawTime()) >= startTime && rawTime < startTime + this.totalDuration() / this._timeScale));
		};

		p._enabled = function (enabled, ignoreTimeline) {
			if (!_tickerActive) {
				_ticker.wake();
			}
			this._gc = !enabled;
			this._active = this.isActive();
			if (ignoreTimeline !== true) {
				if (enabled && !this.timeline) {
					this._timeline.add(this, this._startTime - this._delay);
				} else if (!enabled && this.timeline) {
					this._timeline._remove(this, true);
				}
			}
			return false;
		};


		p._kill = function(vars, target) {
			return this._enabled(false, false);
		};

		p.kill = function(vars, target) {
			this._kill(vars, target);
			return this;
		};

		p._uncache = function(includeSelf) {
			var tween = includeSelf ? this : this.timeline;
			while (tween) {
				tween._dirty = true;
				tween = tween.timeline;
			}
			return this;
		};

		p._swapSelfInParams = function(params) {
			var i = params.length,
				copy = params.concat();
			while (--i > -1) {
				if (params[i] === "{self}") {
					copy[i] = this;
				}
			}
			return copy;
		};

//----Animation getters/setters --------------------------------------------------------

		p.eventCallback = function(type, callback, params, scope) {
			if ((type || "").substr(0,2) === "on") {
				var v = this.vars;
				if (arguments.length === 1) {
					return v[type];
				}
				if (callback == null) {
					delete v[type];
				} else {
					v[type] = callback;
					v[type + "Params"] = (_isArray(params) && params.join("").indexOf("{self}") !== -1) ? this._swapSelfInParams(params) : params;
					v[type + "Scope"] = scope;
				}
				if (type === "onUpdate") {
					this._onUpdate = callback;
				}
			}
			return this;
		};

		p.delay = function(value) {
			if (!arguments.length) {
				return this._delay;
			}
			if (this._timeline.smoothChildTiming) {
				this.startTime( this._startTime + value - this._delay );
			}
			this._delay = value;
			return this;
		};

		p.duration = function(value) {
			if (!arguments.length) {
				this._dirty = false;
				return this._duration;
			}
			this._duration = this._totalDuration = value;
			this._uncache(true); //true in case it's a TweenMax or TimelineMax that has a repeat - we'll need to refresh the totalDuration.
			if (this._timeline.smoothChildTiming) if (this._time > 0) if (this._time < this._duration) if (value !== 0) {
				this.totalTime(this._totalTime * (value / this._duration), true);
			}
			return this;
		};

		p.totalDuration = function(value) {
			this._dirty = false;
			return (!arguments.length) ? this._totalDuration : this.duration(value);
		};

		p.time = function(value, suppressEvents) {
			if (!arguments.length) {
				return this._time;
			}
			if (this._dirty) {
				this.totalDuration();
			}
			return this.totalTime((value > this._duration) ? this._duration : value, suppressEvents);
		};

		p.totalTime = function(time, suppressEvents, uncapped) {
			if (!_tickerActive) {
				_ticker.wake();
			}
			if (!arguments.length) {
				return this._totalTime;
			}
			if (this._timeline) {
				if (time < 0 && !uncapped) {
					time += this.totalDuration();
				}
				if (this._timeline.smoothChildTiming) {
					if (this._dirty) {
						this.totalDuration();
					}
					var totalDuration = this._totalDuration,
						tl = this._timeline;
					if (time > totalDuration && !uncapped) {
						time = totalDuration;
					}
					this._startTime = (this._paused ? this._pauseTime : tl._time) - ((!this._reversed ? time : totalDuration - time) / this._timeScale);
					if (!tl._dirty) { //for performance improvement. If the parent's cache is already dirty, it already took care of marking the ancestors as dirty too, so skip the function call here.
						this._uncache(false);
					}
					//in case any of the ancestor timelines had completed but should now be enabled, we should reset their totalTime() which will also ensure that they're lined up properly and enabled. Skip for animations that are on the root (wasteful). Example: a TimelineLite.exportRoot() is performed when there's a paused tween on the root, the export will not complete until that tween is unpaused, but imagine a child gets restarted later, after all [unpaused] tweens have completed. The startTime of that child would get pushed out, but one of the ancestors may have completed.
					if (tl._timeline) {
						while (tl._timeline) {
							if (tl._timeline._time !== (tl._startTime + tl._totalTime) / tl._timeScale) {
								tl.totalTime(tl._totalTime, true);
							}
							tl = tl._timeline;
						}
					}
				}
				if (this._gc) {
					this._enabled(true, false);
				}
				if (this._totalTime !== time || this._duration === 0) {
					this.render(time, suppressEvents, false);
					if (_lazyTweens.length) { //in case rendering caused any tweens to lazy-init, we should render them because typically when someone calls seek() or time() or progress(), they expect an immediate render.
						_lazyRender();
					}
				}
			}
			return this;
		};

		p.progress = p.totalProgress = function(value, suppressEvents) {
			return (!arguments.length) ? this._time / this.duration() : this.totalTime(this.duration() * value, suppressEvents);
		};

		p.startTime = function(value) {
			if (!arguments.length) {
				return this._startTime;
			}
			if (value !== this._startTime) {
				this._startTime = value;
				if (this.timeline) if (this.timeline._sortChildren) {
					this.timeline.add(this, value - this._delay); //ensures that any necessary re-sequencing of Animations in the timeline occurs to make sure the rendering order is correct.
				}
			}
			return this;
		};

		p.endTime = function(includeRepeats) {
			return this._startTime + ((includeRepeats != false) ? this.totalDuration() : this.duration()) / this._timeScale;
		};

		p.timeScale = function(value) {
			if (!arguments.length) {
				return this._timeScale;
			}
			value = value || _tinyNum; //can't allow zero because it'll throw the math off
			if (this._timeline && this._timeline.smoothChildTiming) {
				var pauseTime = this._pauseTime,
					t = (pauseTime || pauseTime === 0) ? pauseTime : this._timeline.totalTime();
				this._startTime = t - ((t - this._startTime) * this._timeScale / value);
			}
			this._timeScale = value;
			return this._uncache(false);
		};

		p.reversed = function(value) {
			if (!arguments.length) {
				return this._reversed;
			}
			if (value != this._reversed) {
				this._reversed = value;
				this.totalTime(((this._timeline && !this._timeline.smoothChildTiming) ? this.totalDuration() - this._totalTime : this._totalTime), true);
			}
			return this;
		};

		p.paused = function(value) {
			if (!arguments.length) {
				return this._paused;
			}
			var tl = this._timeline,
				raw, elapsed;
			if (value != this._paused) if (tl) {
				if (!_tickerActive && !value) {
					_ticker.wake();
				}
				raw = tl.rawTime();
				elapsed = raw - this._pauseTime;
				if (!value && tl.smoothChildTiming) {
					this._startTime += elapsed;
					this._uncache(false);
				}
				this._pauseTime = value ? raw : null;
				this._paused = value;
				this._active = this.isActive();
				if (!value && elapsed !== 0 && this._initted && this.duration()) {
					this.render((tl.smoothChildTiming ? this._totalTime : (raw - this._startTime) / this._timeScale), true, true); //in case the target's properties changed via some other tween or manual update by the user, we should force a render.
				}
			}
			if (this._gc && !value) {
				this._enabled(true, false);
			}
			return this;
		};


/*
 * ----------------------------------------------------------------
 * SimpleTimeline
 * ----------------------------------------------------------------
 */
		var SimpleTimeline = _class("core.SimpleTimeline", function(vars) {
			Animation.call(this, 0, vars);
			this.autoRemoveChildren = this.smoothChildTiming = true;
		});

		p = SimpleTimeline.prototype = new Animation();
		p.constructor = SimpleTimeline;
		p.kill()._gc = false;
		p._first = p._last = p._recent = null;
		p._sortChildren = false;

		p.add = p.insert = function(child, position, align, stagger) {
			var prevTween, st;
			child._startTime = Number(position || 0) + child._delay;
			if (child._paused) if (this !== child._timeline) { //we only adjust the _pauseTime if it wasn't in this timeline already. Remember, sometimes a tween will be inserted again into the same timeline when its startTime is changed so that the tweens in the TimelineLite/Max are re-ordered properly in the linked list (so everything renders in the proper order).
				child._pauseTime = child._startTime + ((this.rawTime() - child._startTime) / child._timeScale);
			}
			if (child.timeline) {
				child.timeline._remove(child, true); //removes from existing timeline so that it can be properly added to this one.
			}
			child.timeline = child._timeline = this;
			if (child._gc) {
				child._enabled(true, true);
			}
			prevTween = this._last;
			if (this._sortChildren) {
				st = child._startTime;
				while (prevTween && prevTween._startTime > st) {
					prevTween = prevTween._prev;
				}
			}
			if (prevTween) {
				child._next = prevTween._next;
				prevTween._next = child;
			} else {
				child._next = this._first;
				this._first = child;
			}
			if (child._next) {
				child._next._prev = child;
			} else {
				this._last = child;
			}
			child._prev = prevTween;
			this._recent = child;
			if (this._timeline) {
				this._uncache(true);
			}
			return this;
		};

		p._remove = function(tween, skipDisable) {
			if (tween.timeline === this) {
				if (!skipDisable) {
					tween._enabled(false, true);
				}

				if (tween._prev) {
					tween._prev._next = tween._next;
				} else if (this._first === tween) {
					this._first = tween._next;
				}
				if (tween._next) {
					tween._next._prev = tween._prev;
				} else if (this._last === tween) {
					this._last = tween._prev;
				}
				tween._next = tween._prev = tween.timeline = null;
				if (tween === this._recent) {
					this._recent = this._last;
				}

				if (this._timeline) {
					this._uncache(true);
				}
			}
			return this;
		};

		p.render = function(time, suppressEvents, force) {
			var tween = this._first,
				next;
			this._totalTime = this._time = this._rawPrevTime = time;
			while (tween) {
				next = tween._next; //record it here because the value could change after rendering...
				if (tween._active || (time >= tween._startTime && !tween._paused)) {
					if (!tween._reversed) {
						tween.render((time - tween._startTime) * tween._timeScale, suppressEvents, force);
					} else {
						tween.render(((!tween._dirty) ? tween._totalDuration : tween.totalDuration()) - ((time - tween._startTime) * tween._timeScale), suppressEvents, force);
					}
				}
				tween = next;
			}
		};

		p.rawTime = function() {
			if (!_tickerActive) {
				_ticker.wake();
			}
			return this._totalTime;
		};

/*
 * ----------------------------------------------------------------
 * TweenLite
 * ----------------------------------------------------------------
 */
		var TweenLite = _class("TweenLite", function(target, duration, vars) {
				Animation.call(this, duration, vars);
				this.render = TweenLite.prototype.render; //speed optimization (avoid prototype lookup on this "hot" method)

				if (target == null) {
					throw "Cannot tween a null target.";
				}

				this.target = target = (typeof(target) !== "string") ? target : TweenLite.selector(target) || target;

				var isSelector = (target.jquery || (target.length && target !== window && target[0] && (target[0] === window || (target[0].nodeType && target[0].style && !target.nodeType)))),
					overwrite = this.vars.overwrite,
					i, targ, targets;

				this._overwrite = overwrite = (overwrite == null) ? _overwriteLookup[TweenLite.defaultOverwrite] : (typeof(overwrite) === "number") ? overwrite >> 0 : _overwriteLookup[overwrite];

				if ((isSelector || target instanceof Array || (target.push && _isArray(target))) && typeof(target[0]) !== "number") {
					this._targets = targets = _slice(target);  //don't use Array.prototype.slice.call(target, 0) because that doesn't work in IE8 with a NodeList that's returned by querySelectorAll()
					this._propLookup = [];
					this._siblings = [];
					for (i = 0; i < targets.length; i++) {
						targ = targets[i];
						if (!targ) {
							targets.splice(i--, 1);
							continue;
						} else if (typeof(targ) === "string") {
							targ = targets[i--] = TweenLite.selector(targ); //in case it's an array of strings
							if (typeof(targ) === "string") {
								targets.splice(i+1, 1); //to avoid an endless loop (can't imagine why the selector would return a string, but just in case)
							}
							continue;
						} else if (targ.length && targ !== window && targ[0] && (targ[0] === window || (targ[0].nodeType && targ[0].style && !targ.nodeType))) { //in case the user is passing in an array of selector objects (like jQuery objects), we need to check one more level and pull things out if necessary. Also note that <select> elements pass all the criteria regarding length and the first child having style, so we must also check to ensure the target isn't an HTML node itself.
							targets.splice(i--, 1);
							this._targets = targets = targets.concat(_slice(targ));
							continue;
						}
						this._siblings[i] = _register(targ, this, false);
						if (overwrite === 1) if (this._siblings[i].length > 1) {
							_applyOverwrite(targ, this, null, 1, this._siblings[i]);
						}
					}

				} else {
					this._propLookup = {};
					this._siblings = _register(target, this, false);
					if (overwrite === 1) if (this._siblings.length > 1) {
						_applyOverwrite(target, this, null, 1, this._siblings);
					}
				}
				if (this.vars.immediateRender || (duration === 0 && this._delay === 0 && this.vars.immediateRender !== false)) {
					this._time = -_tinyNum; //forces a render without having to set the render() "force" parameter to true because we want to allow lazying by default (using the "force" parameter always forces an immediate full render)
					this.render(-this._delay);
				}
			}, true),
			_isSelector = function(v) {
				return (v && v.length && v !== window && v[0] && (v[0] === window || (v[0].nodeType && v[0].style && !v.nodeType))); //we cannot check "nodeType" if the target is window from within an iframe, otherwise it will trigger a security error in some browsers like Firefox.
			},
			_autoCSS = function(vars, target) {
				var css = {},
					p;
				for (p in vars) {
					if (!_reservedProps[p] && (!(p in target) || p === "transform" || p === "x" || p === "y" || p === "width" || p === "height" || p === "className" || p === "border") && (!_plugins[p] || (_plugins[p] && _plugins[p]._autoCSS))) { //note: <img> elements contain read-only "x" and "y" properties. We should also prioritize editing css width/height rather than the element's properties.
						css[p] = vars[p];
						delete vars[p];
					}
				}
				vars.css = css;
			};

		p = TweenLite.prototype = new Animation();
		p.constructor = TweenLite;
		p.kill()._gc = false;

//----TweenLite defaults, overwrite management, and root updates ----------------------------------------------------

		p.ratio = 0;
		p._firstPT = p._targets = p._overwrittenProps = p._startAt = null;
		p._notifyPluginsOfEnabled = p._lazy = false;

		TweenLite.version = "1.16.1";
		TweenLite.defaultEase = p._ease = new Ease(null, null, 1, 1);
		TweenLite.defaultOverwrite = "auto";
		TweenLite.ticker = _ticker;
		TweenLite.autoSleep = 120;
		TweenLite.lagSmoothing = function(threshold, adjustedLag) {
			_ticker.lagSmoothing(threshold, adjustedLag);
		};

		TweenLite.selector = window.$ || window.jQuery || function(e) {
			var selector = window.$ || window.jQuery;
			if (selector) {
				TweenLite.selector = selector;
				return selector(e);
			}
			return (typeof(document) === "undefined") ? e : (document.querySelectorAll ? document.querySelectorAll(e) : document.getElementById((e.charAt(0) === "#") ? e.substr(1) : e));
		};

		var _lazyTweens = [],
			_lazyLookup = {},
			_internals = TweenLite._internals = {isArray:_isArray, isSelector:_isSelector, lazyTweens:_lazyTweens}, //gives us a way to expose certain private values to other GreenSock classes without contaminating tha main TweenLite object.
			_plugins = TweenLite._plugins = {},
			_tweenLookup = _internals.tweenLookup = {},
			_tweenLookupNum = 0,
			_reservedProps = _internals.reservedProps = {ease:1, delay:1, overwrite:1, onComplete:1, onCompleteParams:1, onCompleteScope:1, useFrames:1, runBackwards:1, startAt:1, onUpdate:1, onUpdateParams:1, onUpdateScope:1, onStart:1, onStartParams:1, onStartScope:1, onReverseComplete:1, onReverseCompleteParams:1, onReverseCompleteScope:1, onRepeat:1, onRepeatParams:1, onRepeatScope:1, easeParams:1, yoyo:1, immediateRender:1, repeat:1, repeatDelay:1, data:1, paused:1, reversed:1, autoCSS:1, lazy:1, onOverwrite:1},
			_overwriteLookup = {none:0, all:1, auto:2, concurrent:3, allOnStart:4, preexisting:5, "true":1, "false":0},
			_rootFramesTimeline = Animation._rootFramesTimeline = new SimpleTimeline(),
			_rootTimeline = Animation._rootTimeline = new SimpleTimeline(),
			_nextGCFrame = 30,
			_lazyRender = _internals.lazyRender = function() {
				var i = _lazyTweens.length,
					tween;
				_lazyLookup = {};
				while (--i > -1) {
					tween = _lazyTweens[i];
					if (tween && tween._lazy !== false) {
						tween.render(tween._lazy[0], tween._lazy[1], true);
						tween._lazy = false;
					}
				}
				_lazyTweens.length = 0;
			};

		_rootTimeline._startTime = _ticker.time;
		_rootFramesTimeline._startTime = _ticker.frame;
		_rootTimeline._active = _rootFramesTimeline._active = true;
		setTimeout(_lazyRender, 1); //on some mobile devices, there isn't a "tick" before code runs which means any lazy renders wouldn't run before the next official "tick".

		Animation._updateRoot = TweenLite.render = function() {
				var i, a, p;
				if (_lazyTweens.length) { //if code is run outside of the requestAnimationFrame loop, there may be tweens queued AFTER the engine refreshed, so we need to ensure any pending renders occur before we refresh again.
					_lazyRender();
				}
				_rootTimeline.render((_ticker.time - _rootTimeline._startTime) * _rootTimeline._timeScale, false, false);
				_rootFramesTimeline.render((_ticker.frame - _rootFramesTimeline._startTime) * _rootFramesTimeline._timeScale, false, false);
				if (_lazyTweens.length) {
					_lazyRender();
				}
				if (_ticker.frame >= _nextGCFrame) { //dump garbage every 120 frames or whatever the user sets TweenLite.autoSleep to
					_nextGCFrame = _ticker.frame + (parseInt(TweenLite.autoSleep, 10) || 120);
					for (p in _tweenLookup) {
						a = _tweenLookup[p].tweens;
						i = a.length;
						while (--i > -1) {
							if (a[i]._gc) {
								a.splice(i, 1);
							}
						}
						if (a.length === 0) {
							delete _tweenLookup[p];
						}
					}
					//if there are no more tweens in the root timelines, or if they're all paused, make the _timer sleep to reduce load on the CPU slightly
					p = _rootTimeline._first;
					if (!p || p._paused) if (TweenLite.autoSleep && !_rootFramesTimeline._first && _ticker._listeners.tick.length === 1) {
						while (p && p._paused) {
							p = p._next;
						}
						if (!p) {
							_ticker.sleep();
						}
					}
				}
			};

		_ticker.addEventListener("tick", Animation._updateRoot);

		var _register = function(target, tween, scrub) {
				var id = target._gsTweenID, a, i;
				if (!_tweenLookup[id || (target._gsTweenID = id = "t" + (_tweenLookupNum++))]) {
					_tweenLookup[id] = {target:target, tweens:[]};
				}
				if (tween) {
					a = _tweenLookup[id].tweens;
					a[(i = a.length)] = tween;
					if (scrub) {
						while (--i > -1) {
							if (a[i] === tween) {
								a.splice(i, 1);
							}
						}
					}
				}
				return _tweenLookup[id].tweens;
			},

			_onOverwrite = function(overwrittenTween, overwritingTween, target, killedProps) {
				var func = overwrittenTween.vars.onOverwrite, r1, r2;
				if (func) {
					r1 = func(overwrittenTween, overwritingTween, target, killedProps);
				}
				func = TweenLite.onOverwrite;
				if (func) {
					r2 = func(overwrittenTween, overwritingTween, target, killedProps);
				}
				return (r1 !== false && r2 !== false);
			},
			_applyOverwrite = function(target, tween, props, mode, siblings) {
				var i, changed, curTween, l;
				if (mode === 1 || mode >= 4) {
					l = siblings.length;
					for (i = 0; i < l; i++) {
						if ((curTween = siblings[i]) !== tween) {
							if (!curTween._gc) {
								if (_onOverwrite(curTween, tween) && curTween._enabled(false, false)) {
									changed = true;
								}
							}
						} else if (mode === 5) {
							break;
						}
					}
					return changed;
				}
				//NOTE: Add 0.0000000001 to overcome floating point errors that can cause the startTime to be VERY slightly off (when a tween's time() is set for example)
				var startTime = tween._startTime + _tinyNum,
					overlaps = [],
					oCount = 0,
					zeroDur = (tween._duration === 0),
					globalStart;
				i = siblings.length;
				while (--i > -1) {
					if ((curTween = siblings[i]) === tween || curTween._gc || curTween._paused) {
						//ignore
					} else if (curTween._timeline !== tween._timeline) {
						globalStart = globalStart || _checkOverlap(tween, 0, zeroDur);
						if (_checkOverlap(curTween, globalStart, zeroDur) === 0) {
							overlaps[oCount++] = curTween;
						}
					} else if (curTween._startTime <= startTime) if (curTween._startTime + curTween.totalDuration() / curTween._timeScale > startTime) if (!((zeroDur || !curTween._initted) && startTime - curTween._startTime <= 0.0000000002)) {
						overlaps[oCount++] = curTween;
					}
				}

				i = oCount;
				while (--i > -1) {
					curTween = overlaps[i];
					if (mode === 2) if (curTween._kill(props, target, tween)) {
						changed = true;
					}
					if (mode !== 2 || (!curTween._firstPT && curTween._initted)) {
						if (mode !== 2 && !_onOverwrite(curTween, tween)) {
							continue;
						}
						if (curTween._enabled(false, false)) { //if all property tweens have been overwritten, kill the tween.
							changed = true;
						}
					}
				}
				return changed;
			},

			_checkOverlap = function(tween, reference, zeroDur) {
				var tl = tween._timeline,
					ts = tl._timeScale,
					t = tween._startTime;
				while (tl._timeline) {
					t += tl._startTime;
					ts *= tl._timeScale;
					if (tl._paused) {
						return -100;
					}
					tl = tl._timeline;
				}
				t /= ts;
				return (t > reference) ? t - reference : ((zeroDur && t === reference) || (!tween._initted && t - reference < 2 * _tinyNum)) ? _tinyNum : ((t += tween.totalDuration() / tween._timeScale / ts) > reference + _tinyNum) ? 0 : t - reference - _tinyNum;
			};


//---- TweenLite instance methods -----------------------------------------------------------------------------

		p._init = function() {
			var v = this.vars,
				op = this._overwrittenProps,
				dur = this._duration,
				immediate = !!v.immediateRender,
				ease = v.ease,
				i, initPlugins, pt, p, startVars;
			if (v.startAt) {
				if (this._startAt) {
					this._startAt.render(-1, true); //if we've run a startAt previously (when the tween instantiated), we should revert it so that the values re-instantiate correctly particularly for relative tweens. Without this, a TweenLite.fromTo(obj, 1, {x:"+=100"}, {x:"-=100"}), for example, would actually jump to +=200 because the startAt would run twice, doubling the relative change.
					this._startAt.kill();
				}
				startVars = {};
				for (p in v.startAt) { //copy the properties/values into a new object to avoid collisions, like var to = {x:0}, from = {x:500}; timeline.fromTo(e, 1, from, to).fromTo(e, 1, to, from);
					startVars[p] = v.startAt[p];
				}
				startVars.overwrite = false;
				startVars.immediateRender = true;
				startVars.lazy = (immediate && v.lazy !== false);
				startVars.startAt = startVars.delay = null; //no nesting of startAt objects allowed (otherwise it could cause an infinite loop).
				this._startAt = TweenLite.to(this.target, 0, startVars);
				if (immediate) {
					if (this._time > 0) {
						this._startAt = null; //tweens that render immediately (like most from() and fromTo() tweens) shouldn't revert when their parent timeline's playhead goes backward past the startTime because the initial render could have happened anytime and it shouldn't be directly correlated to this tween's startTime. Imagine setting up a complex animation where the beginning states of various objects are rendered immediately but the tween doesn't happen for quite some time - if we revert to the starting values as soon as the playhead goes backward past the tween's startTime, it will throw things off visually. Reversion should only happen in TimelineLite/Max instances where immediateRender was false (which is the default in the convenience methods like from()).
					} else if (dur !== 0) {
						return; //we skip initialization here so that overwriting doesn't occur until the tween actually begins. Otherwise, if you create several immediateRender:true tweens of the same target/properties to drop into a TimelineLite or TimelineMax, the last one created would overwrite the first ones because they didn't get placed into the timeline yet before the first render occurs and kicks in overwriting.
					}
				}
			} else if (v.runBackwards && dur !== 0) {
				//from() tweens must be handled uniquely: their beginning values must be rendered but we don't want overwriting to occur yet (when time is still 0). Wait until the tween actually begins before doing all the routines like overwriting. At that time, we should render at the END of the tween to ensure that things initialize correctly (remember, from() tweens go backwards)
				if (this._startAt) {
					this._startAt.render(-1, true);
					this._startAt.kill();
					this._startAt = null;
				} else {
					if (this._time !== 0) { //in rare cases (like if a from() tween runs and then is invalidate()-ed), immediateRender could be true but the initial forced-render gets skipped, so there's no need to force the render in this context when the _time is greater than 0
						immediate = false;
					}
					pt = {};
					for (p in v) { //copy props into a new object and skip any reserved props, otherwise onComplete or onUpdate or onStart could fire. We should, however, permit autoCSS to go through.
						if (!_reservedProps[p] || p === "autoCSS") {
							pt[p] = v[p];
						}
					}
					pt.overwrite = 0;
					pt.data = "isFromStart"; //we tag the tween with as "isFromStart" so that if [inside a plugin] we need to only do something at the very END of a tween, we have a way of identifying this tween as merely the one that's setting the beginning values for a "from()" tween. For example, clearProps in CSSPlugin should only get applied at the very END of a tween and without this tag, from(...{height:100, clearProps:"height", delay:1}) would wipe the height at the beginning of the tween and after 1 second, it'd kick back in.
					pt.lazy = (immediate && v.lazy !== false);
					pt.immediateRender = immediate; //zero-duration tweens render immediately by default, but if we're not specifically instructed to render this tween immediately, we should skip this and merely _init() to record the starting values (rendering them immediately would push them to completion which is wasteful in that case - we'd have to render(-1) immediately after)
					this._startAt = TweenLite.to(this.target, 0, pt);
					if (!immediate) {
						this._startAt._init(); //ensures that the initial values are recorded
						this._startAt._enabled(false); //no need to have the tween render on the next cycle. Disable it because we'll always manually control the renders of the _startAt tween.
						if (this.vars.immediateRender) {
							this._startAt = null;
						}
					} else if (this._time === 0) {
						return;
					}
				}
			}
			this._ease = ease = (!ease) ? TweenLite.defaultEase : (ease instanceof Ease) ? ease : (typeof(ease) === "function") ? new Ease(ease, v.easeParams) : _easeMap[ease] || TweenLite.defaultEase;
			if (v.easeParams instanceof Array && ease.config) {
				this._ease = ease.config.apply(ease, v.easeParams);
			}
			this._easeType = this._ease._type;
			this._easePower = this._ease._power;
			this._firstPT = null;

			if (this._targets) {
				i = this._targets.length;
				while (--i > -1) {
					if ( this._initProps( this._targets[i], (this._propLookup[i] = {}), this._siblings[i], (op ? op[i] : null)) ) {
						initPlugins = true;
					}
				}
			} else {
				initPlugins = this._initProps(this.target, this._propLookup, this._siblings, op);
			}

			if (initPlugins) {
				TweenLite._onPluginEvent("_onInitAllProps", this); //reorders the array in order of priority. Uses a static TweenPlugin method in order to minimize file size in TweenLite
			}
			if (op) if (!this._firstPT) if (typeof(this.target) !== "function") { //if all tweening properties have been overwritten, kill the tween. If the target is a function, it's probably a delayedCall so let it live.
				this._enabled(false, false);
			}
			if (v.runBackwards) {
				pt = this._firstPT;
				while (pt) {
					pt.s += pt.c;
					pt.c = -pt.c;
					pt = pt._next;
				}
			}
			this._onUpdate = v.onUpdate;
			this._initted = true;
		};

		p._initProps = function(target, propLookup, siblings, overwrittenProps) {
			var p, i, initPlugins, plugin, pt, v;
			if (target == null) {
				return false;
			}

			if (_lazyLookup[target._gsTweenID]) {
				_lazyRender(); //if other tweens of the same target have recently initted but haven't rendered yet, we've got to force the render so that the starting values are correct (imagine populating a timeline with a bunch of sequential tweens and then jumping to the end)
			}

			if (!this.vars.css) if (target.style) if (target !== window && target.nodeType) if (_plugins.css) if (this.vars.autoCSS !== false) { //it's so common to use TweenLite/Max to animate the css of DOM elements, we assume that if the target is a DOM element, that's what is intended (a convenience so that users don't have to wrap things in css:{}, although we still recommend it for a slight performance boost and better specificity). Note: we cannot check "nodeType" on the window inside an iframe.
				_autoCSS(this.vars, target);
			}
			for (p in this.vars) {
				v = this.vars[p];
				if (_reservedProps[p]) {
					if (v) if ((v instanceof Array) || (v.push && _isArray(v))) if (v.join("").indexOf("{self}") !== -1) {
						this.vars[p] = v = this._swapSelfInParams(v, this);
					}

				} else if (_plugins[p] && (plugin = new _plugins[p]())._onInitTween(target, this.vars[p], this)) {

					//t - target 		[object]
					//p - property 		[string]
					//s - start			[number]
					//c - change		[number]
					//f - isFunction	[boolean]
					//n - name			[string]
					//pg - isPlugin 	[boolean]
					//pr - priority		[number]
					this._firstPT = pt = {_next:this._firstPT, t:plugin, p:"setRatio", s:0, c:1, f:true, n:p, pg:true, pr:plugin._priority};
					i = plugin._overwriteProps.length;
					while (--i > -1) {
						propLookup[plugin._overwriteProps[i]] = this._firstPT;
					}
					if (plugin._priority || plugin._onInitAllProps) {
						initPlugins = true;
					}
					if (plugin._onDisable || plugin._onEnable) {
						this._notifyPluginsOfEnabled = true;
					}

				} else {
					this._firstPT = propLookup[p] = pt = {_next:this._firstPT, t:target, p:p, f:(typeof(target[p]) === "function"), n:p, pg:false, pr:0};
					pt.s = (!pt.f) ? parseFloat(target[p]) : target[ ((p.indexOf("set") || typeof(target["get" + p.substr(3)]) !== "function") ? p : "get" + p.substr(3)) ]();
					pt.c = (typeof(v) === "string" && v.charAt(1) === "=") ? parseInt(v.charAt(0) + "1", 10) * Number(v.substr(2)) : (Number(v) - pt.s) || 0;
				}
				if (pt) if (pt._next) {
					pt._next._prev = pt;
				}
			}

			if (overwrittenProps) if (this._kill(overwrittenProps, target)) { //another tween may have tried to overwrite properties of this tween before init() was called (like if two tweens start at the same time, the one created second will run first)
				return this._initProps(target, propLookup, siblings, overwrittenProps);
			}
			if (this._overwrite > 1) if (this._firstPT) if (siblings.length > 1) if (_applyOverwrite(target, this, propLookup, this._overwrite, siblings)) {
				this._kill(propLookup, target);
				return this._initProps(target, propLookup, siblings, overwrittenProps);
			}
			if (this._firstPT) if ((this.vars.lazy !== false && this._duration) || (this.vars.lazy && !this._duration)) { //zero duration tweens don't lazy render by default; everything else does.
				_lazyLookup[target._gsTweenID] = true;
			}
			return initPlugins;
		};

		p.render = function(time, suppressEvents, force) {
			var prevTime = this._time,
				duration = this._duration,
				prevRawPrevTime = this._rawPrevTime,
				isComplete, callback, pt, rawPrevTime;
			if (time >= duration) {
				this._totalTime = this._time = duration;
				this.ratio = this._ease._calcEnd ? this._ease.getRatio(1) : 1;
				if (!this._reversed ) {
					isComplete = true;
					callback = "onComplete";
					force = (force || this._timeline.autoRemoveChildren); //otherwise, if the animation is unpaused/activated after it's already finished, it doesn't get removed from the parent timeline.
				}
				if (duration === 0) if (this._initted || !this.vars.lazy || force) { //zero-duration tweens are tricky because we must discern the momentum/direction of time in order to determine whether the starting values should be rendered or the ending values. If the "playhead" of its timeline goes past the zero-duration tween in the forward direction or lands directly on it, the end values should be rendered, but if the timeline's "playhead" moves past it in the backward direction (from a postitive time to a negative time), the starting values must be rendered.
					if (this._startTime === this._timeline._duration) { //if a zero-duration tween is at the VERY end of a timeline and that timeline renders at its end, it will typically add a tiny bit of cushion to the render time to prevent rounding errors from getting in the way of tweens rendering their VERY end. If we then reverse() that timeline, the zero-duration tween will trigger its onReverseComplete even though technically the playhead didn't pass over it again. It's a very specific edge case we must accommodate.
						time = 0;
					}
					if (time === 0 || prevRawPrevTime < 0 || (prevRawPrevTime === _tinyNum && this.data !== "isPause")) if (prevRawPrevTime !== time) { //note: when this.data is "isPause", it's a callback added by addPause() on a timeline that we should not be triggered when LEAVING its exact start time. In other words, tl.addPause(1).play(1) shouldn't pause.
						force = true;
						if (prevRawPrevTime > _tinyNum) {
							callback = "onReverseComplete";
						}
					}
					this._rawPrevTime = rawPrevTime = (!suppressEvents || time || prevRawPrevTime === time) ? time : _tinyNum; //when the playhead arrives at EXACTLY time 0 (right on top) of a zero-duration tween, we need to discern if events are suppressed so that when the playhead moves again (next time), it'll trigger the callback. If events are NOT suppressed, obviously the callback would be triggered in this render. Basically, the callback should fire either when the playhead ARRIVES or LEAVES this exact spot, not both. Imagine doing a timeline.seek(0) and there's a callback that sits at 0. Since events are suppressed on that seek() by default, nothing will fire, but when the playhead moves off of that position, the callback should fire. This behavior is what people intuitively expect. We set the _rawPrevTime to be a precise tiny number to indicate this scenario rather than using another property/variable which would increase memory usage. This technique is less readable, but more efficient.
				}

			} else if (time < 0.0000001) { //to work around occasional floating point math artifacts, round super small values to 0.
				this._totalTime = this._time = 0;
				this.ratio = this._ease._calcEnd ? this._ease.getRatio(0) : 0;
				if (prevTime !== 0 || (duration === 0 && prevRawPrevTime > 0)) {
					callback = "onReverseComplete";
					isComplete = this._reversed;
				}
				if (time < 0) {
					this._active = false;
					if (duration === 0) if (this._initted || !this.vars.lazy || force) { //zero-duration tweens are tricky because we must discern the momentum/direction of time in order to determine whether the starting values should be rendered or the ending values. If the "playhead" of its timeline goes past the zero-duration tween in the forward direction or lands directly on it, the end values should be rendered, but if the timeline's "playhead" moves past it in the backward direction (from a postitive time to a negative time), the starting values must be rendered.
						if (prevRawPrevTime >= 0 && !(prevRawPrevTime === _tinyNum && this.data === "isPause")) {
							force = true;
						}
						this._rawPrevTime = rawPrevTime = (!suppressEvents || time || prevRawPrevTime === time) ? time : _tinyNum; //when the playhead arrives at EXACTLY time 0 (right on top) of a zero-duration tween, we need to discern if events are suppressed so that when the playhead moves again (next time), it'll trigger the callback. If events are NOT suppressed, obviously the callback would be triggered in this render. Basically, the callback should fire either when the playhead ARRIVES or LEAVES this exact spot, not both. Imagine doing a timeline.seek(0) and there's a callback that sits at 0. Since events are suppressed on that seek() by default, nothing will fire, but when the playhead moves off of that position, the callback should fire. This behavior is what people intuitively expect. We set the _rawPrevTime to be a precise tiny number to indicate this scenario rather than using another property/variable which would increase memory usage. This technique is less readable, but more efficient.
					}
				}
				if (!this._initted) { //if we render the very beginning (time == 0) of a fromTo(), we must force the render (normal tweens wouldn't need to render at a time of 0 when the prevTime was also 0). This is also mandatory to make sure overwriting kicks in immediately.
					force = true;
				}
			} else {
				this._totalTime = this._time = time;

				if (this._easeType) {
					var r = time / duration, type = this._easeType, pow = this._easePower;
					if (type === 1 || (type === 3 && r >= 0.5)) {
						r = 1 - r;
					}
					if (type === 3) {
						r *= 2;
					}
					if (pow === 1) {
						r *= r;
					} else if (pow === 2) {
						r *= r * r;
					} else if (pow === 3) {
						r *= r * r * r;
					} else if (pow === 4) {
						r *= r * r * r * r;
					}

					if (type === 1) {
						this.ratio = 1 - r;
					} else if (type === 2) {
						this.ratio = r;
					} else if (time / duration < 0.5) {
						this.ratio = r / 2;
					} else {
						this.ratio = 1 - (r / 2);
					}

				} else {
					this.ratio = this._ease.getRatio(time / duration);
				}
			}

			if (this._time === prevTime && !force) {
				return;
			} else if (!this._initted) {
				this._init();
				if (!this._initted || this._gc) { //immediateRender tweens typically won't initialize until the playhead advances (_time is greater than 0) in order to ensure that overwriting occurs properly. Also, if all of the tweening properties have been overwritten (which would cause _gc to be true, as set in _init()), we shouldn't continue otherwise an onStart callback could be called for example.
					return;
				} else if (!force && this._firstPT && ((this.vars.lazy !== false && this._duration) || (this.vars.lazy && !this._duration))) {
					this._time = this._totalTime = prevTime;
					this._rawPrevTime = prevRawPrevTime;
					_lazyTweens.push(this);
					this._lazy = [time, suppressEvents];
					return;
				}
				//_ease is initially set to defaultEase, so now that init() has run, _ease is set properly and we need to recalculate the ratio. Overall this is faster than using conditional logic earlier in the method to avoid having to set ratio twice because we only init() once but renderTime() gets called VERY frequently.
				if (this._time && !isComplete) {
					this.ratio = this._ease.getRatio(this._time / duration);
				} else if (isComplete && this._ease._calcEnd) {
					this.ratio = this._ease.getRatio((this._time === 0) ? 0 : 1);
				}
			}
			if (this._lazy !== false) { //in case a lazy render is pending, we should flush it because the new render is occurring now (imagine a lazy tween instantiating and then immediately the user calls tween.seek(tween.duration()), skipping to the end - the end render would be forced, and then if we didn't flush the lazy render, it'd fire AFTER the seek(), rendering it at the wrong time.
				this._lazy = false;
			}
			if (!this._active) if (!this._paused && this._time !== prevTime && time >= 0) {
				this._active = true;  //so that if the user renders a tween (as opposed to the timeline rendering it), the timeline is forced to re-render and align it with the proper time/frame on the next rendering cycle. Maybe the tween already finished but the user manually re-renders it as halfway done.
			}
			if (prevTime === 0) {
				if (this._startAt) {
					if (time >= 0) {
						this._startAt.render(time, suppressEvents, force);
					} else if (!callback) {
						callback = "_dummyGS"; //if no callback is defined, use a dummy value just so that the condition at the end evaluates as true because _startAt should render AFTER the normal render loop when the time is negative. We could handle this in a more intuitive way, of course, but the render loop is the MOST important thing to optimize, so this technique allows us to avoid adding extra conditional logic in a high-frequency area.
					}
				}
				if (this.vars.onStart) if (this._time !== 0 || duration === 0) if (!suppressEvents) {
					this.vars.onStart.apply(this.vars.onStartScope || this, this.vars.onStartParams || _blankArray);
				}
			}
			pt = this._firstPT;
			while (pt) {
				if (pt.f) {
					pt.t[pt.p](pt.c * this.ratio + pt.s);
				} else {
					pt.t[pt.p] = pt.c * this.ratio + pt.s;
				}
				pt = pt._next;
			}

			if (this._onUpdate) {
				if (time < 0) if (this._startAt && time !== -0.0001) { //if the tween is positioned at the VERY beginning (_startTime 0) of its parent timeline, it's illegal for the playhead to go back further, so we should not render the recorded startAt values.
					this._startAt.render(time, suppressEvents, force); //note: for performance reasons, we tuck this conditional logic inside less traveled areas (most tweens don't have an onUpdate). We'd just have it at the end before the onComplete, but the values should be updated before any onUpdate is called, so we ALSO put it here and then if it's not called, we do so later near the onComplete.
				}
				if (!suppressEvents) if (this._time !== prevTime || isComplete) {
					this._onUpdate.apply(this.vars.onUpdateScope || this, this.vars.onUpdateParams || _blankArray);
				}
			}
			if (callback) if (!this._gc || force) { //check _gc because there's a chance that kill() could be called in an onUpdate
				if (time < 0 && this._startAt && !this._onUpdate && time !== -0.0001) { //-0.0001 is a special value that we use when looping back to the beginning of a repeated TimelineMax, in which case we shouldn't render the _startAt values.
					this._startAt.render(time, suppressEvents, force);
				}
				if (isComplete) {
					if (this._timeline.autoRemoveChildren) {
						this._enabled(false, false);
					}
					this._active = false;
				}
				if (!suppressEvents && this.vars[callback]) {
					this.vars[callback].apply(this.vars[callback + "Scope"] || this, this.vars[callback + "Params"] || _blankArray);
				}
				if (duration === 0 && this._rawPrevTime === _tinyNum && rawPrevTime !== _tinyNum) { //the onComplete or onReverseComplete could trigger movement of the playhead and for zero-duration tweens (which must discern direction) that land directly back on their start time, we don't want to fire again on the next render. Think of several addPause()'s in a timeline that forces the playhead to a certain spot, but what if it's already paused and another tween is tweening the "time" of the timeline? Each time it moves [forward] past that spot, it would move back, and since suppressEvents is true, it'd reset _rawPrevTime to _tinyNum so that when it begins again, the callback would fire (so ultimately it could bounce back and forth during that tween). Again, this is a very uncommon scenario, but possible nonetheless.
					this._rawPrevTime = 0;
				}
			}
		};

		p._kill = function(vars, target, overwritingTween) {
			if (vars === "all") {
				vars = null;
			}
			if (vars == null) if (target == null || target === this.target) {
				this._lazy = false;
				return this._enabled(false, false);
			}
			target = (typeof(target) !== "string") ? (target || this._targets || this.target) : TweenLite.selector(target) || target;
			var i, overwrittenProps, p, pt, propLookup, changed, killProps, record, killed;
			if ((_isArray(target) || _isSelector(target)) && typeof(target[0]) !== "number") {
				i = target.length;
				while (--i > -1) {
					if (this._kill(vars, target[i])) {
						changed = true;
					}
				}
			} else {
				if (this._targets) {
					i = this._targets.length;
					while (--i > -1) {
						if (target === this._targets[i]) {
							propLookup = this._propLookup[i] || {};
							this._overwrittenProps = this._overwrittenProps || [];
							overwrittenProps = this._overwrittenProps[i] = vars ? this._overwrittenProps[i] || {} : "all";
							break;
						}
					}
				} else if (target !== this.target) {
					return false;
				} else {
					propLookup = this._propLookup;
					overwrittenProps = this._overwrittenProps = vars ? this._overwrittenProps || {} : "all";
				}

				if (propLookup) {
					killProps = vars || propLookup;
					record = (vars !== overwrittenProps && overwrittenProps !== "all" && vars !== propLookup && (typeof(vars) !== "object" || !vars._tempKill)); //_tempKill is a super-secret way to delete a particular tweening property but NOT have it remembered as an official overwritten property (like in BezierPlugin)
					if (overwritingTween && (TweenLite.onOverwrite || this.vars.onOverwrite)) {
						for (p in killProps) {
							if (propLookup[p]) {
								if (!killed) {
									killed = [];
								}
								killed.push(p);
							}
						}
						if (!_onOverwrite(this, overwritingTween, target, killed)) { //if the onOverwrite returned false, that means the user wants to override the overwriting (cancel it).
							return false;
						}
					}

					for (p in killProps) {
						if ((pt = propLookup[p])) {
							if (pt.pg && pt.t._kill(killProps)) {
								changed = true; //some plugins need to be notified so they can perform cleanup tasks first
							}
							if (!pt.pg || pt.t._overwriteProps.length === 0) {
								if (pt._prev) {
									pt._prev._next = pt._next;
								} else if (pt === this._firstPT) {
									this._firstPT = pt._next;
								}
								if (pt._next) {
									pt._next._prev = pt._prev;
								}
								pt._next = pt._prev = null;
							}
							delete propLookup[p];
						}
						if (record) {
							overwrittenProps[p] = 1;
						}
					}
					if (!this._firstPT && this._initted) { //if all tweening properties are killed, kill the tween. Without this line, if there's a tween with multiple targets and then you killTweensOf() each target individually, the tween would technically still remain active and fire its onComplete even though there aren't any more properties tweening.
						this._enabled(false, false);
					}
				}
			}
			return changed;
		};

		p.invalidate = function() {
			if (this._notifyPluginsOfEnabled) {
				TweenLite._onPluginEvent("_onDisable", this);
			}
			this._firstPT = this._overwrittenProps = this._startAt = this._onUpdate = null;
			this._notifyPluginsOfEnabled = this._active = this._lazy = false;
			this._propLookup = (this._targets) ? {} : [];
			Animation.prototype.invalidate.call(this);
			if (this.vars.immediateRender) {
				this._time = -_tinyNum; //forces a render without having to set the render() "force" parameter to true because we want to allow lazying by default (using the "force" parameter always forces an immediate full render)
				this.render(-this._delay);
			}
			return this;
		};

		p._enabled = function(enabled, ignoreTimeline) {
			if (!_tickerActive) {
				_ticker.wake();
			}
			if (enabled && this._gc) {
				var targets = this._targets,
					i;
				if (targets) {
					i = targets.length;
					while (--i > -1) {
						this._siblings[i] = _register(targets[i], this, true);
					}
				} else {
					this._siblings = _register(this.target, this, true);
				}
			}
			Animation.prototype._enabled.call(this, enabled, ignoreTimeline);
			if (this._notifyPluginsOfEnabled) if (this._firstPT) {
				return TweenLite._onPluginEvent((enabled ? "_onEnable" : "_onDisable"), this);
			}
			return false;
		};


//----TweenLite static methods -----------------------------------------------------

		TweenLite.to = function(target, duration, vars) {
			return new TweenLite(target, duration, vars);
		};

		TweenLite.from = function(target, duration, vars) {
			vars.runBackwards = true;
			vars.immediateRender = (vars.immediateRender != false);
			return new TweenLite(target, duration, vars);
		};

		TweenLite.fromTo = function(target, duration, fromVars, toVars) {
			toVars.startAt = fromVars;
			toVars.immediateRender = (toVars.immediateRender != false && fromVars.immediateRender != false);
			return new TweenLite(target, duration, toVars);
		};

		TweenLite.delayedCall = function(delay, callback, params, scope, useFrames) {
			return new TweenLite(callback, 0, {delay:delay, onComplete:callback, onCompleteParams:params, onCompleteScope:scope, onReverseComplete:callback, onReverseCompleteParams:params, onReverseCompleteScope:scope, immediateRender:false, lazy:false, useFrames:useFrames, overwrite:0});
		};

		TweenLite.set = function(target, vars) {
			return new TweenLite(target, 0, vars);
		};

		TweenLite.getTweensOf = function(target, onlyActive) {
			if (target == null) { return []; }
			target = (typeof(target) !== "string") ? target : TweenLite.selector(target) || target;
			var i, a, j, t;
			if ((_isArray(target) || _isSelector(target)) && typeof(target[0]) !== "number") {
				i = target.length;
				a = [];
				while (--i > -1) {
					a = a.concat(TweenLite.getTweensOf(target[i], onlyActive));
				}
				i = a.length;
				//now get rid of any duplicates (tweens of arrays of objects could cause duplicates)
				while (--i > -1) {
					t = a[i];
					j = i;
					while (--j > -1) {
						if (t === a[j]) {
							a.splice(i, 1);
						}
					}
				}
			} else {
				a = _register(target).concat();
				i = a.length;
				while (--i > -1) {
					if (a[i]._gc || (onlyActive && !a[i].isActive())) {
						a.splice(i, 1);
					}
				}
			}
			return a;
		};

		TweenLite.killTweensOf = TweenLite.killDelayedCallsTo = function(target, onlyActive, vars) {
			if (typeof(onlyActive) === "object") {
				vars = onlyActive; //for backwards compatibility (before "onlyActive" parameter was inserted)
				onlyActive = false;
			}
			var a = TweenLite.getTweensOf(target, onlyActive),
				i = a.length;
			while (--i > -1) {
				a[i]._kill(vars, target);
			}
		};



/*
 * ----------------------------------------------------------------
 * TweenPlugin   (could easily be split out as a separate file/class, but included for ease of use (so that people don't need to include another script call before loading plugins which is easy to forget)
 * ----------------------------------------------------------------
 */
		var TweenPlugin = _class("plugins.TweenPlugin", function(props, priority) {
					this._overwriteProps = (props || "").split(",");
					this._propName = this._overwriteProps[0];
					this._priority = priority || 0;
					this._super = TweenPlugin.prototype;
				}, true);

		p = TweenPlugin.prototype;
		TweenPlugin.version = "1.10.1";
		TweenPlugin.API = 2;
		p._firstPT = null;

		p._addTween = function(target, prop, start, end, overwriteProp, round) {
			var c, pt;
			if (end != null && (c = (typeof(end) === "number" || end.charAt(1) !== "=") ? Number(end) - start : parseInt(end.charAt(0) + "1", 10) * Number(end.substr(2)))) {
				this._firstPT = pt = {_next:this._firstPT, t:target, p:prop, s:start, c:c, f:(typeof(target[prop]) === "function"), n:overwriteProp || prop, r:round};
				if (pt._next) {
					pt._next._prev = pt;
				}
				return pt;
			}
		};

		p.setRatio = function(v) {
			var pt = this._firstPT,
				min = 0.000001,
				val;
			while (pt) {
				val = pt.c * v + pt.s;
				if (pt.r) {
					val = Math.round(val);
				} else if (val < min) if (val > -min) { //prevents issues with converting very small numbers to strings in the browser
					val = 0;
				}
				if (pt.f) {
					pt.t[pt.p](val);
				} else {
					pt.t[pt.p] = val;
				}
				pt = pt._next;
			}
		};

		p._kill = function(lookup) {
			var a = this._overwriteProps,
				pt = this._firstPT,
				i;
			if (lookup[this._propName] != null) {
				this._overwriteProps = [];
			} else {
				i = a.length;
				while (--i > -1) {
					if (lookup[a[i]] != null) {
						a.splice(i, 1);
					}
				}
			}
			while (pt) {
				if (lookup[pt.n] != null) {
					if (pt._next) {
						pt._next._prev = pt._prev;
					}
					if (pt._prev) {
						pt._prev._next = pt._next;
						pt._prev = null;
					} else if (this._firstPT === pt) {
						this._firstPT = pt._next;
					}
				}
				pt = pt._next;
			}
			return false;
		};

		p._roundProps = function(lookup, value) {
			var pt = this._firstPT;
			while (pt) {
				if (lookup[this._propName] || (pt.n != null && lookup[ pt.n.split(this._propName + "_").join("") ])) { //some properties that are very plugin-specific add a prefix named after the _propName plus an underscore, so we need to ignore that extra stuff here.
					pt.r = value;
				}
				pt = pt._next;
			}
		};

		TweenLite._onPluginEvent = function(type, tween) {
			var pt = tween._firstPT,
				changed, pt2, first, last, next;
			if (type === "_onInitAllProps") {
				//sorts the PropTween linked list in order of priority because some plugins need to render earlier/later than others, like MotionBlurPlugin applies its effects after all x/y/alpha tweens have rendered on each frame.
				while (pt) {
					next = pt._next;
					pt2 = first;
					while (pt2 && pt2.pr > pt.pr) {
						pt2 = pt2._next;
					}
					if ((pt._prev = pt2 ? pt2._prev : last)) {
						pt._prev._next = pt;
					} else {
						first = pt;
					}
					if ((pt._next = pt2)) {
						pt2._prev = pt;
					} else {
						last = pt;
					}
					pt = next;
				}
				pt = tween._firstPT = first;
			}
			while (pt) {
				if (pt.pg) if (typeof(pt.t[type]) === "function") if (pt.t[type]()) {
					changed = true;
				}
				pt = pt._next;
			}
			return changed;
		};

		TweenPlugin.activate = function(plugins) {
			var i = plugins.length;
			while (--i > -1) {
				if (plugins[i].API === TweenPlugin.API) {
					_plugins[(new plugins[i]())._propName] = plugins[i];
				}
			}
			return true;
		};

		//provides a more concise way to define plugins that have no dependencies besides TweenPlugin and TweenLite, wrapping common boilerplate stuff into one function (added in 1.9.0). You don't NEED to use this to define a plugin - the old way still works and can be useful in certain (rare) situations.
		_gsDefine.plugin = function(config) {
			if (!config || !config.propName || !config.init || !config.API) { throw "illegal plugin definition."; }
			var propName = config.propName,
				priority = config.priority || 0,
				overwriteProps = config.overwriteProps,
				map = {init:"_onInitTween", set:"setRatio", kill:"_kill", round:"_roundProps", initAll:"_onInitAllProps"},
				Plugin = _class("plugins." + propName.charAt(0).toUpperCase() + propName.substr(1) + "Plugin",
					function() {
						TweenPlugin.call(this, propName, priority);
						this._overwriteProps = overwriteProps || [];
					}, (config.global === true)),
				p = Plugin.prototype = new TweenPlugin(propName),
				prop;
			p.constructor = Plugin;
			Plugin.API = config.API;
			for (prop in map) {
				if (typeof(config[prop]) === "function") {
					p[map[prop]] = config[prop];
				}
			}
			Plugin.version = config.version;
			TweenPlugin.activate([Plugin]);
			return Plugin;
		};


		//now run through all the dependencies discovered and if any are missing, log that to the console as a warning. This is why it's best to have TweenLite load last - it can check all the dependencies for you.
		a = window._gsQueue;
		if (a) {
			for (i = 0; i < a.length; i++) {
				a[i]();
			}
			for (p in _defLookup) {
				if (!_defLookup[p].func) {
					window.console.log("GSAP encountered missing dependency: com.greensock." + p);
				}
			}
		}

		_tickerActive = false; //ensures that the first official animation forces a ticker.tick() to update the time when it is instantiated

})((typeof(module) !== "undefined" && module.exports && typeof(global) !== "undefined") ? global : this || window, "TweenLite");
;(function() {
  var zen = function(string) {
    var replace = {
        '\\[([a-zA-Z\\-]+)=([^\\]]+)\\]': function(parsed) {
          var props = {};
          props[parsed[1]] = parsed[2];

          return props;
        },
        '#([a-zA-Z][a-zA-Z0-9\\-_]*)': function(parsed) {
          return { 'id': parsed[1] }
        },
        '\\.([a-zA-Z][a-zA-Z0-9\\-_]*)': function(parsed) {
          return { 'class': parsed[1] }
        }
      },
      props = {};

    new MooTools.Hash(replace).each(function(parser, regexp) {
      var match;

      regexp = new RegExp(regexp);

      while(string.test(regexp)) {
        match = string.match(regexp);

        MooTools.$extend(props, parser(match));
        string = string.replace(match[0], '');
      }
    })

    return [string, props];
  };

  var microjungle = function(template) {

    // they just doing their job.
    var monkeys =  function(what, who) {
      what.each(function(j) {

        if (!j) {
          return;
        }

        if (typeof j == 'string') {
          who.innerHTML += j;
        }
        else if (typeof j[0] == 'string') {
          var parsed_zen = zen(j.shift()),
              el = new MooTools.Element(parsed_zen[0]),
              attrs = {}.toString.call(j[0]) === '[object Object]' ? j.shift() : {};

          MooTools.$extend(attrs, parsed_zen[1]);

          attrs && el.setProperties(attrs);
          who.appendChild(monkeys(j, el));
        }
        else if (j.nodeType === 1 || j.nodeType === 11) {
          who.appendChild(j);
        }
        else {
          monkeys(j, who);
        }
      })

      return who;
    };

    return monkeys(template, document.createDocumentFragment());
  };

  MooTools.Element.implement({

    microjungle: function(template) {
      this.appendChild(microjungle(template));

      return this;
    }
  })
  $.microjungle = microjungle

})();

/**
 * lscache library
 * Copyright (c) 2011, Pamela Fox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*jshint undef:true, browser:true */

/* global MooTools */

/**
 * Creates a namespace for the lscache functions.
 */
var lscache = function() {

  // Prefix for all lscache keys
  var CACHE_PREFIX = 'lscache-';

  // Suffix for the key name on the expiration items in localStorage
  var CACHE_SUFFIX = '-cacheexpiration';

  // expiration date radix (set to Base-36 for most space savings)
  var EXPIRY_RADIX = 10;

  // time resolution in minutes
  var EXPIRY_UNITS = 60 * 1000;

  // ECMAScript max Date (epoch + 1e8 days)
  var MAX_DATE = Math.floor(8.64e15/EXPIRY_UNITS);

  var cachedStorage;
  var cachedJSON;
  var cacheBucket = '';

  // Determines if localStorage is supported in the browser;
  // result is cached for better performance instead of being run each time.
  // Feature detection is based on how Modernizr does it;
  // it's not straightforward due to FF4 issues.
  // It's not run at parse-time as it takes 200ms in Android.
  function supportsStorage() {
    var key = '__lscachetest__';
    var value = key;

    if (cachedStorage !== undefined) {
      return cachedStorage;
    }

    try {
      setItem(key, value);
      removeItem(key);
      cachedStorage = true;
    } catch (exc) {
      cachedStorage = false;
    }
    return cachedStorage;
  }

  // Determines if native JSON (de-)serialization is supported in the browser.
  function supportsJSON() {
    /*jshint eqnull:true */
    if (cachedJSON === undefined) {
      cachedJSON = (window.MooTools.JSON != null);
    }
    return cachedJSON;
  }

  /**
   * Returns the full string for the localStorage expiration item.
   * @param {String} key
   * @return {string}
   */
  function expirationKey(key) {
    return key + CACHE_SUFFIX;
  }

  /**
   * Returns the number of minutes since the epoch.
   * @return {number}
   */
  function currentTime() {
    return Math.floor((new Date().getTime())/EXPIRY_UNITS);
  }

  /**
   * Wrapper functions for localStorage methods
   */

  function getItem(key) {
    return localStorage.getItem(CACHE_PREFIX + cacheBucket + key);
  }

  function setItem(key, value) {
    // Fix for iPad issue - sometimes throws QUOTA_EXCEEDED_ERR on setItem.
    localStorage.removeItem(CACHE_PREFIX + cacheBucket + key);
    localStorage.setItem(CACHE_PREFIX + cacheBucket + key, value);
  }

  function removeItem(key) {
    localStorage.removeItem(CACHE_PREFIX + cacheBucket + key);
  }

  /**
   * Stores the value in localStorage. Expires after specified number of minutes.
   * @param {string} key
   * @param {Object|string} value
   * @param {number} time
   */
  function set(key, value, time) {
    if (!supportsStorage()) return;

    // If we don't get a string value, try to stringify
    // In future, localStorage may properly support storing non-strings
    // and this can be removed.
    if (typeof value !== 'string') {
      if (!supportsJSON()) return;
      try {
        value = MooTools.JSON.stringify(value);
      } catch (e) {
        // Sometimes we can't stringify due to circular refs
        // in complex objects, so we won't bother storing then.
        return;
      }
    }

    try {
      setItem(key, value);
    } catch (e) {
      if (e.name === 'QUOTA_EXCEEDED_ERR' || e.name === 'NS_ERROR_DOM_QUOTA_REACHED') {
        // If we exceeded the quota, then we will sort
        // by the expire time, and then remove the N oldest
        var storedKeys = [];
        var storedKey;
        for (var i = 0; i < localStorage.length; i++) {
          storedKey = localStorage.key(i);

          if (storedKey.indexOf(CACHE_PREFIX + cacheBucket) === 0 && storedKey.indexOf(CACHE_SUFFIX) < 0) {
            var mainKey = storedKey.substr((CACHE_PREFIX + cacheBucket).length);
            var exprKey = expirationKey(mainKey);
            var expiration = getItem(exprKey);
            if (expiration) {
              expiration = parseInt(expiration, EXPIRY_RADIX);
            } else {
              // TODO: Store date added for non-expiring items for smarter removal
              expiration = MAX_DATE;
            }
            storedKeys.push({
              key: mainKey,
              size: (getItem(mainKey)||'').length,
              expiration: expiration
            });
          }
        }
        // Sorts the keys with oldest expiration time last
        storedKeys.sort(function(a, b) { return (b.expiration-a.expiration); });

        var targetSize = (value||'').length;
        while (storedKeys.length && targetSize > 0) {
          storedKey = storedKeys.pop();
          removeItem(storedKey.key);
          removeItem(expirationKey(storedKey.key));
          targetSize -= storedKey.size;
        }
        try {
          setItem(key, value);
        } catch (e) {
          // value may be larger than total quota
          return;
        }
      } else {
        // If it was some other error, just give up.
        return;
      }
    }

    // If a time is specified, store expiration info in localStorage
    if (time) {
      setItem(expirationKey(key), (currentTime() + time).toString(EXPIRY_RADIX));
    } else {
      // In case they previously set a time, remove that info from localStorage.
      removeItem(expirationKey(key));
    }
  }

  /**
   * Retrieves specified value from localStorage, if not expired.
   * @param {string} key
   * @return {string|Object}
   */
  function get(key) {
    if (!supportsStorage()) return null;

    // Return the de-serialized item if not expired
    var exprKey = expirationKey(key);
    var expr = getItem(exprKey);

    if (expr) {
      var expirationTime = parseInt(expr, EXPIRY_RADIX);

      // Check if we should actually kick item out of storage
      if (currentTime() >= expirationTime) {
        removeItem(key);
        removeItem(exprKey);
        return null;
      }
    }

    // Tries to de-serialize stored value if its an object, and returns the normal value otherwise.
    var value = getItem(key);
    if (!value || !supportsJSON()) {
      return value;
    }

    try {
      // We can't tell if its JSON or a string, so we try to parse
      return MooTools.JSON.parse(value);
    } catch (e) {
      // If we can't parse, it's probably because it isn't an object
      return value;
    }
  }

  /**
   * Removes a value from localStorage.
   * Equivalent to 'delete' in memcache, but that's a keyword in JS.
   * @param {string} key
   */
  function remove(key) {
    if (!supportsStorage()) return null;
    removeItem(key);
    removeItem(expirationKey(key));
  }

  return {
    set: set,
    get: get,
    remove: remove,

    setItem: set,
    getItem: get,
    removeItem: remove,

    /**
     * Returns whether local storage is supported.
     * Currently exposed for testing purposes.
     * @return {boolean}
     */
    supported: function() {
      return supportsStorage();
    },

    /**
     * Flushes all lscache items and expiry markers without affecting rest of localStorage
     */
    flush: function() {
      if (!supportsStorage()) return;

      // Loop in reverse as removing items will change indices of tail
      for (var i = localStorage.length-1; i >= 0 ; --i) {
        var key = localStorage.key(i);
        if (key.indexOf(CACHE_PREFIX + cacheBucket) === 0) {
          localStorage.removeItem(key);
        }
      }
    },
    
    /**
     * Appends CACHE_PREFIX so lscache will partition data in to different buckets.
     * @param {string} bucket
     */
    setBucket: function(bucket) {
      cacheBucket = bucket;
    },
    
    /**
     * Calls callback to access data for given bucket
     * 
     * @param {string} bucket
     * @param {function} callback
     * @return {mixed}
     */
    bucket: function(bucket, callback) {
      var old_bucket = cacheBucket,
          result;
          
      if(old_bucket == bucket) {
        return callback();
      }
      
      this.setBucket(bucket);
      result = callback();
      this.setBucket(old_bucket);
      
      return result;
    },
    
    /**
     * Resets the string being appended to CACHE_PREFIX so lscache will use the default storage behavior.
     */
    resetBucket: function() {
      cacheBucket = '';
    },

  };
}();

var hashuri = (function() {
  var hash_map = {},
      raw_hash,
      map = MooTools.Array.prototype.map,
      each = MooTools.Array.prototype.forEach;
  
  if(!each) {
    each = function(callback) {
          
      for(var i = 0; i < this.length; i++) {
        callback(this[i], i);
      }
    }
  }
  
  if(!map) {
    map = function(callback) {
      var mapped = [];
         
      each.call(this, function(value) {
        mapped.push(callback(value));
      })
      
      return mapped;
    }
  }
  
  var create_hash_map = function() {
    var hash;
    
    if(document.location.hash == raw_hash) {
      return;
    }
    
    raw_hash = document.location.hash;
    hash = raw_hash.replace(/^#\??/, '');
    hash_map = {};
    
    if(!hash) {
      return;
    }
    
    map.call(hash.split('&'), function(arg) {
      var parts = arg.split('=');

      hash_map[parts[0]] = parts[1] ? decodeURI(parts[1]) : null;
    });
  }
  
  var to_uri = function(params) {
    var hash_uri_parts = [],
        key;
      
    for(key in params) {
      params.hasOwnProperty(key) && hash_uri_parts.push(key + '=' + encodeURI(params[key]));
    }

    return '#?' + hash_uri_parts.join('&');
  }
  
  var update_location = function() {
    document.location = to_uri(hash_map);
  }
  
  create_hash_map();
  
  // support for legacy IEs
  (window.attachEvent || window.addEventListener)('hashchange', create_hash_map);
  
  return {
    
    toUri: to_uri,
    
    get: function(param) {
      return hash_map[param];
    },
    
    set: function(param, value) {
      
      if('[object Object]' == Object.prototype.toString.call(param)) {
        hash_map = param;
      }
      else {
        
        if(undefined === value) {
          delete hash_map[param];
        }
        else {
          hash_map[param] = value;
        }
      }
      
      update_location();
    }
  }
})();

/* globals AnimationEditor */
/* globals DPMainToolbar_Widget_Anchor */
/* globals DPMainToolbar_Widget_Button */
/* globals DPMainToolbar_Widget_ButtonGroup */
/* globals DPMainToolbar_Widget_ButtonToolbar */
/* globals DPMainToolbar_Widget_Button_Align */
/* globals DPMainToolbar_Widget_Button_Configure */
/* globals DPMainToolbar_Widget_Button_Group */
/* globals DPMainToolbar_Widget_Button_Image */
/* globals DPMainToolbar_Widget_Button_Iterations */
/* globals DPMainToolbar_Widget_Button_LayersPanel */
/* globals DPMainToolbar_Widget_Button_LayersPanelUnavailable */
/* globals DPMainToolbar_Widget_Button_Link */
/* globals DPMainToolbar_Widget_Button_Lock */
/* globals DPMainToolbar_Widget_Button_Preview */
/* globals DPMainToolbar_Widget_Button_UXPatterns */
/* globals DPMainToolbar_Widget_Button_UXPatternsUnavailable */
/* globals DPMainToolbar_Widget_Height */
/* globals DPMainToolbar_Widget_Opacity */
/* globals DPMainToolbar_Widget_Page */
/* globals DPMainToolbar_Widget_PositionX */
/* globals DPMainToolbar_Widget_PositionY */
/* globals DPMainToolbar_Widget_Rotation */
/* globals DPMainToolbar_Widget_Toolbar_Arrange */
/* globals DPMainToolbar_Widget_UndoRedo */
/* globals DPMainToolbar_Widget_Width */
/* globals DPWorkBench */
/* globals FEATURE_LAYERS */
/* globals FEATURE_UXPATTERNS */
/* globals FEATURE_UXPIN30_EDITOR_UI */
/* globals ShareProjectModal */
/* globals dmsDPPropertyToolbar_Toolbar_Core */
/* globals dpManager */
/* globals uxpft */
/* jscs: disable requireCamelCaseOrUpperCaseIdentifiers, requireCapitalizedConstructors */

/**
 * Tworzy toolbar z itemami ktorych zawartosc moze zmieniac sie podczas odswiezania ich (metoda refresh)
 *
 * Toolbar podczas odswiezania wywoluje metode refresh() na kazdym z elementow a ten decyduje czy cos powinno sie zmienic
 * (np. ikona moze pozostac aktywa lub deaktywowac sie)
 *
 */
var DPMainToolbar = new MooTools.Class({

  mItems: [],

  mParentEl: document.body,

  mHistoryWidget: null,

  mShareModal: null,

  initialize: function() {
    this.mParentEl = DPWorkBench.getMainContainer();

    // sorry, nie wiem jak to inaczej zrobic :(
    this.pageWidget = new DPMainToolbar_Widget_Page();
    this.link_button =  new DPMainToolbar_Widget_Button_Link();
    this.mHistoryWidget = new DPMainToolbar_Widget_UndoRedo();

    // tools widget group
    var toolsWidgetGroup = [];
    var action_widget_group = [];

    toolsWidgetGroup.push(
      this.mHistoryWidget,
      new DPMainToolbar_Widget_Button_Group(),
      new DPMainToolbar_Widget_ButtonToolbar({
        text: 'Arrange',
        tooltip: 'Arrange',
        class_name: 'arrange',
        button_style: 'inactive only-icon icon-arrange',
        onRefresh: function() {
          var lockedStates = dpManager.dpWorkBench.getActivePage().getActiveElements()
            .map(function(pEl) {
              return pEl.getProperty('locked');
            })
            .unique();

          // button jest aktywny gdy zaznaczone elementy nie sa zablokowane
          if (lockedStates.length > 1 || (lockedStates.length === 1 && lockedStates.pop() === 0)) {
            this.mTextEl.removeClass('inactive');
          } else {
            this.mTextEl.addClass('inactive');
          }
        },

        showToolbar: function(pEl) {
          if (this.mTextEl.hasClass('inactive')) {
            return;
          }

          this.toolbar = new dmsDPPropertyToolbar_Toolbar_Core(pEl, null, {
            position: 'bottom',
            container: dpManager.dpMainToolbar.getEl(),
            class_name: 'arrange',
            offset: 7,
            onDestroy: function() {
              this.mTextEl.removeClass('pushed');
              this.toolbar = null;
            }.bind(this),
          });

          this.toolbar.addWidget(new DPMainToolbar_Widget_Toolbar_Arrange());
          this.toolbar.toggle(true);
          this.mTextEl.addClass('pushed');
        },
      }),
      new DPMainToolbar_Widget_Button_Align(),
      this.link_button,
      new DPMainToolbar_Widget_Button_Lock(),
      new DPMainToolbar_Widget_Button_Image()
    );

    // Layers button
    if (uxpft.enabled(FEATURE_LAYERS)) {
      toolsWidgetGroup.push(new DPMainToolbar_Widget_Button_LayersPanel());
    }

    // Layers teaser
    else if (uxpft.disabledInPlan(FEATURE_LAYERS)) {
      toolsWidgetGroup.push(new DPMainToolbar_Widget_Button_LayersPanelUnavailable());
    }

    action_widget_group.push(
      new DPMainToolbar_Widget_Button_Iterations(),
      new DPMainToolbar_Widget_Button_Configure(),
      this.createShareButton(),
      new DPMainToolbar_Widget_Button_Preview()
    );

    this.mItems = [
      this.pageWidget,
      new DPMainToolbar_Widget_ButtonGroup('tools', toolsWidgetGroup),
      new DPMainToolbar_Widget_ButtonGroup('parameters', [
        new DPMainToolbar_Widget_Anchor(),
        new DPMainToolbar_Widget_Width(),
        new DPMainToolbar_Widget_Height(),
        new DPMainToolbar_Widget_PositionX(),
        new DPMainToolbar_Widget_PositionY(),
        new DPMainToolbar_Widget_Rotation(),
        new DPMainToolbar_Widget_Opacity(),
      ]),
      new DPMainToolbar_Widget_ButtonGroup('action', action_widget_group),
    ];

    this.render();

    // kiedy tworzy sie DPMainToolbar nie ma jeszcze zaladowanej strony
    // dopiero po jej zaladowaniu mozna podpiac event onActivateElements
    dpManager.loaded().then(function() {
      'use strict';
      dpManager.dpWorkBench.addEvent('onPageLoaded', function() {
        dpManager.dpWorkBench.getActivePage().addEvent('onActivateElements', this.refresh.bind(this));
      }.bind(this));

      dpManager.animationEditor.on(AnimationEditor.ON_START, function() {
        this.pageWidget.lock();
      }.bind(this));

      dpManager.animationEditor.on(AnimationEditor.ON_FINISH, function() {
        this.pageWidget.unlock();
      }.bind(this));
    }.bind(this));


  },

  hideToolbars: function() {
    this.mItems.forEach(function(value) {
      if (value instanceof DPMainToolbar_Widget_ButtonGroup) {
        value.mItems.forEach(function(value) {
          if (value instanceof DPMainToolbar_Widget_ButtonToolbar && value.isToolbarOpen()) {
            value.hideToolbar();
          }
        });
      }
    });
  },

  /**
   * Renderuje glowny element toolbara oraz elementy takie jak przycisk dropdowna, klawisze Share, HTML
   *
   */
  render: function() {
    this.mEl = new MooTools.Element('div', { id: 'DPMainToolbar' });

    this.mItems.each(function(pItem) {
      pItem.render(this.mEl);
    }.bind(this));

    this.mItems.forEach(function(value) {
      if (value instanceof DPMainToolbar_Widget_ButtonGroup) {
        value.mItems.forEach(function(value) {
          if (value instanceof DPMainToolbar_Widget_ButtonToolbar) {
            value.addEvent(DPMainToolbar_Widget_ButtonToolbar.OPEN_TOOLBAR, this.hideToolbars.bind(this));
          } else if (value instanceof DPMainToolbar_Widget_Button) {
            value.addEvent('click', this.hideToolbars.bind(this));
          }
        }.bind(this));
      }
    }.bind(this));

    this.mEl.inject(this.mParentEl);
    this.refresh();

    if (document.getElement('#trial-info')) {
      this.mEl.setStyle('top', 25);
    }

    return this;
  },

  getEl: function() {
    return this.mEl;
  },

  /**
   * Wysyla sygnal odswiezenia do kazdego elementu toolbara (Item)
   *
   * @param hash pProps wlasciwosci elementu ktory np. border-width, width, height itp.
   */
  refresh: function(pProps) {
    if (dpManager.isMppEditor) {
      return;
    }

    this.mItems.each(function(pItem) {
      pItem.refresh(pProps);
    }.bind(this));

    return this;
  },

  createShareButton: function() {
    return new DPMainToolbar_Widget_ButtonToolbar({
      text: 'Share',
      class_name: 'sharing',
      button_style: 'only-icon-font icon-font-sharing',
      tooltip: 'Share (Preview, Share by email, Download as HTML)',
      tooltip_position: 'top-right-arrow',
      showToolbar: this.showShareModal.bind(this, 'share-project-template'),
    });
  },

  showShareModal: function(templateId) {
    var DEFAULT_TEMPLATE_ID = 'share-project-template';
    var template = document.getElementById(templateId || DEFAULT_TEMPLATE_ID).innerHTML;

    this.mShareModal = new ShareProjectModal({
      id: dpManager.dpProject.getIdProject(),
      name: dpManager.dpProject.getName(),
      url: dpManager.dpProject.getPreviewURL(),
      password: dpManager.dpProject.getPassword(),
      type: dpManager.dpProject.getType(),
      hash: dpManager.dpProject.getHash(),
    });

    this.mShareModal.on('passwordChanged', dpManager.dpProject.savePassword.bind(dpManager.dpProject));

    document.body.appendChild(this.mShareModal.render(template).el);
    this.mShareModal.initializeComponents();
    try {
      dpManager.Analytics.intercomEvent('Clicked Share Button');
    } catch (e) {
      console.error('event not sent', e);
    }

    setTimeout(this.mShareModal.show.bind(this.mShareModal, 1)); //Timeout for animation
  },

});

/* globals dmsDPPropertyToolbar_Toolbar_Position */
/* globals dpManager */

var dmsDPPropertyToolbar_Toolbar_Core = new MooTools.Class({

  Implements: [MooTools.Events, MooTools.Options],

  mWidgets: [],

  mRelativeEl: null,

  mEl: null,

  mDestroyRef: null,

  mPreventDestroy: false,

  mToggleRefs: {},

  mRendered: false,

  options: {
    'position': null,
    'container': null,
    'offset': null,
    'class_name': null
  },

  mPhantomMode: false,

  initialize: function(pRelativeEl, pWidgets, pOptions) {
    this.mRelativeEl = pRelativeEl;
    pOptions = pOptions || {};

    if(pWidgets) {
      pWidgets.each(this.addWidget.bind(this));
    }

    this.mDestroyRef = this.destroy.bind(this);

    if(!pOptions.container) {
      pOptions.container = dpManager.dpWorkBench.getWorkBenchEl();
    }

    this.setOptions(pOptions);
  },

  show: function() {
    this.render();
    this.setPosition();

    window.addEvent('click', this.mDestroyRef);
  },

  quickHide: function () {
    this.mEl.hide();
  },

  quickShow: function () {
    this.mEl.show();
    this.setPosition();
  },

  toggle: function(pShow) {

    if(undefined === pShow) {
      pShow = this.isActive() ? false : true;
    }

    if(!pShow) {
      this.mEl.hide();
      this.fireEvent('toggle', [false]);
    }
    else if(this.mRendered) {
      this.mEl.show();
      this.setPosition();
      this.fireEvent('toggle', [true]);
    }
    else {
      this.show();
      this.fireEvent('toggle', [true]);
    }
  },

  addWidget: function(pWidget) {
    this.mWidgets.push(pWidget);

    pWidget.setToolbar(this);
  },

  render: function() {
    this.mEl = new MooTools.Element('div', {
        'class': 'property-toolbar'
      })
      .inject(this.options.container);

    if(this.options.class_name) {
      this.mEl.addClass(this.options.class_name);
    }
    if(this.options.id_name) {
      this.mEl.set('id', this.options.id_name);
    }

    this.mEl.microjungle([
      ['span', {'class': 'arrow'}]
    ]);

    this.mWidgets.each(function(pWidget) {
      pWidget.render(this.mEl);
    }.bind(this));

    this.mRendered = true;
  },

  refresh: function() {
    this.mWidgets.each(function(pWidget){
      pWidget.refresh();
    });
  },

  isActive: function() {
    return this.mEl && this.mEl.isVisible();
  },

  destroy: function(pE) {

    if(this.mPreventDestroy) {
      return;
    }

    var elements = this.mEl.getElements('*');
    elements.push(this.mEl);
    elements.push(this.mRelativeEl);
    elements.extend(this.mRelativeEl.getElements('*'));

    if(pE && elements.contains(pE.target)) {
      return;
    }

    this.mWidgets.each(function(pWidget) {
      pWidget.destroy();
      pWidget.removeEvents();
    });

    this.mWidgets = [];
    this.mEl.destroy();
    this.cleanUp();

    // only when editing wireframe
    if (dpManager.dpWorkBench) {
      dpManager.dpWorkBench.mCurrentToolsBox = null;
    }

    this.fireEvent('destroy');
  },

  cleanUp: function() {
    window.removeEvent('click', this.mDestroyRef);

    this.mDestroyRef = null;
  },

  /**
   * Ustawia pozycje toolbara
   *
   * W zaleznosci od polozenia elementu toolbar bedzie pojawiac sie w takim miejscu aby byl widoczny.
   * Domyslnie toolbar znajduje sie pod elementem (w centrum)
   *
   * W zaleznosci od pozycji toolbarowi zostania nadana odpowiednia klasa:
   * - bottom-left - gdy jest na dole zrownany lewa krawedzia z elementem
   * - bottom-center
   * - bottom-right
   * - right-top
   * - right-center
   * - right-bottom
   * - top-right
   * - top-center
   * - top-left
   * - left-top
   * - left-center
   * - left-bottom
   */
  setPosition: function() {
    var helper = new dmsDPPropertyToolbar_Toolbar_Position(this.mEl, this.mRelativeEl, this.options.position);

    if(this.options.offset) {
      helper.setOffset(this.options.offset);
    }

    helper.position();
  },

  preventDestroy: function(pState) {
    this.mPreventDestroy = pState;
  }

});

/* globals DSWindowSkin */

var DPPropertyToolbar = new MooTools.Class({

  mToolbarIcons: {},
  
  initialize: function() {
    this.mToolbarIcons = {};
  },

  /**
   * Toolbar z ikonkami
   *
   * @param pIcons - array z nazwami obiektow-ikonek z this.mToolbarIcons
   */
  generateToolbar: function(pIcons) {
    this.mToolbar = new MooTools.Element('div', {
      'class': 'Toolbar TextareaToolbar'
    });
    pIcons.each(function(pIcon) {
    	var icon = this.mToolbarIcons[pIcon];
      
    	var link = new MooTools.Element('a', {'title': icon.name, 'class': 'ButtonActive'}).injectInside(this.mToolbar);
    	new MooTools.Element('img', {'src': DSWindowSkin.mIconBaseP + icon.icon, 'alt': icon.name}).injectInside(link).disableSelection();
    	new MooTools.Element('span').set('html', icon.name).injectInside(link);
	    link.addEvent('click', icon.action);
	    this.mToolbarIcons[pIcon]['el'] = link;
    }.bind(this));

    this.mDSWindow.mContentElement.adopt(this.mToolbar);
  }  

});

var dmsDPPropertyToolbar_Widget_Interface = new MooTools.Class({
  
  Implements: [MooTools.Events, MooTools.Options],
  
  mToolbar: null,
  
  initialize: function(pOptions) {
    this.setOptions(pOptions);
  },
  
  setToolbar: function(pToolbar) {
    this.mToolbar = pToolbar;
  },
  
  /**
   * Pobiera wartość danej właściwości. 
   * Jesli przekazano, to wykorzystuje callback (options.getValue), a jeśli nie, pobiera wartość bezposrednio z getProperty aktywnego elementu
   * 
   * @param mixed pProperty 
   */
  getValue: function(pProperty) {
    if(typeof(this.options.getValue) == 'function') {
      return this.options.getValue(pProperty);
    }
    
    return this.mToolbar.getActiveElement().getProperty(pProperty);
  },
  
  render: function(pEl) {
  },
  
  refresh: function() {
  },
  
  destroy: function() {
  }
  
});

/* globals dpManager */

var DPMainToolbar_Widget_Button = new MooTools.Class({

  Implements: [MooTools.Events, MooTools.Options],

  options: {
    'id': '',
    'text': '',
    'tooltip': '',
    'tooltip_class': '',
    'tooltip_position': '',
    'class_name': '',
    'button_style': ''
  },

  mEl: null,

  mTextEl: null,

  mTooltipEl: null,

  initialize: function(pOptions) {
    this.setOptions(pOptions);
  },

  render: function(pContainerEl) {

    this.mEl = new MooTools.Element('a', {
      'id': this.options.id,
      'class': 'btn ' + this.options.class_name,
      'data-show-node': 'tooltip',
      'href': '#',
      'events': {
        'click': function(pE) {
          this.fireEvent('click', [this.mEl, pE, this]);
          return false;
        }.bind(this)
      }
    }).inject(pContainerEl);

    this.mTextEl = new MooTools.Element('span', {
      'class': 'btn-flat ' + this.options.button_style,
      'data-show-node': 'tooltip'
    }).inject(this.mEl);
    this.setText();

    this.mTooltipEl = this.mEl.getElement('span.tooltip');

    if(this.options.tooltip) {
      this.renderTooltip();
      this.mTooltipEl.set('html', this.options.tooltip);
    }

    if(this.options.tooltip_position) {
      this.mTooltipEl.classList.add(this.options.tooltip_position);
    }

    return this;
  },

  renderTooltip: function() {
    this.mTooltipEl = new MooTools.Element('span', {
      'class': 'tooltip ' + this.options.tooltip_class
    });

    this.mEl.microjungle([
      this.mTooltipEl
    ]);
  },

  refresh: function() {

    try {
      if(!dpManager.dpWorkBench.getActivePage()) {
        return;
      }

      this.fireEvent('refresh', [this.mEl]);
    }
    catch(pError) {
      return false;
    }
  },

  setText: function(pText) {
    this.mTextEl.set('html', pText || this.options.text);
  }

});

;(function UXPAnimationsMapScope(nsp) {
  'use strict';

  nsp.UXPAnimationsMap = {
    easing: {
      LINEAR: {
        value: 'easeNone',
        label: 'Linear'
      },
      EASE_IN: {
        value: 'easeIn',
        label: 'Ease In'
      },
      EASE_OUT: {
        value: 'easeOut',
        label: 'Ease Out'
      },
      EASE_IN_OUT: {
        value: 'easeInOut',
        label: 'Ease In Out'
      }
    },

    easingType: {
      POWER0: {
        value: 'Power0',
        label: 'None',
        hidden: true
      },
      QUAD: {
        value: 'Power1',
        label: 'Quadratic',
      },
      CUBIC: {
        value: 'Power2',
        label: 'Cubic',
      },
      QUART: {
        value: 'Power3',
        label: 'Quartic',
      },
      QUINT: {
        value: 'Power4',
        label: 'Quintic',
      },
      SINE: {
        value: 'Sine',
        label: 'Sine',
      },
      EXPO: {
        value: 'Expo',
        label: 'Exponential',
      },
      CIRC: {
        value: 'Circ',
        label: 'Circular',
      },
      ELASTIC: {
        value: 'Elastic',
        label: 'Elastic',
      },
      BOUNCE: {
        value: 'Bounce',
        label: 'Bounce',
      },
      BACK: {
        value: 'Back',
        label: 'Back',
      }
    }
  };

  UXPAnimationsMap.legacyEasing = {
    'linear': {
      easing: UXPAnimationsMap.easing.LINEAR.value,
      easingType: UXPAnimationsMap.easingType.POWER0.value,
    },
    'ease-in': {
      easing: UXPAnimationsMap.easing.EASE_IN.value,
      easingType: UXPAnimationsMap.easingType.CUBIC.value
    },
    'ease-out': {
      easing: UXPAnimationsMap.easing.EASE_OUT.value,
      easingType: UXPAnimationsMap.easingType.CUBIC.value
    },
    'ease-in-out': {
      easing: UXPAnimationsMap.easing.EASE_IN_OUT.value,
      easingType: UXPAnimationsMap.easingType.CUBIC.value
    },
    'back-out': {
      easing: UXPAnimationsMap.easing.EASE_OUT.value,
      easingType: UXPAnimationsMap.easingType.BACK.value
    },
    'elastic-out': {
      easing: UXPAnimationsMap.easing.EASE_OUT.value,
      easingType: UXPAnimationsMap.easingType.ELASTIC.value
    }
  };

  nsp.UXPAnimationsMap = UXPAnimationsMap;

})(window);

var DPMainToolbar_Widget = new MooTools.Class({
  
  Implements: [MooTools.Options],
  
  mEl: null,
  
  initialize: function(pOptions) {
    this.setOptions(pOptions);
  },
  
  isActive: function() {
    
  },
  
  refresh: function(pProps) {
    
  },
  
  render: function(pContainerEl) {
    
  },
  
  /**
   * Odswieza wlasciwosci inputa do edycji z delikatnym opoznieniem
   */
  refreshMinMax: function(pActiveEl, pMin, pMax, pProps) {
    (function() {
      this.mInput.mActiveEl = pActiveEl;
      
      if('undefined' !== typeof pMin && 'undefined' !== typeof pMax) {
        this.mInput.setMinMax(pMin, pMax);
      }
      
      if(this.mInput.isEnabled()) {
        this.mInput.refresh(pProps);
      }
    }.delay(200, this));
  }
  
})

/* globals DPMainToolbar_Widget */

var DPMainToolbar_Widget_ButtonGroup = new MooTools.Class({
  
  Extends: DPMainToolbar_Widget,
  
  options: {
    'class_name': ''
  },
  
  mItems: [],
  
  initialize: function(pClassName, pWidgets) {
    this.setOptions({
      'class_name': pClassName
    })
    
    this.mItems = pWidgets || [];
  },
  
  render: function(pContainerEl) {
    var class_name = 'group';
    
    if(this.options.class_name) {
      class_name += ' ' + this.options.class_name;
    }
    
    this.mEl = new MooTools.Element('span', {
        'class': class_name
      })
      .inject(pContainerEl);
      
    this.mItems.each(function(pWidget) {
      pWidget.render(this.mEl);
    }.bind(this))
  },
  
  refresh: function(pProps) {
    this.mItems.each(function(pWidget) {
      pWidget.refresh(pProps);
    })
  }
  
})

/* globals DPMainToolbar_Widget_Button */

var DPMainToolbar_Widget_ButtonToolbar = new MooTools.Class({

  Extends: DPMainToolbar_Widget_Button,

  toolbar: null,

  initialize: function(pOptions) {
    this.setOptions(pOptions);
    this.addEvent('click', this.onClick);
    if (typeof this.options.showToolbar == 'function') {
      this.showToolbar = this.options.showToolbar;
    }
  },

  isToolbarOpen: function() {
    return this.toolbar ? true : false;
  },

  hideToolbar: function() {
    this.toolbar.destroy();
    this.toolbar = null;
  },

  onClick: function(el) {
    if (this.isToolbarOpen()) {
      this.hideToolbar();
    } else if (typeof this.showToolbar == 'function') {
      this.fireEvent(DPMainToolbar_Widget_ButtonToolbar.OPEN_TOOLBAR);
      this.showToolbar(el);
    }
  }

});
DPMainToolbar_Widget_ButtonToolbar.OPEN_TOOLBAR = 'openToolbar';

/* globals dpManager */

var DPMainToolbar_Widget_CheckboxGroup = new MooTools.Class({
  
  Implements: [MooTools.Events, MooTools.Options],
  
  options: {
    'tooltip': '',
    'class_name': '',
    'load': MooTools.$empty,
    'options': [{
      'id': '',
      'label': '',
      'value': '',
      'checked': '',
      'click': MooTools.$empty
    }]
  },
  
  mEl: null,
  
  mTooltipEl: null,
  
  initialize: function(pOptions) {
    this.setOptions(pOptions);
  },
  
  render: function(pContainerEl) {
    this.mEl = new MooTools.Element('span', {
      'class': 'options'
    }).inject(pContainerEl);
    
    var text = new MooTools.Element('span', {
      'text': 'Options',
      'class': 'btn-flat only-icon icon-configure',
      'data-show-node': 'tooltip'
    }).inject(this.mEl);
    
    
    var innersettings = new MooTools.Element('span', {
      'class': 'check ' + this.options.class_name
    }).inject(this.mEl);
    
    this.options.options.each(function(pCheckbox) {
      var span = new MooTools.Element('span').inject(innersettings);
      
      var checkbox = new MooTools.Element('input', {
        'id': pCheckbox.id,
        'type': 'checkbox',
        'value': pCheckbox.value,
        'checked': pCheckbox.checked,
        'events': {
          'click': function(pE) {
            if(typeof pCheckbox.click == 'function') {
              pCheckbox.click(pE, this);
            }
            return true;
          }
        }
      }).inject(span);

      var label = new MooTools.Element('label', {
        'class': 'checkbox-label',
        'for': pCheckbox.id,
        'html': pCheckbox.label
      }).inject(span);
    }.bind(this));
    
    this.mTooltipEl = this.mEl.getElement('span.tooltip');
    
    if(this.options.tooltip) {
      this.renderTooltip();
      this.mTooltipEl.set('html', this.options.tooltip);
    }
    
    this.addEvent('load', this.options.load.bind(this));
    this.fireEvent('load');
    
    return this;
  },
  
  renderTooltip: function() {
    this.mTooltipEl = new MooTools.Element('span', {
      'class': 'tooltip'
    });
    
    this.mEl.microjungle([
      this.mTooltipEl
    ])
  },
  
  refresh: function() {
    
    try {
      if(!dpManager.dpWorkBench.getActivePage()) {
        return;
      }
      
      this.fireEvent('refresh', [this.mEl]);
    }
    catch(pError) {
      return false;
    }
  }
  
})

/* globals DPSidebar_Widget_Annotate */
/* globals DPSidebar_Widget_Avatar */
/* globals DPSidebar_Widget_Components */
/* globals DPSidebar_Widget_Elements */
/* globals DPSidebar_Widget_FloatingToolbar */
/* globals DPSidebar_Widget_Help */
/* globals DPSidebar_Widget_HelpOverlay */
/* globals DPSidebar_Widget_Home */
/* globals DPSidebar_Widget_Team */
/* globals DPSidebar_Widget_TheSearchLaunch */
/* globals DPSidebar_Widget_Video */
/* globals DPSidebar_WorkBenchResizer */
/* globals DPWorkBench */
/* globals hashuri */

var DPSidebar = new MooTools.Class({

  Implements: [MooTools.Events],

  mEl: null,

  mCookieName: 'sidebar_config',

  Resizer: null,

  Cookie: null,

  mItems: null,

  initialize: function(pOptions) {
    var active_tab;
    this.container = DPWorkBench.getMainContainer();
    this.mItems = new MooTools.Hash({
      'home': new DPSidebar_Widget_Home('nav'),
      'elements': new DPSidebar_Widget_Elements('nav'),
      'components': new DPSidebar_Widget_Components('nav'),
      'search': new DPSidebar_Widget_TheSearchLaunch('nav'),
      'annotate': new DPSidebar_Widget_Annotate('nav'),
      'team': new DPSidebar_Widget_Team('nav'),
      'news': new DPSidebar_Widget_HelpOverlay('subnav'),
      'video' : new DPSidebar_Widget_Video('subnav'),
      'help': new DPSidebar_Widget_Help('subnav')
    });

    this.render(pOptions || {});
    this.Cookie = new MooTools.Cookie(this.mCookieName, {
      'duration': 365,
      'path': '/'
    });

    this.setResizer();

    active_tab = this.getConfig('active_tab');

    //jesli ktos nie ma cookie z configiem (czyli prawdopodnie jest 1 raz) to otwieramy elements
    if(!active_tab && this.getConfig() && this.getConfig().active_tab === undefined) {
      active_tab = "elements";
    }

    if(hashuri.get('new')) {
      active_tab = "elements";
    }

    if(active_tab && this.mItems.get(active_tab)) {
      this.mItems.get(active_tab).show({
        'animate': false
      })
    }
  },

  setResizer: function() {
    this.Resizer = new DPSidebar_WorkBenchResizer();
  },

  getWidget: function(pName) {
    return this.mItems.get(pName);
  },

  render: function(pOptions) {
    this.mEl = new MooTools.Element('div', {
      'id': 'sidebar'
    }).inject(this.container);

    var nav_el = new MooTools.Element('ul', {
      'class': 'nav'
    }).inject(this.mEl);

    var subnav_el = new MooTools.Element('ul', {
      'class': 'subnav'
    }).inject(this.mEl);

    this.mItems.each(function(pItem, pKey) {
      var container_el = 'nav' === pItem.getSection() ? nav_el : subnav_el,
          list_el = new MooTools.Element('li')
            .inject(container_el);

      if(pOptions && undefined !== pOptions[pKey]) {
        pItem.setOptions(pOptions[pKey])
      }

      pItem.setSidebar(this);
      pItem.render(list_el);
      pItem.addEvents({
        'show': function(pE, pOptions) {
          this.hideOtherToolbars(pItem);
          this.setConfig('active_tab', pKey);

          if (typeof pOptions.fire === 'undefined' || pOptions.fire) {
            this.fireEvent('toolbarShow', [pE, 'show', pOptions]);
          }
          document.body.classList.add('left-panel-visible');
        }.bind(this),
        'hide': function(pE, pOptions) {
          this.setConfig('active_tab', null);
          this.fireEvent('toolbarHide', [pE, 'hide', pOptions]);
          document.body.classList.remove('left-panel-visible');
        }.bind(this)
      })

      pItem.afterInit && pItem.afterInit();
    }, this);
  },

  /**
   * Ukrywa wszystkie aktywne toolbary poza tym aktywnym
   *
   * Podczas ukrywania toolbara *nie ulegnie zmianie* rozmiar workbencha
   */
  hideOtherToolbars: function(pActiveToolbar) {
    this.mItems.getValues()
      .each(function(pItem) {

        if(!(pItem instanceof DPSidebar_Widget_FloatingToolbar) || pItem === pActiveToolbar) {
          return;
        }

        if(!pItem.isToolbarVisible()) {
          return;
        }

        pItem.hide({
          'resize_workbench': false
        })
      })
  },

  setConfig: function(pName, pValue) {
    var config = this.getConfig();
    config[pName] = pValue;

    this.Cookie.write(MooTools.JSON.encode(config));
  },

  getConfig: function(pName) {
    var cookie_value = this.Cookie.read(),
        config = (cookie_value && MooTools.JSON.parse(cookie_value)) || {};

    if(pName) {
      return config[pName] || null;
    }

    return config;
  },

  getItems: function() {
    return this.mItems;
  },

  isAnyToolbarVisible: function() {
    return this.mItems.some(function(pBar) {return pBar.mOn == true});
  }


});

var DPSidebar_Widget = new MooTools.Class({
  
  Implements: [MooTools.Events, MooTools.Options],
  
  mSection: null,
  
  mSidebar: null,
  
  mEl: null,
  
  initialize: function(pSection, pOptions) {
    this.mSection = pSection;
    this.setOptions(pOptions);
  },
  
  getSection: function() {
    return this.mSection;
  },
  
  setSidebar: function(pSidebar) {
    this.mSidebar = pSidebar;
  },
  
  render: function(pContainerEl) {
    
  }
  
});

/* globals css */
/* globals dpManager */

var DPSidebar_WorkBenchResizer = new MooTools.Class({
  
  Implements: [MooTools.Events],
  
  mMorphLockCounter: 0,

  mMorphs: {},
  
  mElements: null,
  
  initialize: function() {
    this.mElements = new MooTools.Hash({
      'workbench': dpManager.dpWorkBench.getWorkBenchElMain(),
      'container': dpManager.dpWorkBench.getWorkBenchElMain().getParent()
    });
  },
  
  /**
   * Zmienia wymiary workbencha o podana delte
   * 
   * Jesli delta jest ujemna workbench zostanie zmniejszony o tyle pikseli,
   * w przeciwnym razie zostanie powiekszony
   * 
   * @param object pOptions {
   *   'animate' - czy animowac zmiane rozmiaru
   *   'width' - delta,
   *   'only_width'
   * }
   */
  resize: function(pOptions) {

    if(this.isLocked() && !pOptions.force) {
      return;
    }
    
    if(pOptions.animate) {
      this._animate(pOptions);
    }
    else {
      this._resize(pOptions);
    }
  },
  
  isLocked: function() {
    return this.mMorphLockCounter > 0;
  },
  
  _resize: function(pOptions) {
    var css = this.calculateStyles(pOptions);

    this.mElements.each(function(pEl, pKey) {

      // cancel ongoing animation
      if (this.mMorphs[pKey] && this.mMorphs[pKey].cancel) {
        this.mMorphs[pKey].cancel();
      }

      pEl.setStyles(css[pKey]);
    }.bind(this));
    
    this.fireEvent('complete');
  },
  
  _animate: function(pOptions) {
    this.mMorphs = this.prepareFx(),
        css = this.calculateStyles(pOptions);
    
    this.mElements.getKeys().each(function(pKey) {
      this.mMorphs[pKey].start(css[pKey])
    }.bind(this));
  },
  
  /**
   * Oblicza style dla elementow, ktore trzeba zmodyfikowac
   * 
   * @param integer pWidth
   * @return object
   */
  calculateStyles: function(pOptions) {
    var workbench_el = this.mElements.workbench,
        container_el = this.mElements.container,
        width = pOptions['width'];

    var container = {
      'width': container_el.getStyle('width').toIntZero() + width
    }

    if (pOptions.only_width) {
      container.left = container_el.getStyle('left').toIntZero();
    } else {
      container.left = container_el.getStyle('left').toIntZero() - width;
    }

    return {
      'workbench': {
        'width': workbench_el.getStyle('width').toIntZero() + width
      },
      'container': container
    };
  },
  
  prepareFx: function() {
    var events = {
          'onStart': function() {
            this.mMorphLockCounter += 1;
          }.bind(this),
          'onComplete': function() {
            this.mMorphLockCounter -= 1;
            
            if(0 == this.mMorphLockCounter) {
              this.fireEventOnce('complete');
            }
          }.bind(this)
        },
        fxs = {};
        
    this.mElements.each(function(pEl, pKey) {
      fxs[pKey] = new MooTools.Fx.Morph(pEl, events);
    });
        
    return fxs;
  }
  
});

/* globals DPSidebar_Widget */
/* globals DPWorkBench */
/* globals dpManager */

var DPSidebar_Widget_FloatingToolbar = new MooTools.Class({

  Implements: DPSidebar_Widget,

  mToolbarEl: null,

  mToolbarWidth: 240,

  mToolbarName: null,

  mToggleShortcut: null,

  mOn: null,

  mCSidebarH: false,

  getContainer: function() {
    return DPWorkBench.getMainContainer();
  },

  renderToolbar: function(container) {
    this.mToolbarEl = new MooTools.Element('div', {
        'class': 'sidebar-toolbar',
        'styles': {
          'width': this.mToolbarWidth
        }
      })
      .hide()
      .inject(container || this.getContainer());

    if(document.getElement("#trial-info")){
      this.mToolbarEl.setStyle('top', 75);
    }

    if(undefined !== this.mToolbarName) {
      this.mToolbarEl.addClass(this.mToolbarName);
    }
  },

  show: function(pOptions) {
    this.mOn = true;

    var active_other_toolbars = this.areOtherToolbarsActive(),
        options = {
          'animate': true,
          'resize_workbench': active_other_toolbars ? false : true
        };

    MooTools.$extend(options, pOptions || {});

    this.mEl.addClass('active');
    this.mToolbarEl.show();

    this.fireEvent('show', [this, options]);

    if(options.resize_workbench) {
      if(options.animate)  {
        this.fireEvent("showAnimStart");
      }

      this.mSidebar.Resizer.resize({
        'width': -this.mToolbarWidth,
        'animate': options.animate,
        'force': options.force || false
      })
    }

    if(dpManager && dpManager.dpPageInspector) {
      dpManager.dpPageInspector.sidebarShown();
      this.adjustToSitemap()
    }
  },

  afterInit: function() {

    if(this.mToggleShortcut && dpManager) {
      dpManager.dpShortcuts.addShortcut(this.mToggleShortcut, this.toggle.bind(this));
    }
  },

  /**
   * Sprawdza czy sa aktualnie aktywne inne toolbary
   *
   * @return boolean
   */
  areOtherToolbarsActive: function() {
    return this.mSidebar.getItems()
      .getValues()
      .some(function(pItem) {

        if(pItem === this) {
          return false;
        }

        if(pItem instanceof DPSidebar_Widget_FloatingToolbar && pItem.isToolbarVisible()) {
          return true;
        }

        return false;
      }.bind(this));
  },

  hide: function(pOptions) {
    this.mOn = false;
    var options = {
      'animate': true,
      'resize_workbench': true
    };

    MooTools.$extend(options, pOptions || {});

    this.mEl.removeClass('active');
    this.fireEvent('hide', [this, options])

    if(options.resize_workbench) {

      if(options.animate) {
        this.fireEvent("hideAnimStart");
      }

      this.mSidebar.Resizer
        .addEvent('complete', function() {
          this.mToolbarEl.hide();
        }.bind(this))
        .resize({
          'width': this.mToolbarWidth,
          'animate': options.animate
        });

      if (dpManager) {
        dpManager.dpWorkBench.resizeWorkBench({additional: 0});
      }
    }
    else {
      this.mToolbarEl.hide();
    }

    if(dpManager) {
      dpManager.dpPageInspector.sidebarWasHidden();
    }
  },

  toggle: function(pOptions) {

    if(this.mSidebar.Resizer.isLocked()) {
      return;
    }

    if(this.isToolbarVisible()) {
      this.hide(pOptions);
    }
    else {
      this.show(pOptions);
    }
  },

  getToolbarEl: function() {
    return this.mToolbarEl;
  },

  isToolbarVisible: function() {
    return this.mToolbarEl.isVisible();
  }

});

(function DPElement_Helper_Scope() {
  'use strict';

  var DPElement_Helper = new MooTools.Class({});

  DPElement_Helper.generateArrowPath = function(pWidth, pHeight, pCurve, pOrient, pStroke) {
    var curve = pCurve || 0;
    var orient = pOrient || "horizontal";
    var width = pWidth || 0;
    var height = pHeight || 0;
    var stroke = pStroke || 1;
    var start_x,
      start_y,
      curve_x,
      curve_y,
      end_x,
      end_y;
    /*
     * budowanie glownej sciezki strzalki
     * tylko pozycje startowe sa bezwzgledne, kolejne (curve, end) to przesuniencie wzgledem pozycji startowej
     */
    var strokemod = stroke*3.2;
    if(orient == "horizontal") {

      start_x = 20 + strokemod;
      curve_x = parseInt(width)/2 - 20 - strokemod;
      if (curve == 1) {
        start_y = parseInt(height) - 20 - strokemod;
        curve_y = 0-parseInt(height)*2 + 20*3 + strokemod*2.5;
        if(curve_y > 0) {
          curve_y = 0;
        }
      }
      else if (curve == 2) {
        start_y = 20 + strokemod;
        curve_y = parseInt(height)*2 - 20*3 - strokemod*2.5;
        if(curve_y < 0) {
          curve_y = 0;
        }
      }
      else {
        start_y = parseInt(height)/2;
        curve_y = 0;
      }
      end_x = parseInt(width) - 20 - 20 - strokemod*2;
      end_y = 0;
    }
    else {
      start_y = 20 + strokemod;
      if (curve == 1) {
        start_x = 20 + strokemod;
        curve_x = parseInt(width)*2 - 3*start_x;
        if(curve_x < 0) {
          curve_x = 0;
        }
      }
      else if (curve == 2) {
        start_x = parseInt(width) - 20 - strokemod;
        curve_x = 0-parseInt(width)*2 + 20*3  + strokemod * 2.5;
        if(curve_x > 0) {
          curve_x = 0;
        }
      }
      else {
        start_x = parseInt(width)/2;
        curve_x = 0;
      }
      curve_y = parseInt(height)/2-start_y;
      end_x = 0;
      end_y = parseInt(height) - 2*start_y;
    }

    return "M"+start_x+","+start_y+ "q"+curve_x+","+curve_y+ " "+end_x+","+end_y;
  };

  window.DPElement_Helper = DPElement_Helper;
})();

/* globals Preview */

var dmsDPPreview_Comet = new MooTools.Class({

  mActiveLayer: 1,

  mMenu: null,

  initialize: function() {
    // Aby ladnie wygladalo "przejscie" pomiedzy nowym kontentem a starym sa 2 divy ktore sie podmieniaja
    $('main1').setStyles('opacity', 1);
  },

   loadPage: function(pHTML) {

    $('main1').set('html', pHTML);
    pHTML.stripScripts(true);
    Preview.fixModals();
  },

  /**
   * Laduje menu (drzewko) z lewej strony
   *
   * Ta metoda jest osobno po to, zeby sama mogla sobie pobierac updateniete drzewko
   * w przypadku zmiany w trybie collaboration
   *
   */
  loadMenu: function(pData) {
    Preview.Menu.load(pData);
  }

});

/* globals MooTreeControl */
/* globals Preview */
/* globals UXPStack */
/* globals hashuri */
/* globals lscache */

var dmsDPPreview_Menu = new MooTools.Class({
  
  mTreeControl: null,
  
  mIconsMap: {
    'Analytics': '21',
    'Business Model Canvas': '22',
    'Project Canvas': '23',
    'Conceptual Diagram': '24',
    'Persona': '25',
    'Sitemap': '26',
    'Specification': '27',
    'Userflow': '28',
    'Visual Design': '29',
    'Prototype': '30',
    'Other': '31'
  },
  
  mEl: null,
  
  mParentEl: null,
  
  mIconsPath: null,
  
  initialize: function(pParentEl) {
    this.mParentEl = pParentEl;
    
    this.mIconsPath = this.getIconsPath();
  },
  
  getIconsPath: function() {
    
    if(!Preview.isExport()) {
      return '/mp/MooTreeControl/';
    }
    
    if(window.location.href.test(/\/pages\//)) {
      return '../data/'
    }
    
    return 'data/';
  },
  
  load: function(pData) {
    UXPStack.add('dmsDPPreview_Menu.load', [pData]);

    // Brak danych do drzewka
    if (!pData) {
      return;
    }

    // Wyrzuca drzewko jesli juz istnialo (zeby je zregenerowac)
    if (this.mTreeControl) {
      this.mTreeControl.remove();
    }
    
    var el = new MooTools.Element('div', {
      'styles': {
        'margin-left': '-10px'
      }
    }).inject(this.mParentEl);
    
    // Tworzy nowe drzewko
    this.mTreeControl = new MooTreeControl({
        'div': el,
        'mode': 'files',
        'grid': true,
        'draggable': false,
        'theme': this.mIconsPath + 'mootree.png?v2',
        'nodeOptions': {
          'text': 'Pages',
          'open': true
        }
      }
    );
      
    // Chowa nazwe projektu
    this.mTreeControl.div.getElement('.mooTree_node:first-child').hide();

    this.mTreeControl.root.options.draggable = false;
    this.mTreeControl.root.divs.node.removeEvents('click');
    this.mTreeControl.root.divs.node.disableSelection();
    
    var load_menu_page = function(pData) {

      if(pData.url) {
        window.location = pData.url + "#" +Preview.mCurrentVersion;
        return;
      }
      
      this.loadMenuPage(pData.id_page || 'd'+pData.id_document);
    }.bind(this);
    
    var project_id = Preview.getData().id_project;
    var just_select = null;
    var hash_map = new MooTools.Hash(Preview.mData.current_map);

    // Dodaje strony
    pData.each(function(pItem) {

      if(!pItem.version_of) {
        var icon = this.mIconsPath + 'mootree.png?v2#',
            parent_node,
            node;

        if (pItem.parent) {
          parent_node = this.mTreeControl.get(pItem.parent);
        }
        else {
          parent_node = this.mTreeControl.root;
        }

        if(pItem.is_category) {

          if(MooTools.$defined(this.mIconsMap[pItem.name])) {
            icon += this.mIconsMap[pItem.name];
          }
          else {
            icon += this.mIconsMap['Other'] || '11';
          }
        }
        else {
          icon += '11';
        }

        var node_id_page;

        if(Preview.mCurrentVersion != 0 && Preview.mData.current_map[pItem.id_page]) {
          node_id_page = Preview.mData.current_map[pItem.id_page];
        }
        else {
          node_id_page = pItem.id_page;
        }

        if(pItem.just_select_it === true) {
          just_select = pItem.id_page || pItem.id;
        }

        if(parent_node) {
          node = parent_node.insert({
            'text': pItem.name,
            'id': pItem.id || pItem.id_page,
            'selected': true,
            'open': false,
            'icon': icon,
            'data': {
              'id_page': node_id_page || null,
              'id_document': pItem.id_document,
              'url': pItem.url
            },

            'onSelect': function() {
              var next_node = this;

              if(this.data && (this.data.id_page || this.data.id_document)) {
                load_menu_page(this.data);
                return;
              }

              while(next_node.getFirst()) {
                next_node = next_node.getFirst();

                if(next_node.data.id_page) {
                  load_menu_page(next_node.data);
                  return;
                }
              }
            },
            /**
             * onExpand jest wywolywany przy zwijaniu i rozwijaniu (parametr okresla stan)
             */
            'onExpand': function(expanding) {
              var open_nodes = lscache.bucket(project_id, function() {
                return lscache.get('open_nodes') || []
              });

              //use one id for responsive to identify open nodes
              var page = hash_map.keyOf(this.data.id_page) || this.data.id_page;
              // Jesli rozwijamy i elementu nie ma w tabeli to dodajemy, jesli nie zwijamy i element jest na liscie to usuwamy
              if (expanding === true && open_nodes.contains(page) === false) {
                open_nodes.push(page);
              }
              else if (expanding === false && open_nodes.contains(page)) {
                var index = open_nodes.indexOf(page);
                open_nodes.splice(index,1);
              }

              lscache.bucket(project_id, function() {
                lscache.set('open_nodes', MooTools.JSON.encode(open_nodes));
              });
            }
          }, false);

          var open_nodes = lscache.bucket(project_id, function() {
            return lscache.get('open_nodes') || []
          });

          // Otwiera drzewko jesli bylo wczesniej otwarte lub jesli znajduje sie na liscie otwartych elementow
          //use one id for responsive to identify open nodes
          var the_id = Preview.getData().version_of || Preview.getData().id_page;

          if ((pItem.id_page && the_id == pItem.id_page) || (pItem.id_document && Preview.getData().id_document == pItem.id_document) || open_nodes.contains(pItem.id_page)) {
            node.toggleParent();
          }
        }
      }
    }.bind(this));

    // Przebudowuje drzewko, zeby wygladalo po ludzku (drugi parametr do node.insert wylacza updateowanie podczas dodawania, zeby bylo szybciej)
    this.mTreeControl.root.update(true);

    // Zaznacza strone na ktorej jest uzytkownik
    var selected_page;

    if(just_select) {
      selected_page = just_select;
    }
    else {
      if(Preview.mCurrentVersion != 0) {
        selected_page = Preview.getData().version_of
      }
      else {
        selected_page = Preview.getCurrentPageId();
      }
    }

    this.mTreeControl.select(this.mTreeControl.get(selected_page), true);
  },
  
  remove: function() {
    this.mTreeControl.remove();
  },
  
  /**
   * Laduje nastepna strone po kliknieciu w element menu
   */
  loadMenuPage: function(pIdPage) {
    var url = Preview.getData().page_base_url.replace(':id_page', pIdPage);

    if (false === Preview.getData().active_collaboration) {
      url += 'd';
    }

    if (false === Preview.getData().menu_enabled) {
      url += 'm';
    }

    if(hashuri.get('R')) {
      url += "#R=1";
    }
    else if(Preview.mCurrentVersion) {
      url += "#"+Preview.mCurrentVersion
    }

    window.location = url;
  }
  
});

/* globals DPSidebar */
/* globals dmsDPPreview_Sidebar_Resizer */

var dmsDPPreview_Sidebar = new MooTools.Class({

  Extends: DPSidebar,

  mCookieName: 'new_preview_sidebar_config',

  mItems: new MooTools.Hash(),

  initialize: function() {
    this.container = document.body;
    this.Cookie = new MooTools.Cookie(this.mCookieName, {
      'path': '/',
      'duration': 365
    });

    this.setResizer();
  },

  setActiveTab: function() {
    var active_tab = this.getConfig('active_tab');

    if(this.getConfig()['active_tab'] === undefined && this.mItems.get('sitemap')) {
      this.mItems.get('sitemap').show({
        'animate': false
      })
    }

    if(active_tab && this.mItems.get(active_tab)) {
      this.mItems.get(active_tab).show({
        'animate': false
      })
    }

  },

  setResizer: function() {
    this.Resizer = new dmsDPPreview_Sidebar_Resizer();
  },

  addWidget: function(pName, pWidget) {
    this.mItems.set(pName, pWidget);
  },

  getWidget:function(pName) {
    return this.mItems[pName];
  }

});

/* globals DPSidebar_Widget */
/* globals Preview */

var dmsDPPreview_Sidebar_Back = new MooTools.Class({
  
  Extends: DPSidebar_Widget,

  mTooltipEl: 'Back to Dashboard',

  render: function(pContainerEl) {
    this.mEl = new MooTools.Element('a', {
        'href': '/dms/Collections/Show/%d'.sprintf(Preview.getData().id_collection),
        'data-tooltip': 'Dashboard',
        'data-show-node': 'tooltip',
        'text': 'Dashboard',
        'class': 'icon-font-back'
      })
      .inject(pContainerEl);

    this.mEl.microjungle([
      ['span', {'class': 'tooltip left-arrow'}, this.mTooltipEl]
    ])
  }

})

/* globals DPSidebar_Widget_FloatingToolbar */
/* globals InteractionsAnimations */

;(function dmsDPPreview_Sidebar_InteractionsScope() {
  'use strict';

  var dmsDPPreview_Sidebar_Interactions = new MooTools.Class({

    Extends: DPSidebar_Widget_FloatingToolbar,

    mToolbarName: 'interactions',

    hideTimeout: null,

    mCopyStyles: ['width','height','left','top','position',window.utils.prefix.css+'transition',window.utils.prefix.css+'transform'],

    mHighlighted:false,

    mObserver: null,

    mTooltipEl: 'Highlight Interactions',

    mAnimationTime:150,

    initialize: function(pSection, pOptions) {
      this.mObserver = new MutationObserver(function observing(mutations) {
        mutations.forEach(function(mutation) {
          var comp_id = 'companion_'+mutation.target.id;
          var comp_el = document.getElementById(comp_id);
          var prefix = [InteractionsAnimations.prefixes[window.utils.prefix.lowercase]];
          comp_el.style[prefix+'TransitionProperty'] = mutation.target.style[prefix+'TransitionProperty'];
          comp_el.style[prefix+'TransitionDuration'] = mutation.target.style[prefix+'TransitionDuration'];
          comp_el.style[prefix+'TransitionTimingFunction'] = mutation.target.style[prefix+'TransitionTimingFunction'];
          comp_el.style[prefix+'Transform'] = mutation.target.style[prefix+'Transform'];

          comp_el.style.top = mutation.target.style.top;
          comp_el.style.left = mutation.target.style.left;
        });
      });
      this.parent(pSection, pOptions);
    },

    render: function(pContainerEl) {
      this.mEl = new MooTools.Element('a', {
        'href': '#',
        'data-tooltip': 'Highlight Interactions',
        'data-show-node': 'tooltip',
        'text': 'Interactions',
        'class': 'icon-font-interactions',
        'events': {
          'click': function(pE) {
            pE.preventDefault();

            if(this.mHighlighted){
              clearTimeout(this.hideTimeout);
              this.hideTimeout = setTimeout(this.disableInteractionHighlight.bind(this),1000);
              return false;
            }

            var inters = document.querySelectorAll('.interactive');
            var config = {attributes:true, childList:false, characterData:false, attributeFilter:['style']};
            var eachStyleCallback = function(pStyle) {
              style += pStyle+':'+inters[i].style[pStyle]+';';
            };

            for(var i=0; i<inters.length; i++) {
              var companion = document.createElement('div');
              companion.id = 'companion_'+inters[i].id;
              companion.classList.add('interactive', 'highlighted','companion');
              var style = '';

              this.mCopyStyles.each(eachStyleCallback);

              companion.style.cssText = style;
              var border = parseInt(inters[i].style.borderWidth);

              if(border) {
                companion.style.width = parseInt(inters[i].style.width) + 2*border + 'px';
                companion.style.height = parseInt(inters[i].style.height) + 2*border + 'px';
              }

              setTimeout(companion.classList.add.bind(companion.classList, ['visible-highlighted']),30);
              this.mObserver.observe(inters[i],config);
              inters[i].parentElement.insertBefore(companion,inters[i]);
            }

            this.mHighlighted = true;

            this.hideTimeout = setTimeout(this.disableInteractionHighlight.bind(this),1500);
            return true;
          }.bind(this)
        }
      }).inject(pContainerEl);

      this.mEl.microjungle([
        ['span', {'class': 'tooltip left-arrow'}, this.mTooltipEl]
      ]);

      this.renderToolbar(document.body);
    },

    disableInteractionHighlight: function() {
      this.mHighlighted = false;
      var companions = document.querySelectorAll('.companion');
      for(var i=0;i<companions.length;i++) {
        companions[i].classList.remove('visible-highlighted');
      }

      setTimeout(function(){
        this.mObserver.disconnect();
        this.removeCompanions();
      }.bind(this),this.mAnimationTime);
    },

    removeCompanions: function() {
      var companions = document.querySelectorAll('.companion');
      for(var i=0;i<companions.length;i++) {
        companions[i].parentElement.removeChild(companions[i]);
      }
    }

  });

  window.dmsDPPreview_Sidebar_Interactions = dmsDPPreview_Sidebar_Interactions;

})();

/* globals DPSidebar_WorkBenchResizer */

var dmsDPPreview_Sidebar_Resizer = new MooTools.Class({
  
  Extends: DPSidebar_WorkBenchResizer,
  
  initialize: function() {
    this.mElements = new MooTools.Hash({
      'canvas': document.getElement('#canvas')
    });
  },
  
  calculateStyles: function(pOptions) {
    return {
      'canvas': {
        'left': this.mElements.canvas.getStyle('left').toIntZero() - pOptions['width']
      }
    };
  }
})

/* globals DPSidebar_Widget_FloatingToolbar */

var dmsDPPreview_Sidebar_Sitemap = new MooTools.Class({
  
  Extends: DPSidebar_Widget_FloatingToolbar,
  
  mToolbarName: 'sitemap',

  mTooltipEl: 'Sitemap',
	
  render: function(pContainerEl) {
    this.mEl = new MooTools.Element('a', {
        'href': '#',
        'data-tooltip': 'Sitemap',
        'data-show-node': 'tooltip',
        'text': 'Sitemap',
        'class': 'icon-font-sitemap',
        'events': {
          'click': function() {
            this.toggle({
              'animate': true
            });
            
            return false;
          }.bind(this)
        }
      })
      .inject(pContainerEl);

    this.mEl.microjungle([
      ['span', {'class': 'tooltip left-arrow'}, this.mTooltipEl]
    ]);

    this.renderToolbar(document.body);
  }
  
});

/* globals Interactions */

;(function Preview_ElementScope() {
  'use strict';

  var Preview_Elements = new MooTools.Class({

    mIndex: {},

    set: function(pId, pEl, pOptions, pProperties) {
      var id = parseInt(pId, 10);
      var el = typeof pEl == 'object' ? pEl : Interactions.getSetElRef(id);

      if (!el) {
        console.error('NO ELEMENT ' + id);
        return;
      }

      var type = Preview_Elements.getElementType(el);
      var class_name = 'Preview_Elements_' + type;

      if (typeof window[class_name] == 'undefined') {
        class_name = 'Preview_Elements_Generic';
      }

      this.mIndex[id] = new window[class_name](pEl, pOptions, pProperties);
      this.mIndex[id].setId(id);
    },

    get: function(pId) {
      return this.mIndex[parseInt(pId, 10)];
    },

    getEach: function(pCallback) {
      if (typeof pCallback != 'function') {
        return false;
      }

      for (var id in this.mIndex) {
        if (this .mIndex.hasOwnProperty(id)) {
          pCallback.call(this.mIndex[id], this.mIndex[id], id);
        }
      }
    },

    highlightInteractive: function() {
      this.getEach(function(pElement) {
        if (pElement.isInteractive()) {
          pElement.highlight();
        }
      });
    }

  });

  Preview_Elements.getElementType = function(pEl) {
    var match;
    var type;

    for (var i = 0, l = pEl.classList.length; i < l; i++) {
      match = pEl.classList[i].match(/^El(.*)$/);

      if (match && match[1]) {
        type = match[1];
        break;
      }
    }

    // off course some exceptions ...
    if (!type) {
      if (pEl.classList.contains('line')) {
        type = 'Line';
      }
      if (pEl.classList.contains('arrow')) {
        type = 'Arrow';
      }
      else if (pEl.classList.contains('GeoSVGEl')) {
        type = 'GeoSVG';
      }
      else if (pEl.classList.contains('UniSVGEl')) {
        type = 'GeoSVG';
      }
      else if (pEl.classList.contains('IconSVG')) {
        type = 'IconSVG';
      }
      else if (pEl.classList.contains('Component')) {
        type = 'Component';
      }
      else if (pEl.classList.length === 0 && pEl.querySelector('img.ElImage')) {
        type = 'Image';
      }
    }

    return type;
  };

  window.Preview_Elements = Preview_Elements;
})();

/* globals Interactions */
/* globals InteractionsAnimations */
/* globals Preview */
/* globals Preview_Elements */
/* globals TweenLite */
/* globals _ */

/*jshint strict: false */
;(function Preview_Elements_Generic_Scope() {

    var convertToPx = function(value) {
        return parseInt(value) + 'px';
    };

    var getRadiusString = function(obj) {
        return convertToPx(obj['border-top-left-radius']) + ' ' +
            convertToPx(obj['border-top-right-radius']) + ' ' +
            convertToPx(obj['border-bottom-right-radius']) + ' ' +
            convertToPx(obj['border-bottom-left-radius']);
    };

    var Preview_Elements_Generic = new MooTools.Class({

        Implements: [MooTools.Options],

        options: {
            interactive: false,
            in_group: false,
            in_component: false,
            anchor_x: 0,
            anchor_y: 0,
            in_multistate: false,
            in_clip: false
        },

        mType: null,

        mId: null,

        mEl: null,

        mElementAnimations: null,

        mProperties: {
            left: 0,
            top: 0,
            angle: 0,
            width: 0,
            height: 0
        },

        initialize: function(pEl, pOptions, pProperties) {
            this.mEl = pEl;
            this.mType = Preview_Elements.getElementType(pEl);
            this.setOptions(pOptions);

            if (this.options.interactive) {
                this.setInteractive();
            }

            if (this.options.in_component) {
                this.mEl.classList.add('component-child');
            }

            this.mProperties = _.clone(_.extend(this.mProperties, {
                'left': parseInt(pEl.style.left),
                'top': parseInt(pEl.style.top),
                'width': parseInt(pEl.style.width),
                'height': parseInt(pEl.style.height)
            }, pProperties || {}));
        },

        getElement: function() {
            return this.mEl;
        },

        getProperty: function(pProperty) {
            return this.mProperties[pProperty];
        },

        setProperty: function(pProperty, pValue) {
            this.mProperties[pProperty] = pValue;
            return this;
        },

        setId: function(pId) {
            this.mId = pId;
        },

        setInteractive: function() {
            var el = this.getElement();
            el.classList.add('interactive');
        },

        isInteractive: function() {
            return this.options.interactive;
        },

        isInGroup: function() {
            return this.options.in_group;
        },

        isInMultistate: function() {
            return this.options.in_multistate;
        },

        highlight: function() {
        },

        cssToJsStyle: function(pCssStyle) {
            return pCssStyle.replace(/\-([a-z]){1}/, function(match, $1) {
                return $1.toUpperCase();
            });
        },

        animate: function(pElement, pAnimation, pProps) {
            pAnimation = InteractionsAnimations.setDefaultEasingValues(pAnimation);

            var propsToKill = _.omit(_.omit(pProps, 'onComplete'), function(value) {
                return value === null;
            });

            TweenLite.killTweensOf(pElement, false, _.object(_.keys(propsToKill), _.map(propsToKill, function() {
                return true;
            })));

            TweenLite.to(pElement, pAnimation.duration / 1000, _.defaults(_.omit(pProps, function(value) {
                return value === null;
            }), {
                ease: InteractionsAnimations.getEasing(pAnimation)
            }));
        },

        show: function(pValue, pAnimation) {
            if (pValue === 'hidden') {
                this.hide(pAnimation);
                return;
            }

            var el = this.getElement();
            var isSlide = pAnimation && pAnimation.show === 'slide';

            if (el.style.visibility !== 'visible') {
                el.style.opacity = 0;
                if (isSlide) {
                    el.style.marginLeft = '-100px';
                }
            }
            el.style.visibility = 'visible';

            this.animate(el, pAnimation, {
                marginLeft: isSlide ? 0 : null,
                opacity: Interactions.ori_styles[this.mId].opacity || 1
            });
        },

        hide: function(pAnimation) {
            var el = this.getElement();
            var isSlide = pAnimation && pAnimation.show === 'slide';

            this.animate(el, pAnimation, {
                opacity: 0,
                marginLeft: isSlide ? 100 : null,
                onComplete: function() {
                    if (isSlide) {
                        el.style.marginLeft = '0px';
                    }

                    el.style.visibility = 'hidden';
                }
            });
        },

        isVisible: function() {
            return window.getComputedStyle(this.getElement()).getPropertyValue('opacity') > 0 && window.getComputedStyle(this.getElement()).getPropertyValue('visibility') !== 'hidden';
        },

        /**
         * Sets style to the element
         *
         * @param pStyle
         * @param pValue
         * @param pUpdateOriginal
         * @param pAnimate [optional]
         */
        setStyle: function(pStyle, pValue, pUpdateOriginal, pAnimation) {
            var el = this.getElement();
            var updateOriginal = typeof pUpdateOriginal == 'undefined' || pUpdateOriginal;
            var styleProperty = this.cssToJsStyle(pStyle);
            var currentStyle = this.getProperty(pStyle) || window.getComputedStyle(el)[styleProperty];

            this.setProperty(pStyle, pValue);

            if (pAnimation) {
                var props = {};
                props[styleProperty] = pValue;
                this.animate(el, pAnimation, {
                    css: props
                });
            } else {
                el.style[styleProperty] = pValue;
            }

            if (updateOriginal) {
                Interactions.ori_styles[this.mId][pStyle] = currentStyle;
            }
        },

        /**
         * Function currently supported only in TextElement class. For others, it's only an alias for setStyle
         *
         * @todo handle elements' original styles (currently Interactions.ori_styles) inside elements classes and make use of this method consequently
         *
         * @param pStyle
         * @param pValue
         * @param pUpdateOriginal
         * @param pAnimation
         * @returns {*}
         */
        resetOriginalStyle: function() {
            return this.setStyle.apply(this, arguments);
        },

        bringToFront: function() {
            var max_zindex = null;
            for (var id in Preview.mElements.mIndex) {
                if (Preview.mElements.mIndex.hasOwnProperty(id) &&
                    this.mId != id &&
                    (null === max_zindex || max_zindex < parseInt(Preview.mElements.mIndex[id].mEl.style['z-index'], 10))
                ) {
                    max_zindex = parseInt(Preview.mElements.mIndex[id].mEl.style['z-index'], 10);
                }
            }

            if (null !== max_zindex && max_zindex > parseInt(this.getElement().style['z-index'], 10)) {
                this.setStyle('z-index', 1 + parseInt(max_zindex, 10), true);
            }
        },

        bringToBack: function() {
            var min_zindex = null;
            for (var id in Preview.mElements.mIndex) {
                if (Preview.mElements.mIndex.hasOwnProperty(id) &&
                    this.mId != id &&
                    (null === min_zindex || min_zindex > parseInt(Preview.mElements.mIndex[id].mEl.style['z-index'], 10))
                ) {
                    min_zindex = parseInt(Preview.mElements.mIndex[id].mEl.style['z-index'], 10);
                }
            }

            if (null !== min_zindex && min_zindex > 0 && min_zindex < parseInt(this.getElement().style['z-index'], 10)) {
                this.setStyle('z-index', -1 + parseInt(min_zindex, 10), true);
            }
        },

        resize: function(pWidth, pHeight, pAnimation) {
            this.setProperty('width', pWidth);
            this.setProperty('height', pHeight);

            this.animate(this.getElement(), pAnimation, {
                width: pWidth,
                height: pHeight
            });
        },

        rotate: function(pAngle, pType, pAnimation) {
            var angle_new = this.getProperty('angle');

            if (pType == 'to') {
                angle_new = parseInt(pAngle, 10) || 0;
            } else {
                angle_new += parseInt(pAngle, 10) || 0;
            }

            this.setProperty('angle', angle_new);
            this.animate(this.getElement(), pAnimation, {
                rotation: angle_new
            });
        },

        setPosition: function(pX, pY, pAnimation) {
            this.setProperty('left', pX);
            this.setProperty('top', pY);

            this.animate(this.getElement(), pAnimation, {
                left: pX,
                top: pY
            });

            this.getElement().originalTop = pY;
            this.getElement().originalLeft = pX;
        },

        setBorderRadius: function(pRadius, pAnimation) {
            _.forEach(pRadius, function(value, key) {
                this.setProperty(key, convertToPx(value));
            }.bind(this));

            this.animate(this.getElement(), pAnimation, {
                borderRadius: getRadiusString(pRadius)
            });
        },

        getPosition: function(pRelativeToWorkbench) {
            var el = this.getElement();
            var position = {
                left: parseInt(el.style.left, 10),
                top: parseInt(el.style.top, 10)
            };

            if (pRelativeToWorkbench && this.isInGroup()) {
                var groupEl = el.parentNode;
                position.left += parseInt(groupEl.style.left, 10);
                position.top += parseInt(groupEl.style.top, 10);
            }

            return position;
        },

        setOpacity: function(pValue, pAnimation) {
            this.animate(this.getElement(), pAnimation, {
                opacity: pValue
            });
        }

    });

    window.Preview_Elements_Generic = Preview_Elements_Generic;
})();

/* globals Interactions */

/*jshint strict: false */
;(function Preview_Elements_ShapeInterface_Scope() {

  var Preview_Elements_ShapeInterface = new MooTools.Class({

    resize: function(pWidth, pHeight, pAnimation) {
      var shapeEl = this.mEl.querySelector('.shape');
      var svgEl = this.mEl.querySelector('svg');
      var scale_factor = [100,100];
      if (!this.mEl.classList.contains('GeoSVGEl')) {
        scale_factor = [parseInt(Interactions.ori_styles[this.mId].original_width), parseInt(Interactions.ori_styles[this.mId].original_height)];
      }
      var scale = [parseInt(pWidth)/scale_factor[0], parseInt(pHeight)/scale_factor[1]];
      this.animate(shapeEl, pAnimation, {
        scaleX: scale[0],
        scaleY: scale[1]
      });
      this.animate(svgEl, pAnimation, {
        width: pWidth,
        height: pHeight
      });
      this.parent(pWidth, pHeight, pAnimation);
    }

  });

  window.Preview_Elements_ShapeInterface = Preview_Elements_ShapeInterface;

})();

/* globals DPElement_Helper */
/* globals Interactions */
/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_Arrow_Scope() {

  var Preview_Elements_Arrow = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    mElLine: null,

    mElMarkers: null,

    initialize: function(pEl, pOptions) {
      this.parent(pEl, pOptions);
      this.mElLine = this.mEl.querySelector('svg .arrow-path');
      this.mElMarkers = this.mEl.querySelectorAll('svg defs marker path');
    },

    setStyle: function(pStyle, pValue, pUpdateOriginal, pAnimations) {
      var update_original = typeof pUpdateOriginal == 'undefined' || pUpdateOriginal;

      if ('color' == pStyle) {
        if (update_original) {
          Interactions.ori_styles[this.mId].color = this.mElLine.getAttribute('stroke');
        }

        this.mElLine.setAttribute('stroke', pValue);

        MooTools.Array.prototype.forEach.call(this.mElMarkers, function(pMarker) {
          pMarker.setAttribute('fill', pValue);
        });
      }
      else {
        this.parent(pStyle, pValue, pUpdateOriginal, pAnimations);
      }
    },

    resize: function(pWidth, pHeight, pAnimation) {
      var arrow_el = this.mEl.querySelector('.arrow-path');
      var props = Interactions.ori_styles[this.mId];
      var path = DPElement_Helper.generateArrowPath(parseInt(pWidth), parseInt(pHeight), props['arrow-curve'], props['arrow-orientation'], props.stroke);

      arrow_el.setAttribute('d', path);
      this.parent(pWidth, pHeight, pAnimation);
    }

  });

  window.Preview_Elements_Arrow = Preview_Elements_Arrow;

})();

/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_Component_Scope() {

  var Preview_Elements_Component = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    /**
     * This is workaround for a problem of groups that cannot have their own opacity, because opacity creates z-index stack.
     * To show group with fade animation, group needs to have opacity set to 0.
     *
     * @param pAnimation
     * @todo Remove this when we switch to groups having their own opacity
     */
    show: function(pValue, pAnimation) {
      if (!this.isVisible() && pAnimation && pAnimation.show && pAnimation.time > 0) {
        this.getElement().style.opacity = 0;
      }

      this.parent(pValue, pAnimation);
    }

  });

  window.Preview_Elements_Component = Preview_Elements_Component;

})();

/* globals Interactions */
/* globals Preview_Elements_Generic */
/* globals Preview_Elements_ShapeInterface */

/*jshint strict: false */
;(function Preview_Elements_GeoSVG_Scope() {

  var Preview_Elements_GeoSVG = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    Implements: [Preview_Elements_ShapeInterface],

    mElSvg: null,

    initialize: function(pEl, pOptions) {
      this.parent(pEl, pOptions);
      this.mElSvg = this.mEl.querySelector('svg polygon') || this.mEl.querySelector('svg ellipse');
    },

    setStyle: function(pStyle, pValue, pUpdateOriginal, pAnimation) {
      var update_original = typeof pUpdateOriginal == 'undefined' || pUpdateOriginal;

      if ('color' == pStyle) {
        if (update_original) {
          Interactions.ori_styles[this.mId].color = this.mElSvg.getAttribute('fill');
        }

        this.mElSvg.setAttribute('fill', pValue);
      }
      else {
        this.parent(pStyle, pValue, pUpdateOriginal, pAnimation);
      }
    }

  });

  window.Preview_Elements_GeoSVG = Preview_Elements_GeoSVG;

})();

/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_Group_Scope() {

  var Preview_Elements_Group = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    mGroupElements: [],

    initialize: function(pEl, pOptions) {
      this.parent(pEl, pOptions);

      MooTools.Array.prototype.forEach.call(this.mEl.querySelectorAll('*[id^=el]'), function(pEl) {
        this.mGroupElements.push(pEl);
      }.bind(this));
    },

    /**
     * This is workaround for a problem of groups that cannot have their own opacity, because opacity creates z-index stack.
     * To show group with fade animation, group needs to have opacity set to 0.
     *
     * @param pAnimation
     * @todo Remove this when we switch to groups having their own opacity
     */
    show: function(pValue, pAnimation) {
      if (!this.isVisible() && pAnimation && pAnimation.show && pAnimation.time > 0) {
        this.getElement().style.opacity = 0;
      }

      this.parent(pValue, pAnimation);
    },

    resize: function(pWidth, pHeight, pAnimation) {
      var current = {w: parseInt(this.mEl.style.width), h: parseInt(this.mEl.style.height)};
      var size = {w: parseInt(pWidth), h: parseInt(pHeight)};
      var delta = {w: size.w/current.w, h: size.h/current.h};

      this.mGroupElements.forEach(function eachElementScope(pEl) {
        if (pAnimation) {
            this.animate(pEl, pAnimation, {
                width: parseInt(pEl.style.width) * delta.w,
                height: parseInt(pEl.style.height) * delta.h,
                left: parseInt(pEl.style.left) * delta.w,
                top: parseInt(pEl.style.top) * delta.h
            });
        }
      }.bind(this));

      this.parent(pWidth, pHeight, pAnimation);
    },
  });

  window.Preview_Elements_Group = Preview_Elements_Group;

})();

/* globals Interactions */
/* globals Preview_Elements_Generic */
/* globals Preview_Elements_ShapeInterface */

/*jshint strict: false */
;(function Preview_Elements_IconSVG_Scope() {

  var Preview_Elements_IconSVG = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    Implements: [Preview_Elements_ShapeInterface],

    mElSvg: null,

    mPaths: [],

    initialize: function(pEl, pOptions) {
      this.parent(pEl, pOptions);
      this.mElSvg = this.mEl.querySelector('.shape:first-child');
      this.mPaths = this.mEl.querySelectorAll('.shape *');

      // FontAwesome has also a IconSVG class, but has a different structure :/
      if (!this.mElSvg) {
        this.mElSvg = this.mEl.querySelector('path.shape');
      }
    },

    setStyle: function(pStyle, pValue, pUpdateOriginal, pAnimation) {
      var update_original = typeof pUpdateOriginal == 'undefined' || pUpdateOriginal;

      if ('color' == pStyle) {
        if (update_original) {
          Interactions.ori_styles[this.mId].color = this.mElSvg.getAttribute('fill');
        }

        this.mElSvg.setAttribute('fill', pValue);

        if (this.mPaths.length > 0) {
          for (var i = 0; i < this.mPaths.length; i++) {
            this.mPaths[i].setAttribute('fill', pValue);
          }
        }
      }
      else {
        this.parent(pStyle, pValue, update_original, pAnimation);
      }
    }

  });

  window.Preview_Elements_IconSVG = Preview_Elements_IconSVG;

})();

/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_Image_Scope() {

  var Preview_Elements_Image = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    mElSvg: null,

    initialize: function(pEl, pOptions) {
      this.parent(pEl, pOptions);
      this.mElImage = this.mEl.querySelector('img');
    },

    resize: function(pWidth, pHeight, pAnimation) {
      this.animate(this.mElImage, pAnimation, {
        width: pWidth,
        height: pHeight
      });
    }

  });

  window.Preview_Elements_Image = Preview_Elements_Image;

})();

/* globals Interactions */
/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_Line_Scope() {

  var Preview_Elements_Line = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    mElLine: null,

    initialize: function(pEl, pOptions) {
      this.parent(pEl, pOptions);
      this.mElLine = this.mEl.querySelector('svg line');
    },

    setStyle: function(pStyle, pValue, pUpdateOriginal, pAnimation) {
      var update_original = typeof pUpdateOriginal == 'undefined' || pUpdateOriginal;

      if ('color' == pStyle) {
        if (update_original) {
          Interactions.ori_styles[this.mId].color = this.mElLine.getAttribute('stroke');
        }

        this.mElLine.setAttribute('stroke', pValue);
      }
      else {
        this.parent(pStyle, pValue, pUpdateOriginal, pAnimation);
      }
    }

  });

  window.Preview_Elements_Line = Preview_Elements_Line;

})();

/* globals Preview_Elements_Generic */
/* globals _ */

/*jshint strict: false */
;(function Preview_Elements_Multistate_Scope() {

  var noAnimation = function(animation) {
    return !animation || typeof animation.show === 'undefined';
  };

  var Preview_Elements_Multistate = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    initialize: function(pEl, pOptions, pProperties) {
      this.parent(pEl, pOptions, pProperties);

      this.clipped = false;
      if (pEl.dataset.clipped === '1') {
        this.clipped = true;
      }
      this.mGroupElements = [];
      this.mStates = [];
      this.mCurrentState;

      MooTools.Array.prototype.forEach.call(this.mEl.querySelectorAll('*[id^=el]'), function(pEl) {
        this.mGroupElements.push(pEl);
      }.bind(this));

      MooTools.Array.prototype.forEach.call(this.mEl.querySelectorAll('.state'), function(pStateEl) {
        var state = {
          id: pStateEl.getAttribute('data-state-id'),
          name: pStateEl.getAttribute('data-state-name'),
          default: !!parseInt(pStateEl.getAttribute('data-state-default')),
          ref: pStateEl.querySelector('.state-content')
        };
        this.mStates.push(state);
        if(state.default) {
          this.mCurrentState = this.mStates.length-1;
        }

      }.bind(this));
    },

    /**
     * This is workaround for a problem of groups that cannot have their own opacity, because opacity creates z-index stack.
     * To show group with fade animation, group needs to have opacity set to 0.
     *
     * @param pAnimation
     * @todo Remove this when we switch to groups having their own opacity
     */
    show: function(pAnimation) {
      if (!this.isVisible() && pAnimation && pAnimation.show && pAnimation.time > 0) {
        this.getElement().style.opacity = 0;
      }

      this.parent(pAnimation);
    },

    setProperty: function(pProperty, pValue) {
      this.mProperties[pProperty] = pValue;
      return this;
    },

    getAnimation: function(pAnimation) {
      if(pAnimation.show && typeof pAnimation.show === 'string' && pAnimation.show.indexOf('-') > -1) {
        var details = pAnimation.show.split('-');
        pAnimation.show = details[0];
        pAnimation.direction = details[1];
      }
      return pAnimation;
    },

    hideState: function(pState, pAnimation) {
      /* jshint maxcomplexity: 12*/

      var el = pState;
      if(noAnimation(pAnimation)) {
        el.parentNode.style.display = 'none';
        return;
      }
      pAnimation = this.getAnimation(pAnimation);
      var animation_data = {};

      var data = {};
      if(pAnimation && pAnimation.show) {
        switch(pAnimation.show) {
          case 'slide':
            el.parentNode.style.overflow = 'hidden';
            if(['down','up'].indexOf(pAnimation.direction) > -1) {
              data.location = 'top';
              data.start = '0px';
              data.stop = parseInt(el.parentNode.style.height);
              if(pAnimation.direction === 'up') {
                data.stop = -1 * data.stop;
              }
            }
            if(['right','left'].indexOf(pAnimation.direction) > -1) {
              data.location = 'left';
              data.start = '0px';
              data.stop = parseInt(el.parentNode.style.width);
              if(pAnimation.direction === 'left') {
                data.stop = -1 * data.stop;
              }
            }
            data.stop +='px';
            el.style[data.location] =  data.start;

            break;
          case 'flip':
            el.parentNode.style.overflow = 'visible';
            this.mEl.style.pointerEvents = 'auto';
            var direction = '_cw';
            if(pAnimation.hover_out) {
              direction = '_ccw';
            }
            if(pAnimation.direction === 'y') {
              el.style['-webkit-transform'] = 'rotateY(0deg)';
              el.style.transform = 'rotateY(0deg)';
              data.location = 'directionalRotation';
              data.stop = {rotationY:'180'+direction};
            }
            else {
              el.style['-webkit-transform'] = 'rotateX(0deg)';
              el.style.transform = 'rotateX(0deg)';
              data.location = 'directionalRotation';
              data.stop = {rotationX:'180'+direction};
            }

            break;
          case 'fade':
            data.location = 'opacity';
            data.start = 1;
            data.stop = 0;
            el.style[data.location] =  data.start;
            break;
        }

        animation_data[data.location] = data.stop;
      }

      animation_data.onComplete = function() {
        el.style.left = '0px';
        el.style.top = '0px';
        el.parentNode.style.display = 'none';
        el.style.removeProperty('-webkit-transform');
        el.style.removeProperty('transform');
        el.parentNode.style.overflow = 'visible';
        this.mEl.style.pointerEvents = 'none';
      }.bind(this);

      this.animate(el, pAnimation, animation_data);
    },

    showState: function(pState,pAnimation) {
      /* jshint maxcomplexity: 12 */

      var el = pState;
      if(!pAnimation) {
        el.parentNode.style.display = 'initial';
        return;
      }

      pAnimation = this.getAnimation(pAnimation);
      var data = {};
      var animation_data = {};

      if(pAnimation && pAnimation.show) {
        switch(pAnimation.show) {
          case 'slide':
            el.parentNode.style.overflow = 'hidden';
            if(['down','up'].indexOf(pAnimation.direction) > -1) {
              data.location = 'top';
              data.start = parseInt(el.parentNode.style.height);
              data.stop = 0;
              if(pAnimation.direction === 'down') {
                data.start = -1 * data.start;
              }
            }
            if(['right','left'].indexOf(pAnimation.direction) > -1) {
              data.location = 'left';
              data.start = parseInt(el.parentNode.style.width);
              data.stop = 0;
              if(pAnimation.direction === 'right') {
                data.start = -1 * data.start;
              }
            }
            data.start +='px';
            data.stop +='px';
            el.style[data.location] = data.start;
            break;

          case 'flip':
            el.parentNode.style.overflow = 'visible';
            var direction = '_cw';
            if(pAnimation.hover_out) {
              direction = '_ccw';
            }
            if(pAnimation.direction === 'y') {
              el.style['-webkit-transform'] = 'rotateY(180deg)';
              el.style.transform = 'rotateY(180deg)';
              data.location = 'directionalRotation';
              data.stop = {rotationY:'0'+direction};
            }
            else {
              el.style['-webkit-transform'] = 'rotateX(180deg)';
              el.style.transform = 'rotateX(180deg)';
              data.location = 'directionalRotation';
              data.stop = {rotationX:'0'+direction};
            }
            break;

          case 'fade':
            data.location = 'opacity';
            data.start = 0;
            data.stop = 1;
            el.style[data.location] =  data.start;
            break;
        }

        animation_data[data.location] = data.stop;
      }

      animation_data.onComplete = function() {
        el.parentNode.style.overflow = 'visible';
        el.style.removeProperty('transform');
        el.style.removeProperty('-webkit-transform');
      };

      el.parentNode.style.display = 'block';
      this.animate(el, pAnimation, animation_data);
    },

    nextState: function(pAnimation) {
      var next_state = this.mCurrentState + 1;
      if(next_state > this.mStates.length - 1) {
        next_state = 0;
      }

      if(next_state === this.mCurrentState) {
        return;
      }

      this.showState(this.mStates[next_state].ref, pAnimation);
      this.hideState(this.mStates[this.mCurrentState].ref, pAnimation);

      this.mCurrentState = next_state;
    },

    changeState: function(pNewStateId, pAnimation) {
      var next = _.findWhere(this.mStates, {id : pNewStateId+''});
      if(!next || this.mCurrentState == this.mStates.indexOf(next)) {
        return;
      }
      else {
        this.showState(next.ref, pAnimation);
        this.hideState(this.mStates[this.mCurrentState].ref, pAnimation);
      }
      this.mCurrentState = this.mStates.indexOf(next);

    },

    moveBy: function(pX, pY, pAnimation) {
      var pos = this.getPosition();
      this.setPosition(pos.left + pX ,pos.top + pY, pAnimation);

      if(!this.clipped) {
        var elem = this.getElement();
        elem.setAttribute('data-left', parseInt(elem.dataset.left) + pX);
        elem.setAttribute('data-top', parseInt(elem.dataset.top) + pY);
      }
    },

    moveTo: function(pPosition, pAnimation) {
      var x = pPosition.x;
      var y = pPosition.y;

      if(this.clipped) {
        this.setPosition(x ,y, pAnimation);
      }
      else if(pPosition.goBack) {
        this.setPosition(0, 0, pAnimation);
      }
      else {
        var new_left;
        var new_top;
        var elem = this.getElement();
        var html_position = this.getPosition();
        var visible_left = elem.dataset.left;
        var visible_top = elem.dataset.top;

        if (visible_left === x) {
          new_left = html_position.left;
        }
        else {
          new_left = x - visible_left;
        }

        if(visible_top === y) {
          new_top = html_position.top;
        }
        else {
          new_top = y - visible_top;
        }
        this.setPosition(new_left , new_top, pAnimation);
        elem.setAttribute('data-left', x);
        elem.setAttribute('data-top', y);
      }
    },

    prevState: function(pAnimation) {
      var prev_state = this.mCurrentState - 1;
      if(prev_state < 0) {
        prev_state = this.mStates.length-1;
      }

      if(prev_state === this.mCurrentState) {
        return;
      }
      this.showState(this.mStates[prev_state].ref, pAnimation);
      this.hideState(this.mStates[this.mCurrentState].ref, pAnimation);

      this.mCurrentState = prev_state;
    },

    resize: function(pWidth, pHeight, pAnimation) {
      var current = {w: parseInt(this.getProperty('width')), h: parseInt(this.getProperty('height'))};
      var size = {w: parseInt(pWidth), h: parseInt(pHeight)};
      var delta = {w: size.w/current.w, h: size.h/current.h};

      this.mGroupElements.forEach(function eachElementScope(pEl) {
        if (pAnimation) {
          this.animate(pEl, pAnimation, {
            width: parseInt(pEl.style.width) * delta.w,
            height: parseInt(pEl.style.height) * delta.h,
            left: parseInt(pEl.style.left) * delta.w,
            top: parseInt(pEl.style.top) * delta.h
          });
        }
      }.bind(this));

      MooTools.Array.prototype.forEach.call(this.mEl.querySelectorAll('.state'), function(pStateEl) {
        if (pAnimation) {
          this.animate(pStateEl, pAnimation, {
            width: parseInt(pStateEl.style.width) * delta.w,
            height: parseInt(pStateEl.style.height) * delta.h,
            left: parseInt(pStateEl.style.left) * delta.w,
            top: parseInt(pStateEl.style.top) * delta.h
          });
        }
      }.bind(this));

      this.setProperty('width', size.w);
      this.setProperty('height', size.h);
    }

  });

  window.Preview_Elements_Multistate = Preview_Elements_Multistate;

})();

/* globals Interactions */
/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_Navigation_Scope() {

    var Preview_Elements_Navigation = new MooTools.Class({

        Extends: Preview_Elements_Generic,

        mElements: null,

        mElementsInactive: null,

        initialize: function(pEl, pOptions) {
            this.parent(pEl, pOptions);
            this.mElements = this.mEl.querySelectorAll('li');
            this.mElementsInactive = this.mEl.querySelectorAll('li.inactive');
        },

        setStyle: function(pStyle, pValue, pUpdateOriginal, pAnimate) {
            var el_id = this.mId;
            var update_original = typeof pUpdateOriginal == 'undefined' || pUpdateOriginal;
            var current_style;

            // font color set in standard way
            if ('color' == pStyle) {
                return this.parent(pStyle, pValue, pUpdateOriginal, pAnimate);
            }

            // set styles
            if (['inactive-background-color', 'background-color'].indexOf(pStyle) > -1) {
                MooTools.Array.prototype.forEach.call(this.mElementsInactive, function(pElement) {
                    current_style = window.getComputedStyle(pElement).backgroundColor;
                    this.animate(pElement, pAnimate, {
                        css: {
                            backgroundColor: pValue
                        }
                    });
                }.bind(this));

                if (update_original) {
                    Interactions.ori_styles[el_id]['background-color'] = current_style;
                }
            }
            else if ('element-border-color' == pStyle) {
                MooTools.Array.prototype.forEach.call(this.mElements, function(pElement) {
                    current_style = window.getComputedStyle(pElement).borderColor;
                    this.animate(pElement, pAnimate, {
                        css: {
                            borderColor: pValue
                        }
                    });
                }.bind(this));

                if (update_original) {
                    Interactions.ori_styles[el_id]['border-color'] = current_style;
                }
            }
            else {
                this.parent(pStyle, pValue, pUpdateOriginal, pAnimate);
            }
        }

    });

    window.Preview_Elements_Navigation = Preview_Elements_Navigation;
})();

/* globals Interactions */
/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_Tab_Scope() {

  /**
   * @todo This class should extend Preview_Elements_Navigation, but we need to fix multi-level extending
   * @type {Class}
   */
  var Preview_Elements_Tab = new MooTools.Class({

      Extends: Preview_Elements_Generic,

      mElements: null,

      mElementsInactive: null,

      initialize: function(pEl, pOptions) {
          this.parent(pEl, pOptions);
          this.mElements = this.mEl.querySelectorAll('li');
          this.mElementsInactive = this.mEl.querySelectorAll('li.inactive');
      },

      setStyle: function(pStyle, pValue, pUpdateOriginal, pAnimate) {
          var el_id = this.mId;
          var update_original = typeof pUpdateOriginal == 'undefined' || pUpdateOriginal;
          var current_style;

          // font color set in standard way
          if ('color' == pStyle) {
              return this.parent(pStyle, pValue, pUpdateOriginal, pAnimate);
          }

          // set styles
          if (['inactive-background-color', 'background-color'].indexOf(pStyle) > -1) {
              MooTools.Array.prototype.forEach.call(this.mElementsInactive, function(pElement) {
                  current_style = window.getComputedStyle(pElement).backgroundColor;
                  this.animate(pElement, pAnimate, {
                      css: {
                          backgroundColor: pValue
                      }
                  });
              }.bind(this));

              if (update_original) {
                  Interactions.ori_styles[el_id]['background-color'] = current_style;
              }
          }
          else if ('element-border-color' == pStyle) {
              MooTools.Array.prototype.forEach.call(this.mElements, function(pElement) {
                  current_style = window.getComputedStyle(pElement).borderColor;
                  this.animate(pElement, pAnimate, {
                      css: {
                          borderColor: pValue
                      }
                  });
              }.bind(this));

              if (update_original) {
                  Interactions.ori_styles[el_id]['border-color'] = current_style;
              }
          }
          else {
              this.parent(pStyle, pValue, pUpdateOriginal, pAnimate);
          }
      }

  });

  window.Preview_Elements_Tab = Preview_Elements_Tab;
})();

/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_TextElement_Scope() {

    var Preview_Elements_TextElement = new MooTools.Class({

        Extends: Preview_Elements_Generic,

        mChildElements: null,

        mChildrenProperties: [],

        mChildrenOriginalStyles: [],

        initialize: function(pEl, pOptions) {
            this.parent(pEl, pOptions);
            this.mChildElements = this.mEl.querySelectorAll('span[style]');
            this.mChildrenProperties = [].map.call(this.mChildElements, function() { return {}; });
            this.mChildrenOriginalStyles = [].map.call(this.mChildElements, function() { return {}; });
        },

        setStyle: function(pStyle, pValue, pUpdateOriginal, pAnimation) {
            var styleProperty = this.cssToJsStyle(pStyle);
            this.parent.apply(this, arguments);

            [].forEach.call(this.mChildElements, function(pChild, pIndex) {
                var props = {};

                if (pUpdateOriginal || typeof pUpdateOriginal === 'undefined') {
                    this.mChildrenOriginalStyles[pIndex][pStyle] = this.mChildrenProperties[pIndex][pStyle] || window.getComputedStyle(pChild)[styleProperty];
                }

                this.mChildrenProperties[pIndex][pStyle] = pValue;

                if (pAnimation) {
                    props[styleProperty] = pValue;

                    this.animate(pChild, pAnimation, {
                        css: props
                    });
                } else {
                    pChild.style[styleProperty] = pValue;
                }
            }.bind(this));
        },

        resetOriginalStyle: function(pStyle, pValue, pUpdateOriginal, pAnimation) {
            var styleProperty = this.cssToJsStyle(pStyle);
            this.parent.apply(this, arguments);

            [].forEach.call(this.mChildElements, function(pChild, pIndex) {
                var props = {};
                var value = this.mChildrenOriginalStyles[pIndex][pStyle];

                if (pUpdateOriginal || typeof pUpdateOriginal === 'undefined') {
                    this.mChildrenOriginalStyles[pIndex][pStyle] = this.mChildrenProperties[pIndex][pStyle] || window.getComputedStyle(pChild)[styleProperty];
                }

                this.mChildrenProperties[pIndex][pStyle] = value;

                if (pAnimation) {
                    props[styleProperty] = value;

                    this.animate(pChild, pAnimation, {
                        css: props
                    });
                } else {
                    pChild.style[styleProperty] = value;
                }
            }.bind(this));
        }

    });

    window.Preview_Elements_TextElement = Preview_Elements_TextElement;

})();

/* globals Preview_Elements_Generic */

/*jshint strict: false */
;(function Preview_Elements_Tooltip_Scope() {

  var Preview_Elements_Tooltip = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    initialize: function(pEl, pOptions) {
      pEl = pEl.parentNode;
      this.parent(pEl, pOptions);
    }

  });

  window.Preview_Elements_Tooltip = Preview_Elements_Tooltip;

})();

/* globals Preview_Elements_Generic */
/* globals Preview_Elements_ShapeInterface */

/*jshint strict: false */
;(function Preview_Elements_UniSVG_Scope() {

  var Preview_Elements_UniSVG = new MooTools.Class({

    Extends: Preview_Elements_Generic,

    Implements: [Preview_Elements_ShapeInterface]

  });

  window.Preview_Elements_UniSVG = Preview_Elements_UniSVG;

})();

/* globals Dejax */
/* globals Preview */
/* globals Preview_Elements */
/* globals dmsDPPreview_MainToolbar */
/* globals dmsDPPreview_Menu */
/* globals dmsDPPreview_Sidebar */
/* globals dmsDPPreview_Sidebar_Back */
/* globals dmsDPPreview_Sidebar_Interactions */
/* globals dmsDPPreview_Sidebar_Sitemap */
/* globals hashuri */

var dmsDPPreview_Preview = new MooTools.Class({

  Sidebar: null,

  MainToolbar: null,

  Menu: null,

  mData: null,

  mCanvasEl: null,

  mCurrentVersion: 0,

  mPageBuffer: {},

  mExportMode: false,

  mElements: null,

  mDejax: new Dejax,

  mVersions:  new MooTools.Hash({
   "0": {"version":0, "text":"Default", "width": "5000"},
   "1": {"version":1, "text":"iPhone (320px)", "width": 320, "breakwidth":400},
   "2": {"version":2, "text":"iPhone Landscape (480px)", "width": 480, "breakwidth": 500},
   "3": {"version":3, "text":"Tablets (600px)", "width": 600, "breakwidth": 620},
   "4": {"version":4, "text":"iPads (768px)", "width": 768, "breakwidth": 790},
   "5": {"version":5, "text":"Standard websites (992px)", "width": 992, "breakwidth": 1012},
   "6": {"version":6, "text":"iPad Landscape (1024px)", "width": 1024, "breakwidth": 1044},
   "7": {"version":7, "text":"Wide websites (1224px)", "width": 1224, "breakwidth": 1244}
  }),

  mFixedElems: [],

  initialize: function() {
    this.mElements = new Preview_Elements();
  },

  attachEvents: function() {
    this.mCanvasEl = document.getElement('#canvas');
    this.canvasResize();

    window.addEvent('resize', this.canvasResize.bind(this));
    window.addEvent('resize', this.responsiveAdjust);
    window.addEventListener("orientationchange", function() {
      Preview.canvasResize();
      Preview.responsiveAdjust();
    }, false);
  },

  setData: function(pData) {

    this.mData = pData;

    if(this.mData.versions) {
      this.mData.versions = new MooTools.Hash(MooTools.JSON.decode(this.mData.versions));
    }

    if (this.mData.version_data) {
      this.mData.version_data = new MooTools.Hash(this.mData.version_data);
      this.mData.versions.combine(this.mData.version_data);
      this.mData.version_data.forEach(function(row) {
        "use strict";
        row.breakwidth = row.width + 20;
        Preview.mVersions.set(row.version, row);
      });
    }

    var breaks = [];
    this.mData.versions.each(function(pId, pVer){
      var obj = {};
      if (Preview.mVersions.hasOwnProperty(pVer)) {
        obj[Preview.mVersions[pVer].width] =  pVer;
        obj.width = Preview.mVersions[pVer].breakwidth || Preview.mVersions[pVer].width;
        obj.height = Preview.mVersions[pVer].height || null;
        obj.version = parseInt(pVer, 10);
        obj.breakwidth = parseInt(pVer, 10);
        breaks.push(obj);
      }
    });
    breaks.sort(function(pA,pB){
      if(pA.width < pB.width) return -1;
      if(pA.width > pB.width) return 1;
      return 0;
    });
    this.breaks = breaks

    this.mCurrentPageId = this.mData.id_page;
    this.mCurrentVersion = this.mData.current_version;
    this.canvasResize();
  },


  getData: function() {
    return this.mData;
  },


  getCurrentPageId: function() {
    return this.mCurrentPageId
  },

  getDocumentId: function() {
    return this.mData.id_document;
  },

  createMainToolbar: function() {

    this.MainToolbar = new dmsDPPreview_MainToolbar();

    this.mCanvasEl.getElement('#canvas-wrapper').setStyle('top', '50px');
    this.canvasResize();
  },


  hideVersionList: function(pE) {
    window.removeEvent(this.hideVersionList)
    var elements = this.mVersionListDiv.getElements('*');
    elements.push(this.mVersionListDiv);

    if(pE && elements.contains(pE.target)) {
      return;
    }

    this.mVersionListDiv.hide();
  },


  createVersionSwitch: function(pVersionsHash) {

    var container = new MooTools.Element("div", {
      "class" : "responsive"
    }).microjungle([
        ['label',{'text':'Responsive:'}]
      ]);

    var list = new MooTools.Element("select", {
      "class": "version-list",
      "id": "version-select"
    });

    container.adopt(list);
    pVersionsHash.each(function(pVersion, pType) {
      if (this.mVersions.hasOwnProperty(pType)) {
        var li = new MooTools.Element('option', {
          "class": 'version-item',
          "value": pVersion,
          "text": this.mVersions[pType].text,
          "selected": pType == this.mCurrentVersion ? true : false
        });
        list.adopt(li);
      }
    }.bind(this));

    var li = new MooTools.Element('option', {
      "class": 'version-item',
      "value": "responsive",
      "html": "&larr; Auto (resize) &rarr;"
    });

    list.grab(li,"top");

    list.addEvent('change', function() {
      if(this.getSelected().get('value') != "responsive") {
        hashuri.set("R");
        if(Preview.isExport()) {
          Preview.loadPageCall(this.getSelected().get('value'));
          Preview.canvasResize();
        }
        else {
          Preview.Menu.loadMenuPage(this.getSelected().get('value'));
        }
      }
      else {
        hashuri.set("R",1);
        Preview.responsiveAdjust();
        Preview.canvasResize();
      }
    });

    if(document.getElement('.sidebar-toolbar.sitemap')) {
      document.getElement('.sidebar-toolbar.sitemap').grab(container,'top');
    }

  },

  createSidebar: function(pOptions) {
    this.Sidebar = new dmsDPPreview_Sidebar();

    if(pOptions.show_back_widget) {
      this.Sidebar.addWidget('home', new dmsDPPreview_Sidebar_Back('nav'));
    }


    // miejsce na sitemap
    if(this.mData.pages.length > 0) {
      this.Sidebar.addWidget('sitemap', new dmsDPPreview_Sidebar_Sitemap('nav'));
    }

    this.Sidebar.addWidget('interactions', new dmsDPPreview_Sidebar_Interactions('nav'));

    this.Sidebar.render();
    this.Sidebar.setActiveTab();
    this.mCanvasEl.setStyle('left', this.mCanvasEl.getStyle('left').toIntZero() + this.Sidebar.mEl.getStyle('width').toIntZero());

    // Callback na resizowanie pola podgladu przy otwieraniu i zamykaniu toolbara przy sidebarze
    this.mCanvasResizeRef = function(pE) {
      this.canvasResize.delay(600, this, [pE]);
    }.bind(this);

    this.Sidebar.addEvent('toolbarShow', this.canvasResize.bind(this));
    this.Sidebar.addEvent('toolbarHide', this.canvasResize.bind(this));

    if(this.mData.pages.length > 0) {
      this.Menu = new dmsDPPreview_Menu(this.Sidebar.getWidget('sitemap').getToolbarEl(), this.mData.pages);
      this.Menu.load(this.mData.pages);
    }

    this.canvasResize();
  },

  canvasResize: function(pE, pAction, pShowHideOptions) {
    var width = window.getWidth() - this.mCanvasEl.getStyle('left').toIntZero(),
        height = window.getHeight(),
        morph = false;

    // Jesli animowany event, animujemy zmiane
    if(pE && pAction) {
      width = window.getWidth() - this.Sidebar.mEl.getStyle('width').toIntZero();
      morph = true;

      if('show' == pAction) {
        width -= pE.mToolbarWidth;
      }

      if(MooTools.$defined(pShowHideOptions) && false == pShowHideOptions.resize_workbench) {
        morph = false;
      }
    }

    if(width < 600) {
      document.getElementById("DPMainToolbar") && document.getElementById("DPMainToolbar").addClass('narrow');
    }
    else {
      document.getElementById("DPMainToolbar") && document.getElementById("DPMainToolbar").removeClass('narrow');
    }

    var styles = {
      'width': width,
      'height': height
    };

    var canvasWrapperEl = this.mCanvasEl.getElementById('canvas-wrapper');
    var wrapper_styles = {};
    if (canvasWrapperEl.classList.contains('online')) {
      wrapper_styles = {
        'width': width + 'px',
        'height': parseInt(height - canvasWrapperEl.style.top, 10) + 'px'
      };
    }

    // setting width?
    if (this.mVersions[this.mCurrentVersion]) {

      // hot fix hack for old Default
      // @todo do it properly
      if (this.mVersions[this.mCurrentVersion].width > 0 && this.mVersions[this.mCurrentVersion].text !== 'Default' && this.mVersions[this.mCurrentVersion].width != 5000) {
        wrapper_styles.width = this.mVersions[this.mCurrentVersion].width > 0 ? this.mVersions[this.mCurrentVersion].width + 'px' : '100%';
      }

      if (this.mVersions[this.mCurrentVersion].height) {
        wrapper_styles.height = this.mVersions[this.mCurrentVersion].height + 'px';
      }

      if (this.mVersions[this.mCurrentVersion].scrollx) {
        wrapper_styles.overflowX = 'auto';
      }

      if (this.mVersions[this.mCurrentVersion].scrolly) {
        wrapper_styles.overflowY = 'auto';
      }
    } else {
      wrapper_styles.width = '100%';
    }

    if(morph) {
      this.mCanvasEl.morph(styles);
      canvasWrapperEl.morph(wrapper_styles);
    }
    else {
      this.mCanvasEl.setStyles(styles);
      Object.keys(wrapper_styles).forEach(function(styleKey) {
        canvasWrapperEl.style[styleKey] = wrapper_styles[styleKey];
      }.bind(this));
    }


    if(this.mCurrentVersion.toInt() == 0) {
      canvasWrapperEl.classList.add('default');
    }
    else {
      canvasWrapperEl.classList.remove('default')
    }
  },

  isExport: function() {
    return this.mData.export_mode;
  },

  isShotMode: function() {
    return this.mData.is_shot_mode;
  },

  isMenuEnabled: function() {
    return this.mData.menu_enabled;
  },

  isCollaborationEnabled: function() {
    return this.mData.active_collaboration;
  },

  /**
   * JavaScript Pretty Date
   * Copyright (c) 2011 John Resig (ejohn.org)
   * Licensed under the MIT and GPL licenses.
   */
  prettyDate: function(pTime) {

    var date = new Date((pTime || "").replace(/-/g,"/").replace(/[TZ]/g," ")),
        diff = (((new Date()).getTime() - date.getTime()) / 1000),
        day_diff = Math.floor(diff / 86400);

    if ( isNaN(day_diff) || day_diff < 0 )
      return;

    return day_diff == 0 && (
			diff < 60 && "just now" ||
			diff < 120 && "1 minute ago" ||
			diff < 3600 && Math.floor( diff / 60 ) + " minutes ago" ||
			diff < 7200 && "1 hour ago" ||
			diff < 86400 && Math.floor( diff / 3600 ) + " hours ago") ||
      day_diff == 1 && "Yesterday" ||
      day_diff < 7 && day_diff + " days ago" ||
      Math.ceil( day_diff / 7 ) + " weeks ago";
  },


  loadPageCall: function(pId) {
    var instant = true;
    var version;
    this.mData.versions.each(function(pVersion, pType) {
      if(pVersion == pId) {
        version = pType;
      }
    });

    if(this.mPageBuffer[pId]) {
      this.comet.loadPage(this.mPageBuffer[pId], instant);
    }

    this.fetchPage(pId, version);

    this.mCurrentVersion = version;
    this.mCurrentPageId = pId;
  },


  cachePage: function(pId, pHTML) {
    this.mPageBuffer[pId] = pHTML;
  },


  fetchPage: function(pIdPage, pType) {
    if(!this.isExport()) {
      this.mDejax.newRequest({
        'url': '/ajax/dmsDPPreview/GetPageContent/',
        'eval': true,
        'mode': 'cancel',
        'data': {
          'id_page': pIdPage,
          'variation_type' : pType,
          'id_project': Preview.getData().id_project,
          'project_hash': Preview.getData().project_hash,
          'menu_enabled': Preview.isMenuEnabled(),
          'active_collaboration': Preview.isCollaborationEnabled()
        },
        'onComplete': function(pHTML) {
          this.cachePage(pIdPage,pHTML);
          this.comet.loadPage(pHTML, true)
        }.bind(this)
      });
    }
  },


  preloadPage: function(pVersions) {
    this.mDejax.newRequest({
      'url': '/ajax/dmsDPPreview/GetPageContents/',
      'eval': true,
      'mode': 'queue',
      'data': {
        'variation_data': pVersions,
        'id_project': Preview.getData().id_project,
        'project_hash': Preview.getData().project_hash,
        'menu_enabled': Preview.isMenuEnabled(),
        'active_collaboration': Preview.isCollaborationEnabled()
      },
      'onComplete': function(pData) {
        pData = new MooTools.Hash(pData);
        pData.forEach(function(pData, pIdPage) {
          this.cachePage(pIdPage,pData);
        }.bind(this));

      }.bind(this)
    });
  },


  responsiveAdjust: function() {
    var width = document.getElementById("canvas").getStyle('width').toInt();
    var version = null;
    var found = false;

    if((document.getElement("#version-select") && document.getElement("#version-select").getSelected().get('value')=='responsive') || (!Preview.isMenuEnabled() && Preview.mData.is_shot_mode == false)) {

      Preview.breaks.breakEach(function(pObj) {
        if(pObj.size >= width) {
          if(Preview.current_page_id != Preview.mData.versions[pObj.version]) {
            found = true;
            Preview.current_page_id = Preview.mData.versions[pObj.version];
            Preview.mCurrentVersion = pObj.version;
            Preview.loadPageCall(Preview.mData.versions[pObj.version]);

            return false;
          }
          else {
            found = true;
            return false;
          }
        }
      });

      if(found == false && Preview.breaks.length) {
        Preview.current_page_id = Preview.mData.versions[Preview.breaks[Preview.breaks.length - 1].version];
        Preview.mCurrentVersion = Preview.breaks[Preview.breaks.length - 1].version
        Preview.loadPageCall(Preview.mData.versions[Preview.breaks[Preview.breaks.length - 1].version]);
      }
    }
  },

  checkForResponsiveOption: function() {
    if(hashuri.get('R') == 1 || !this.isMenuEnabled()) {
      document.getElement('.version-list') && document.getElement('.version-list').getSelected().set('selected',false);
      document.getElement('.version-list') && document.getElement('.version-list').getElement('[value="responsive"]').setAttribute('selected','selected');
      this.responsiveAdjust();
      this.canvasResize();

      return true;
    }
    return false;
  },

  loadPageVersion: function(pVersion) {
    var page_id = Preview.mData.versions[pVersion];
    Preview.loadPageCall(page_id);
    if(document && document.getElement('.version-list')) {
      document.getElement('.version-list').selectedIndex = document.getElement('.version-list').getElement('[value='+Preview.mCurrentPageId+']').index
    }
    this.canvasResize();
    //document.getElement('.version-list').getSelected().set('selected',false);

  },

  getFixed: function() {
    var fixed = document.querySelectorAll('.FixedPos');

    for(var i = 0;i<fixed.length;i++) {
      fixed[i].originalTop = fixed[i].style.top;
      fixed[i].originalLeft = fixed[i].style.left;
    }
    return fixed;
  },

  fixModals: function() {
    var positionOverlay = function() {
      var styles = {
        'left': 0,
        'width': $('canvas-wrapper').scrollWidth,
        'height': $('canvas-wrapper').scrollHeight
      };

      $$('.modal-overlay').setStyles(styles);
    };

    positionOverlay();
    window.addEvent('resize', positionOverlay);

    var modal = $$('.ElGroup > .modal-overlay, .Component > .modal-overlay');
    modal.each(function(pEl) {

      if(pEl.getParents('.Component') || pEl.getParents('.ElGroup')) {
        var parent = pEl.getParent();
        var removed = pEl.dispose();

        try{
          removed.inject(parent, 'before')
        }
        catch(e){}
      }
    })
  },

  gotoPage: function(pPage) {
    window.location = pPage;
  },

  adjustTextElements: function(pElems) {
    "use strict";
    MooTools.Array.prototype.forEach.call(pElems, function(pElem) {
      var width = parseInt(pElem.style.width);
      while(pElem.scrollHeight > pElem.clientHeight) {
        width+=1;
        pElem.style.width = width+"px";
      }
    });
  }

});

/* globals DPMainToolbar */

var dmsDPPreview_MainToolbar = new MooTools.Class({
  
  Extends: DPMainToolbar,
  
  initialize: function() {
    this.mParentEl = document.getElement('#canvas');
    this.mItems = [];
  },
  
  addEl: function(pEl) {

    this.mItems.push(pEl);
    this.render();
  },
  
  getEl: function() {
    return this.mEl;
  }
  
});

;(function PreviewInteractionsScope(global) {
  'use strict';

  var Interactions = {
    ori_styles: {},

    active_node_id: 'main1',

    els: {},

    scrolls: [],
    scrollable_els: null,
    scroll_offset_prev: 0,
    scroll_offset: 0,
    on_scroll_ref: null,

    loading_el: null,

    attachScrollEvent: function() {
      var i;
      if (Interactions.scrollable_els) {
        if (Interactions.on_scroll_ref) {
          for (i=0; i<Interactions.scrollable_els.length; i++) {
            Interactions.scrollable_els[i].removeEventListener('scroll', Interactions.on_scroll_ref);
            if (Interactions.scrollable_els[i].scrollTop) {
              Interactions.scroll_offset = Interactions.scrollable_els[i].scrollTop;
            }
          }
          Interactions.on_scroll_ref = null;
        }

        Interactions.scroll_offset_prev = 0;
        Interactions.on_scroll_ref = Interactions.onContentScrolled.bind(Interactions);

        for (i=0; i<Interactions.scrollable_els.length; i++) {
          Interactions.scrollable_els[i].addEventListener('scroll', Interactions.on_scroll_ref);
        }
      }
    },

    getLoadElement: function() {
      if (!Interactions.loading_el) {
        Interactions.loading_el = document.createElement('img');
        Interactions.loading_el.style.opacity = 0;
      }

      return Interactions.loading_el;
    },

    onContentLoaded: function(pLoadFunctions) {
      pLoadFunctions.forEach(function(event) {
        if (typeof event === 'function') {
          event('load');
        }
      });
    },

    onContentScrolled: function(e) {
      Interactions.scroll_offset_prev = Interactions.scroll_offset;
      Interactions.scroll_offset = 0;
      for (var i=0; i<Interactions.scrollable_els.length; i++) {
        Interactions.scroll_offset += Interactions.scrollable_els[i].scrollTop;
      }

      Interactions.scrolls.forEach(function(callback) {
        if (typeof callback === 'function') {
          callback(e);
        }
      });
    },

    onChangeLayer: function(pActiveId, pPrevId) {
      Interactions.active_node_id = 'main' + pActiveId;

      // old references won't be valid anymore
      Interactions.els['main' + pPrevId] = {};
    },

    getSetElRef: function(id) {
      var node_id = Interactions.active_node_id;

      if (typeof Interactions.els[node_id] === 'undefined') {
        Interactions.els[node_id] = {};
      }

      if (typeof Interactions.els[node_id][id] !== 'object') {
        Interactions.els[node_id][id] = document.getElementById(node_id).querySelector('*[id^=el'+id+'_]');

        // exception: tooltip has wrapper el
        if (Interactions.els[node_id][id].classList.contains('ElTooltip')) {
          Interactions.els[node_id][id] = Interactions.els[node_id][id].parentNode;
        }
        // another exception: table has child element
        else if (Interactions.els[node_id][id].classList.contains('ElTableContainer')) {
          Interactions.els[node_id][id] = Interactions.els[node_id][id].querySelector('.ElTable');
        }
      }

      return Interactions.els[node_id][id];
    },

    ifFireOneTimeScroll: function(id, action, offset, reverse) {
      var return_value = 0;

      // if scrolled below boundary offset
      if (Interactions.scroll_offset >= offset && Interactions.scroll_offset_prev < offset) {
        return_value = 1;
      }
      else if (Interactions.scroll_offset < offset && Interactions.scroll_offset_prev >= offset) {
        return_value = -1;
      }

      if (reverse) {
        return_value *= -1;
      }

      return return_value;
    },

    shouldBeFired: function(pE) {
      if(pE === 'load') {
        return true;
      }
      var target_node_name = pE.target.nodeName.toLowerCase();
      if (pE.keyCode && (['input', 'textarea', 'select'].indexOf(target_node_name) > -1 )) {
        return false;
      }
      return true;
    },

    handleEvent: function(e, processed) {
      if(e === 'load') {
        return;
      }
      var target_node_name = e.target.nodeName.toLowerCase();
      var notProcessedCallback = function(pValue) {
        return false === pValue;
      };

      // don't prevent default when keydown inside form element
      if (e.keyCode && (['input', 'textarea', 'select'].indexOf(target_node_name) > -1 || processed.every(notProcessedCallback))) {
        return;
      }

      // Don't preventDefault() on checkbox or label by click event
      if ('click' === e.type) {
        if ('label' === target_node_name) {
          return;
        }

        if ('input' === target_node_name && ['checkbox', 'radio'].indexOf(e.target.getAttribute('type').toLowerCase()) > -1)  {
          return;
        }
      }

      e.preventDefault();
    }

  };

  global.Interactions = Interactions;

})(window);

/* globals TweenLite */
/* globals UXPAnimationsMap */
/* globals _ */

;(function InteractionsAnimationsScope(global) {
  'use strict';

  /**
   * @param el
   * @constructor
   */
  var InteractionsAnimations = function(el) {

    // element
    this.el = el;

    // browser prefix
    this.prefix = this.getPrefix();

    // standardized form of animations set on element
    this.animations = {};

    // check for existing animations
    this.parseCurrentAnimations();
  };

  /**
   * Checks and returns browser prefix for css styles
   *
   * @returns {string}
   */
  InteractionsAnimations.prototype.getPrefix = function() {
    var prefix = '';

    for (var css_prefix in InteractionsAnimations.prefixes) {
      if (InteractionsAnimations.prefixes.hasOwnProperty(css_prefix) && typeof window.getComputedStyle(this.el)[InteractionsAnimations.prefixes[css_prefix] + 'Transform'] !== 'undefined') {
        prefix = css_prefix;
        break;
      }
    }

    return prefix;
  };

  /**
   * Returns reference to this.animations[type] table and creates it if it doesn't exist
   *
   * @param type
   * @returns {*}
   */
  InteractionsAnimations.prototype.getDefineType = function(type) {
    if (typeof this.animations[type] === 'undefined') {
      this.animations[type] = {
        'property': [],
        'duration': [],
        'function': []
      };
    }

    return this.animations[type];
  };

  /**
   * Parses element styles to get currently applied animations. Populates this.animations object
   */
  InteractionsAnimations.prototype.parseCurrentAnimations = function() {
    var transitions = this.parseTransitions();
    var property;
    var index;

    // check for existing animations set on element
    for (var type in InteractionsAnimations.definitions) {
      if (InteractionsAnimations.definitions.hasOwnProperty(type)) {

        // search for properties in transition-property
        for (var i in InteractionsAnimations.definitions[type]) {
          if (InteractionsAnimations.definitions[type].hasOwnProperty(i)) {
            property = InteractionsAnimations.definitions[type][i];

            // add prefix to -transform
            if (property.test(/^\-.*$/)) {
              property = '-' + this.prefix + property;
            }

            index = transitions.property.indexOf(property);

            if (index > -1) {
              var animation = this.getDefineType(type);

              animation.property.push(property);
              animation.duration.push(transitions.duration[index]);
              animation['function'].push(transitions['timing-function'][index]);

              transitions.property.splice(index, 1);
              transitions.duration.splice(index, 1);
              transitions['timing-function'].splice(index, 1);
            }
            // if property of given type of animation not found in transitions table, continue with another type
            else {
              break;
            }
          }
        }
      }
    }
  };

  /**
   * Parses css styles of the element and gets applied transitions
   *
   * @returns {{}}
   */
  InteractionsAnimations.prototype.parseTransitions = function() {
    var transitions = {};

    // map transition property string to array
    transitions.property = window.getComputedStyle(this.el)[InteractionsAnimations.prefixes[this.prefix] + 'TransitionProperty'].split(',').map(function(val) {
      return val.trim();
    });

    transitions.duration = window.getComputedStyle(this.el)[InteractionsAnimations.prefixes[this.prefix] + 'TransitionDuration'].split(',').map(function(val) {
      return val.trim();
    });

    transitions['timing-function'] = window.getComputedStyle(this.el)[InteractionsAnimations.prefixes[this.prefix] + 'TransitionTimingFunction'].split(',').map(function(val) {
      return val.trim();
    });

    return transitions;
  };

  /**
   * Applies animation of given type to the element
   *
   * @param type
   * @param timing_function
   * @param duration
   */
  InteractionsAnimations.prototype.addAnimation = function(type, timing_function, duration) {
    var animation = this.getDefineType(type);
    var prefix = this.prefix;

    InteractionsAnimations.definitions[type].forEach(function(property) {

      // add prefix to -transform
      if (property.test(/^\-.*$/)) {
        property = '-' + prefix + property;
      }

      animation.property.push(property);
      animation.duration.push(duration);
      animation['function'].push(timing_function);
    });

    this.apply();
  };

  /**
   * Converts this.animations object to js styles and applies them to the DOM element
   */
  InteractionsAnimations.prototype.apply = function() {
    var transition_property = [];
    var transition_duration = [];
    var transition_function = [];

    var property_cbk = function(property) {
      transition_property.push(property);
    };

    var duration_cbk = function(duration) {
      transition_duration.push(duration);
    };

    var function_cbk = function(func) {
      transition_function.push(func);
    };

    for (var type in this.animations) {
      if (this.animations.hasOwnProperty(type)) {
        this.animations[type].property.forEach(property_cbk);
        this.animations[type].duration.forEach(duration_cbk);
        this.animations[type]['function'].forEach(function_cbk);
      }
    }

    this.el.style[InteractionsAnimations.prefixes[this.prefix] + 'TransitionProperty'] = transition_property.join(', ');
    this.el.style[InteractionsAnimations.prefixes[this.prefix] + 'TransitionDuration'] = transition_duration.join(', ');
    this.el.style[InteractionsAnimations.prefixes[this.prefix] + 'TransitionTimingFunction'] = transition_function.join(', ');
  };

  /**
   * Removes animations of given type from element
   *
   * @param type
   */
  InteractionsAnimations.prototype.remove = function(type) {
    this.parseCurrentAnimations();

    if (typeof this.animations[type] !== 'undefined') {
      delete this.animations[type];
    }

    this.apply();
  };

  /**
   * Removes all animations from element
   */
  InteractionsAnimations.prototype.removeAll = function() {
    this.animations = {};
    this.apply();
  };

  InteractionsAnimations.prefixes = {
    'webkit': 'webkit',
    'moz': 'Moz',
    'ms': 'ms',
    'o': 'o'
  };

  /**
   * Definitions of a properties that are specific for given animation type.
   * First defined property in each animation type should be unique for given animation (should not be present in other animation types)
   *
   * @type {{show: string[], move: string[], resize: string[], scale: string[], rotate: string[], opacity: string[], style: string[], style_navi: string[]}}
   */
  InteractionsAnimations.definitions = {
    'show': ['opacity', 'visibility', 'margin'],
    'move': ['top', 'left'],
    'resize': ['width', 'height'],
    'scale': ['-transform'],
    'rotate': ['-transform'],
    'opacity': ['opacity'],
    'style': ['color', 'background-color', 'border-color'],
    'style_navi': ['background-color', 'border-color']
  };

  InteractionsAnimations.timeouts = {
    'show': {},
    'move': {},
    'resize': {},
    'scale': {},
    'rotate': {},
    'opacity': {},
    'style': {},
    'style_navi': {}
  };

  InteractionsAnimations.timing = {
    'linear': function(t, b, c, d) {
      t/=d;
      return b+c*(t);
    },
    'ease-out': function(t, b, c, d) {
      var ts = (t/=d)*t;
      var tc = ts*t;
      return b+c*(tc*ts + -5*ts*ts + 10*tc + -10*ts + 5*t);
    },
    'ease-in': function(t, b, c, d) {
      var ts = (t/=d)*t;
      var tc = ts*t;
      return b+c*(tc*ts);
    },
    'ease-in-out': function(t, b, c, d) {
      var ts = (t/=d)*t;
      var tc = ts*t;
      return b+c*(6*tc*ts + -15*ts*ts + 10*tc);
    }
  };

  /**
   * Set default easing value if is not set and map legacy timing value to new easing
   */
  InteractionsAnimations.setDefaultEasingValues = function(data) {
    var easingData = _.defaults(data || {}, {
      easing: UXPAnimationsMap.easing.LINEAR.value,
      easingType: UXPAnimationsMap.easingType.POWER0.value,
      duration: 0
    });

    if (!data || !data.easing) {
      easingData.duration = 0;
    }

    if (easingData.style && Object.keys(UXPAnimationsMap.legacyEasing).indexOf(easingData.style) > -1) {
      easingData.duration = easingData.time || easingData.duration || 0;
      easingData.easing = UXPAnimationsMap.legacyEasing[easingData.style].easing;
      easingData.easingType = UXPAnimationsMap.legacyEasing[easingData.style].easingType;
    }

    return easingData;
  };

  /**
   * Get GreenSock easing function
   */
  InteractionsAnimations.getEasing = function(data) {
    var easing;
    if (data.easingType === UXPAnimationsMap.easingType.ELASTIC.value) {
      var springness = data.springness / Math.pow(10, (MooTools.String(data.springness).length - 2) * 2 + 1);
      easing = window[data.easingType][data.easing].config(data.amplitude / 100, springness);
    } else {
      easing = window[data.easingType][data.easing];
    }

    return easing;
  };

  InteractionsAnimations.processScroll = function(el, to, duration, animation) {
    animation = InteractionsAnimations.setDefaultEasingValues(animation);
    InteractionsAnimations.scroll(el, to, duration, InteractionsAnimations.getEasing(animation));
  };

  /**
   * Scrolls the element (el) to provided offset (to) in (duration) time using (timing) function
   *
   * @param el
   * @param to
   * @param duration
   * @param easing
   */
  InteractionsAnimations.scroll = function(el, to, duration, easing) {
    TweenLite.to(el, duration / 1000, {
      scrollTo: {
        y: to
      },
      ease: easing
    });
  };

  /**
   * Moves the element (el) to (to) position from top in (duration) time using (timing) function starting in (start) position
   *
   * @param el
   * @param to
   * @param duration
   * @param timing
   * @param start [optional]
   */
  InteractionsAnimations.positionTop = function(el, to, duration, timing, start) {
    start = start || parseInt(window.getComputedStyle(el).top);

    var change = to - start;
    var currentTime = 0;
    var increment = 5;

    /**
     * Animation function, increments processing time and calls itself again for another move
     * if duration isn't reached yet. Don't use it outside InteractionsAnimations.positionTop method
     */
    var animateTop = function() {
      currentTime += increment;
      el.style.top = InteractionsAnimations.timing[timing](currentTime, start, change, duration) + 'px';

      if(currentTime < duration) {
        setTimeout(animateTop, increment);
      }
    };

    animateTop();
  };

  InteractionsAnimations.getTimeoutTime = function(time) {
    return time > 20 ? time - 10 : 10;
  };

  global.InteractionsAnimations = InteractionsAnimations;
})(window);

//# sourceMappingURL=maps/offlinePreviewLibs.js.map
